/* Copyright 2004 Jozsef Kadlecsik (kadlec@blackhole.kfki.hu)
 *
 * This program is free software; you can redistribute it and/or modify   
 * it under the terms of the GNU General Public License as published by   
 * the Free Software Foundation; either version 2 of the License, or      
 * (at your option) any later version.                                    
 *                                                                         
 * This program is distributed in the hope that it will be useful,        
 * but WITHOUT ANY WARRANTY; without even the implied warranty of         
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
 * GNU General Public License for more details.                           
 *                                                                         
 * You should have received a copy of the GNU General Public License      
 * along with this program; if not, write to the Free Software            
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */


#include <stdio.h>			/* *printf */
#include <string.h>			/* mem* */

#include "ipset.h"

#include <linux/netfilter_ipv4/ip_set_ipp2p.h>

static char* ipp2p_type[] = {"MSN","YM","ICQ","AIM","QQ","ICHAT","GT","WIM","WMSN","JAJAH","SKYPE","SOULSEEK","EDONKEY","FASTTRACK","GNUTELLA","BITTORRENT",
						  "OPENFT", "OPENNAP", "WINNY", "XUNLEI", "VAGAA", "PP365", "POCO", "CLUBBOX", "ARES", "EZPEER", "PANDO", "HUNTMINE", "KUWO"};

/* Initialize the create. */
static void
ipp2p_create_init(void *data UNUSED)
{
	struct ip_set_req_ipp2p_create *mydata = data;
	
	DP("create INIT");

	mydata->from = IPP2P_FROM;
	mydata->to = IPP2P_TO;
	
	/* Nothing */
}

/* Function which parses command options; returns true if it ate an option */
static int
ipp2p_create_parse(int c, char *argv[] UNUSED, void *data, unsigned *flags)
{
	DP("create_parse");	
	return 1;
}

/* Final check; exit if not ok. */
static void
ipp2p_create_final(void *data, unsigned int flags)
{
	struct ip_set_req_ipp2p_create *mydata = data;

	DP("from : %x to: %x  diff: %d", mydata->from, mydata->to,
	   mydata->to - mydata->from);
}

/* Create commandline options */
static const struct option create_opts[] = {
	{NULL},
};

/* Add, del, test parser */
static ip_set_ip_t
ipp2p_adt_parser(int cmd UNUSED, const char *arg, void *data)
{
	struct ip_set_req_ipp2p *mydata = data;

	DP("%s", arg);

	if (strings_to_index(ipp2p_type, IPP2P_NUM, arg, &mydata->ip) != 0)
	{
		exit_error(PARAMETER_PROBLEM, 
		           "Invalid protcol `%s' specified", arg);
		return 0;
	}
	return 1;	
}

/*
 * Print and save
 */

static void
ipp2p_initheader(struct set *set, const void *data)
{
	const struct ip_set_req_ipp2p_create *header = data;
	struct ip_set_ipp2p *map = set->settype->header;

	memset(map, 0, sizeof(struct ip_set_ipp2p));
	map->first_ip = header->from;
	map->last_ip = header->to;
}

static void
ipp2p_printheader(struct set *set, unsigned options)
{
	struct ip_set_ipp2p *mysetdata = set->settype->header;

	printf("%s %s\n", 
	       set->name,
	       set->settype->typename);

}

static inline void
__ipp2p_printips_sorted(struct set *set, void *data,
			  u_int32_t len UNUSED, unsigned options)
{
	struct ip_set_ipp2p *mysetdata = set->settype->header;
	ip_set_ip_t addr = mysetdata->first_ip;

	DP("%u -- %u", mysetdata->first_ip, mysetdata->last_ip);
	while (addr <= mysetdata->last_ip) {
		if (test_bit(addr - mysetdata->first_ip, data))
			printf("%s\n", ipp2p_type[addr]);
		addr++;
	}
}

static void
ipp2p_printips_sorted(struct set *set, void *data,
			u_int32_t len, unsigned options,
			char dont_align)
{
	ip_set_ip_t *ip;
	size_t offset = 0;
	
	if (dont_align)
		return __ipp2p_printips_sorted(set, data, len, options);
	
	while (offset < len) {
		ip = data + offset;
		printf("%s\n", ipp2p_type[*ip]);
		offset += IPSET_ALIGN(sizeof(ip_set_ip_t));
	}
}

static void
ipp2p_saveheader(struct set *set, unsigned options)
{
	struct ip_set_ipp2p *mysetdata = set->settype->header;

	printf("-N %s %s\n", 
	       set->name,
	       set->settype->typename);
}

static inline void
__ipp2p_saveips(struct set *set, void *data,
		  u_int32_t len UNUSED, unsigned options)
{
	struct ip_set_ipp2p *mysetdata = set->settype->header;
	ip_set_ip_t addr = mysetdata->first_ip;

	while (addr <= mysetdata->last_ip) {
		DP("addr: %lu, last_ip %lu", (long unsigned)addr, (long unsigned)mysetdata->last_ip);
		if (test_bit(addr - mysetdata->first_ip, data))
			printf("-A %s %s\n",
			       set->name,
			       ipp2p_type[addr]);
		addr++;
	}
}

static void
ipp2p_saveips(struct set *set, void *data,
		u_int32_t len, unsigned options,
		char dont_align)
{
	ip_set_ip_t *ip;
	size_t offset = 0;
	
	if (dont_align)
		return __ipp2p_saveips(set, data, len, options);
	
	while (offset < len) {
		ip = data + offset;
		printf("-A %s %s\n", set->name, port_tostring(*ip, options));
		offset += IPSET_ALIGN(sizeof(ip_set_ip_t));
	}
}

static void
ipp2p_usage(void)
{
	printf
	    ("-N set ipp2p\n");
	strings_print(ipp2p_type, IPP2P_NUM, "-ADT %s\n");
}

static struct settype settype_ipp2p = {
	.typename = SETTYPE_NAME,
	.protocol_version = IP_SET_PROTOCOL_VERSION,

	/* Create */
	.create_size = sizeof(struct ip_set_req_ipp2p_create),
	.create_init = ipp2p_create_init,
	.create_parse = ipp2p_create_parse,
	.create_final = ipp2p_create_final,
	.create_opts = create_opts,

	/* Add/del/test */
	.adt_size = sizeof(struct ip_set_req_ipp2p),
	.adt_parser = ipp2p_adt_parser,

	/* Printing */
	.header_size = sizeof(struct ip_set_ipp2p),
	.initheader = ipp2p_initheader,
	.printheader = ipp2p_printheader,
	.printips = ipp2p_printips_sorted,
	.printips_sorted = ipp2p_printips_sorted,
	.saveheader = ipp2p_saveheader,
	.saveips = ipp2p_saveips,
	
	.usage = ipp2p_usage,
};

CONSTRUCTOR(ipp2p)
{
	settype_register(&settype_ipp2p);

}
