#include <linux/init.h>
#include <linux/module.h>
#include <linux/autoconf.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_arp.h>
#include <linux/usb.h>
#include <linux/sockios.h>
#include <linux/timer.h>
#include <linux/wait.h>
#include <linux/fs.h>	
#include <linux/list.h>
#include <linux/workqueue.h>
#include <linux/spinlock.h>
#include <linux/skbuff.h>

#include "lte_driver.h"
#include "lte_user_if.h"
#include "samsung_lte.h"
#include "lg_vl600_lte.h"
#include "lte_at_cmd.h"
#include "os_shim.h"
extern struct file_operations lte_ch_fops;

static LIST_HEAD(dev_list);
static DECLARE_MUTEX(dev_list_lock);
static int lte_probe (struct usb_interface *udev, const struct usb_device_id *prod);
static void lte_disconnect (struct usb_interface *intf);
void * gmodem;
//------------------------------------------------------------------------------------ 
//              debug stuffs
//------------------------------------------------------------------------------------
void lte_dump_buffer( unsigned char * buf, int len, char * begin_str)
{
    char  ascii_str[512];
    int i, j, bytes_per_line;
	char c, * p;

#if LTE_DRIVER_DBG == 0
	return;
#endif

	if( len == 0 ) return;

	if(!buf)
	{
		printk("buf is NULL in MyDumpBuffer()\n");
		return;
	}

	bytes_per_line = 16;

	printk("%s %d bytes\n", begin_str, len);

	for (i = 0; i < len; i += bytes_per_line)
	{
		p = ascii_str;

		sprintf(p, "%04X    ", i);
		p += 8;

		//Raw Hex digits
		for (j = i; j < (i + bytes_per_line); j++)
		{
			if(j < len)
			{
				sprintf(p, "%02X ", buf[j]);
			}
			else
			{
				sprintf(p, "   ");
			}
			p += 3;
		}
    

		sprintf(p, "  ");
		p += 2;
		//Ascii characters
		for (j = i; j < len && j < i + bytes_per_line; j++)
		{

            if( buf[j]>= 32 && buf[j]<=126 )
            {
			    c = buf[j];
            }
            else
            {
                c = '.';
            }
			*p++ = c;
		}
		*p = 0;

		printk("%s\n", ascii_str);

	}

	printk("\n");
}

//-----------------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------------
void lte_delay_ms(int delay_ms)
{
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout(delay_ms * HZ /1000); 
	return;
}

int lte_send_to_usb(struct lte_modem_st * modem, struct sk_buff * skb)
{
	int i, padding_len, len;
	struct urb *urb = 0;
	char * buf = 0;

    if(!modem->link_up || modem->disconnected) 
	{
		goto error;
	}

	if(!skb)
	{
		printk("skb is NULL in samsung_lte_send_to_usb()\n");
		return 0;
	}


	if(!skb->data)
	{
		printk("skb->data is NULL in samsung_lte_send_to_usb()\n");
		goto error;
	}

	if(skb->len > 1600)
	{
		printk("Packet too large[%d bytes] in samsung_lte_send_to_usb()\n", skb->len);
		goto error;
	}

#if 0 //dump ping 168.95.1.1

	if(skb->data[30] == 0xA8 && skb->data[31] == 0x5F)
	{
		printk("---> ping 168.95.1.1\n");
//		lte_dump_buffer(skb->data, skb->len, "");
	}
#endif

	if(skb->data[30] == modem->dns_ip[0] && skb->data[31] == modem->dns_ip[1])
	{
//		printk("---> ping DNS\n");
		if(modem->keep_alive_timer > 0)
		{
			//timer is still running, previous pings did not get a replay
		}
		else
		{
			//start 30 sec timer
			modem->keep_alive_timer = 30;
		}
//		lte_dump_buffer(skb->data, skb->len, "");
	}


	len = skb->len;

#if LTE_USE_PRE_ALLOC_TX_BUF == 1
	if(modem->tx_buf_left == 0)
	{
		printk("Cna't get Tx Buffer\n");
		goto error;
	}
	urb = modem->data_tx_urb[modem->tx_buf_index];
	buf = modem->data_tx_dma_buf[modem->tx_buf_index];
	modem->tx_buf_index++;
	modem->tx_buf_index &= (modem->act_tx_buf_count - 1);
	modem->tx_buf_left--;

#else//LTE_USE_PRE_ALLOC_TX_BUF == 1
	urb = usb_alloc_urb(0, GFP_ATOMIC);
	if(!urb)
	{
		printk("Can't get urb in lte_send_to_usb()\n");
		goto error;
	}

	buf = usb_buffer_alloc(modem->usb_dev, 2048, GFP_ATOMIC | GFP_DMA, &urb->transfer_dma);
	if(!buf)
	{
		printk("Can't get buffer in lte_send_to_usb()\n");
		goto error;
	}
#endif //LTE_USE_PRE_ALLOC_TX_BUF == 1

	if(modem->tx_process)
	{
		if(modem->tx_process(modem, skb->data, buf, &len) < 0) goto error;
	}
	else
	{
		printk("modem does not have tx_process method()\n");
		goto error;
	}

	//send to USB

#if LTE_DRIVER_DUMP_TX_RX == 1
	lte_dump_buffer(buf, len, "-------------> out packet");
#endif //LTE_DRIVER_DUMP_TX_RX == 1

 	usb_fill_bulk_urb(urb, modem->usb_dev,  modem->data_out_pipe, buf, len, lte_tx_comp, modem);
    urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP | URB_ZERO_PACKET;

	if(usb_submit_urb(urb, GFP_ATOMIC) < 0)
	{
		printk("submit urb failed in samsung_lte_send_to_usb()\n");
		goto error;
	}

	modem->current_tx_urb = urb;
	netif_stop_queue(modem->net_dev);
	((struct net_device *)modem->net_dev)->trans_start = jiffies;

	((struct net_device_stats *)modem->statistic)->tx_packets++;
	((struct net_device_stats *)modem->statistic)->tx_bytes += len;
	dev_kfree_skb_any(skb);

	return 0;

error:
	((struct net_device_stats *)modem->statistic)->tx_dropped++;
	if(buf)
	{
#if LTE_USE_PRE_ALLOC_TX_BUF == 1
		modem->tx_buf_left++;
#else //LTE_USE_PRE_ALLOC_TX_BUF == 1
		usb_buffer_free(modem->usb_dev, 2048, ((struct urb *)urb)->transfer_buffer, ((struct urb *)urb)->transfer_dma);
#endif //LTE_USE_PRE_ALLOC_TX_BUF == 1
	}
	if(urb) usb_free_urb(urb);
	dev_kfree_skb_any(skb);
	return 0;
}

int get_config_desc(struct lte_modem_st * modem)
{
	unsigned char desc_9_bytes[9];
	if(usb_control_msg(modem->usb_dev, usb_rcvctrlpipe(((struct usb_device *)modem->usb_dev), 0), 
		               0x06, 0x80, 0x0200, 0, desc_9_bytes, 9, 1000)  < 0)
	{
		printk("get_config_desc() failed 1\n");
		return -1;
	}
	lte_dump_buffer(desc_9_bytes, 9, "config desc 9 bytes\n");
	return 0;
}

int get_device_desc(struct lte_modem_st * modem)
{
	unsigned char desc[18];
	if(usb_control_msg(modem->usb_dev, usb_rcvctrlpipe(((struct usb_device *)modem->usb_dev), 0), 
		               0x06, 0x80, 0x0100, 0, desc, 18, 1000)  < 0)
	{
		printk("get_device_desc() failed 1\n");
		return -1;
	}
	lte_dump_buffer(desc, 18, "device descriptor\n");
	return 0;
}

int get_config(struct lte_modem_st * modem)
{
	unsigned char config;
	if(usb_control_msg(modem->usb_dev, usb_rcvctrlpipe(((struct usb_device *)modem->usb_dev), 0), 
		               0x08, 0x80, 0, 0, &config, 1, 1000)  < 0)
	{
		printk("get_device_desc() failed 1\n");
		return -1;
	}
	LTE_PRINT(("active config = %d\n", config));
	return 0;
}

int set_config(struct usb_device * usb_dev, int config_num)
{
	if(usb_control_msg(usb_dev, usb_sndctrlpipe(((struct usb_device *)usb_dev), 0), 
		               0x09, 0x00, config_num, 0, 0, 0, 1000)  < 0)
	{
		printk("set_config() %d failed\n", config_num);
		return -1;
	}
	return 0;
}

void lte_link_up(struct lte_modem_st * modem)
{
	if(modem->link_up == 1) return;
	printk("********* LINK UP *********\n");
	modem->link_up = 1;
	modem->link_status = 2;
//	modem->connect_timer = 0;
	modem->hs_timer = 0;
	modem->my_ip_dns_ip_valid = 0; 
	lte_send_ap_link_status(modem);
	lte_send_ap_network_type(modem);
	lte_send_ap_link_up(modem);
}

void lte_link_down(struct lte_modem_st * modem)
{
	if(modem->link_up == 0) return;
	printk("******** LINK DOWN ********\n");
	modem->link_status = 0;
	modem->link_up = 0;
	lte_send_ap_link_status(modem);
	lte_send_ap_link_down(modem);
	modem->network_connected = 100; //unknown network type
	lte_send_ap_network_type(modem);
}

//ping packet for 168.95.1.1
static char ping_template[74] =
{
	0x68, 0xEB, 0xAE, 0x38, 0xE6, 0xC7, 0x68, 0xEB, 0xAE, 0x38, 
	0xE6, 0xC6, 0x08, 0x00, 0x45, 0x00, 0x00, 0x3C, 0x19, 0x62,
	0x00, 0x00, 0x7F, 0x01, 0x7D, 0xF4, 0x59, 0x09, 0xA2, 0x01,
	0xA8, 0x5F, 0x01, 0x01, 0x08, 0x00, 0xB2, 0xC1, 0x03, 0x00,
	0x97, 0x9A, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
	0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72,
	0x73, 0x74, 0x75, 0x76, 0x77, 0x61, 0x62, 0x63, 0x64, 0x65, 
	0x66, 0x67, 0x68, 0x69 
};

/* checksum */
static unsigned short checksum( void *ptr, int len)
{
    char        *buf;
    unsigned int  sum = 0;
    unsigned short * uph;
    int            i, cnt = len/2;
  
    if ( len % 2 ) {  // odd length
        buf = (char *)ptr;
        buf[len] = 0x0;
        cnt++;
    }
  
    uph = (unsigned short *)ptr;
  
    for ( i=0; i<cnt; i++ ) {
        sum += uph[i];
    }
  
    sum = (sum & 0xFFFF) + (sum >> 16);
    sum = ~(sum + (sum >> 16)) & 0xFFFF;
  
    if ( sum == 0xffff )
        sum = 0;
  
    return sum;
}

void setup_ping_dns_packet(struct lte_modem_st * modem)
{
    unsigned short sum, s;

	//update source IP
	memcpy(&modem->ping_packet[26], modem->my_ip, 4);

	//update destination IP to DNS 
	memcpy(&modem->ping_packet[30], modem->dns_ip, 4);

	//update destination mac address
	memcpy(&modem->ping_packet[0], modem->bssid, 6);

	//update source mac address
	memcpy(&modem->ping_packet[6], ((struct net_device *)modem->net_dev)->dev_addr, 6);

    //IP checksum
    modem->ping_packet[24] = modem->ping_packet[25] = 0;

    sum = checksum( &modem->ping_packet[14], 20);
    s = htons(sum);

    modem->ping_packet[24] = (s & 0xFF00) >> 8;
    modem->ping_packet[25] = s & 0xFF;

 //   printk("Ping IP chksum = 0x%02X %02X\n", (unsigned char)modem->ping_packet[24], (unsigned char)modem->ping_packet[25]);

    //ICMP check sum
    modem->ping_packet[36] = modem->ping_packet[37] = 0;

    sum = checksum( &modem->ping_packet[34], 40);
    s = htons(sum);
    modem->ping_packet[36] = (s & 0xFF00) >> 8;
    modem->ping_packet[37] = s & 0xFF;

//    printk("Ping ICMP chksum = 0x%02X %02X\n", (unsigned char)modem->ping_packet[36], (unsigned char)modem->ping_packet[37]);

}

static void ping_dns(struct lte_modem_st * modem)
{
	struct sk_buff * skb;
	unsigned long flags;

	skb = dev_alloc_skb(sizeof(ping_template));
	if(!skb)
	{
		printk("Can't allocate skb in ping_dns()\n");
		return;
	}

	memcpy(skb_put(skb, sizeof(ping_template)), modem->ping_packet,  sizeof(ping_template));

//	spin_lock_irqsave(modem->spin_lock, flags);
	lte_send_to_usb(modem, skb);
//	spin_unlock_irqrestore(modem->spin_lock, flags);
}
//------------------------------------------------------------------------------------ 
//              device list utilities
//------------------------------------------------------------------------------------
static struct lte_modem_st * usb_if_to_modem(struct usb_interface * usb_if)
{
	struct list_head * p;
	struct lte_modem_st * modem;
	down_interruptible(&dev_list_lock);
	list_for_each(p, &dev_list)
	{
		modem = list_entry(p, struct lte_modem_st, list);
		if(modem->usb_if == usb_if)
		{
			up(&dev_list_lock);
			return modem;
		}
	}
	up(&dev_list_lock);
	return 0;
}

static struct lte_modem_st * usb_dev_to_modem(struct usb_device * usb_dev)
{
	struct list_head * p;
	struct lte_modem_st * modem;
	down_interruptible(&dev_list_lock);
	list_for_each(p, &dev_list)
	{
		modem = list_entry(p, struct lte_modem_st, list);
		if(modem->usb_dev == usb_dev)
		{
			up(&dev_list_lock);
			return modem;
		}
	}
	up(&dev_list_lock);
	return 0;
}

static struct lte_modem_st * usb_timer_to_modem(void * timer)
{
	struct list_head * p;
	struct lte_modem_st * modem;
	down_interruptible(&dev_list_lock);
	list_for_each(p, &dev_list)
	{
		modem = list_entry(p, struct lte_modem_st, list);
		if(modem->task_level_timer == timer)
		{
			up(&dev_list_lock);
			return modem;
		}
	}
	up(&dev_list_lock);
	return 0;
}

struct lte_modem_st * if_name_to_modem(char * if_name)
{
	struct list_head * p;
	struct lte_modem_st * modem;

	down_interruptible(&dev_list_lock);
	list_for_each(p, &dev_list)
	{
		modem = list_entry(p, struct lte_modem_st, list);
		if(strcmp( ((struct net_device *)modem->net_dev)->name, if_name) == 0)
		{
			up(&dev_list_lock);
			return modem;
		}
	}
	up(&dev_list_lock);
	return 0;
}

static void print_dev_list(void)
{
	struct list_head * p;
	struct lte_modem_st * modem;

	down_interruptible(&dev_list_lock);
	list_for_each(p, &dev_list)
	{
		modem = list_entry(p, struct lte_modem_st, list);
		printk("modem %s\n", ((struct net_device *)modem->net_dev)->name);
	}
	up(&dev_list_lock);
	return;
}
//------------------------------------------------------------------------------------ 
//              AT command utility
//------------------------------------------------------------------------------------
int send_at_cmd_sync(struct lte_modem_st * modem, char * at_cmd)
{
    int i, act_count, res, len;
	char * buf, cmd_buf[20];

	if(modem->disconnected) return -1;

	if(modem->at_rsp_buf) modem->at_rsp_buf[0] = 0; //empty response string first

	buf = kmalloc(1024, GFP_ATOMIC);

	if(!buf)
	{
		printk("Can't get buffer in send_at_cmd_sync()\n");
		return -1;
	}

#if LG_VL600_LTE_SUPPORT == 1
	if(modem->vid == 0x1004 && modem->pid == 0x61AA)
	{
		len = lg_vl600_at_cmd_process(modem, buf, at_cmd);
	}
	else
#endif
	{
		strcpy(buf, at_cmd);
		len = strlen(buf);
	}

    modem->got_rsp = 0;   //this must be before sending AT command, otherwise could have timing issue
//	lte_dump_buffer(buf, len, "------->");
	res = usb_bulk_msg(modem->usb_dev, modem->at_cmd_pipe, buf, len, &act_count, 1000);
	if(res < 0)
	{
		printk("send AT cmd %s failed in send_at_cmd_sync()\n", at_cmd);
		kfree(buf);
		reset_device(modem);
		return -1;
	}
    
    printk("--> %s\n", at_cmd);
	kfree(buf);
    i = 0;
	while(modem->got_rsp == 0 && i < 300 && !modem->disconnected)
    {
		lte_delay_ms(200); //delay 200 ms
        i++;
    }

	switch(modem->got_rsp)
	{
	case 0://timeout
        printk("AT cmd %s timeout in send_at_cmd_sync()\n", at_cmd);

		//serious problem, power cycle modem
		reset_device(modem);

		return 0;

	case 3://ERROR
		break;

	case 1://OK
		break;

	case 2://ATTACHDONEIND
		break;
	}

	lte_delay_ms(1000); //delay 1 s

    return 0;
}

int send_at_cmd_rsp_sync(struct lte_modem_st * modem, char * at_cmd, char * rsp_buf, int rsp_buf_len)
{
    int i, act_count, res, len;
	char * buf;

	buf = kmalloc(1024, GFP_ATOMIC);

	if(!buf)
	{
		printk("Can't get buffer in send_at_cmd_sync()\n");
		return -1;
	}

	if(strlen(at_cmd) == 0) goto read_rsp_only;

#if LG_VL600_LTE_SUPPORT == 1
	if(modem->vid == 0x1004 && modem->pid == 0x61AA)
	{
		len = lg_vl600_at_cmd_process(modem, buf, at_cmd);
	}
	else
#endif
	{
		strcpy(buf, at_cmd);
		len = strlen(buf);
	}

	if(usb_bulk_msg(modem->usb_dev, modem->at_cmd_pipe, buf, len, &act_count, 1000) < 0 )
	{
        printk("send AT cmd %s failed in send_at_cmd_rsp_sync()\n", at_cmd);
 		kfree(buf);
        return -1;
	}

    printk("--> %s\n", at_cmd);

read_rsp_only:

	if((res = usb_bulk_msg(modem->usb_dev, modem->at_rsp_pipe, buf, 1024, &act_count, 5000)) < 0 )
	{
        printk("AT cmd %s timeout in send_at_cmd_rsp_sync(), res = %d\n", at_cmd, res);
		kfree(buf);
		reset_device(modem);
		return -1;
	}

#if LG_VL600_LTE_SUPPORT == 1
	if(modem->vid == 0x1004 && modem->pid == 0x61AA)
	{		
		if(buf[12] != 0x11)
		{
			printk("Not an AT response!!!!!!!\n");
			return 0;
		}

		if(act_count < 14)
		{
			printk("AT response buffer too small!!!!!!!\n");
			return 0;
		}
		act_count = (unsigned char) buf[8] | ((unsigned char) buf[9] << 8) | 
			        ((unsigned char) buf[10] << 16) | ((unsigned char) buf[11] << 24); 
		memcpy(rsp_buf, &buf[14], act_count);
	}
	else
#endif
	{
		memcpy(rsp_buf, buf, act_count);
	}

	kfree(buf);

	//zero terminate the response string
	rsp_buf[act_count] = 0;
	printk("<--- %s\n", rsp_buf);

    return act_count;
}

static void at_rsp_comp(struct urb * urb)
{
    int len;
	char * p;
	struct lte_modem_st * modem = urb->context;

//	lte_dump_buffer(urb->transfer_buffer, urb->actual_length, "<---------- AT RESPONSE\n");

	if(modem->disconnected || urb->status == -ESHUTDOWN /*|| urb->status == -EPROTO*/) return;
//    printk("at_rsp_comp() status = %d\n", urb->status);		
    if(urb->status != 0)
	{
        printk("at_rsp_comp() error, status = %d\n", urb->status);		
	}

	if(urb->status == 0)
	{
		p = (char *)urb->transfer_buffer;
		len = urb->actual_length;

#if LG_VL600_LTE_SUPPORT == 1
		if(modem->vid == 0x1004 && modem->pid == 0x61AA)
		{
			switch(p[12])
			{
			case 0x11:
				len = (unsigned char) p[8] | ((unsigned char) p[9] << 8) | 
			        ((unsigned char) p[10] << 16) | ((unsigned char) p[11] << 24); 

				process_at_rsp(modem, p + 14, len);
				break;

			case 0x21:
				LTE_PRINT(("<------ VL600 P Response 0x%02X%02X\n", (unsigned char)p[14], (unsigned char)p[15]));
//				lte_dump_buffer(p, len, "");
				break;

			default:
				printk("VL600: unknown type!!!!\n");
				break;
			}
		}
		else
#endif
		{
			process_at_rsp(modem, p, len);
		}

	}

	if(usb_submit_urb(urb, GFP_ATOMIC) < 0)
	{
        printk("resubmit transfer failed in at_rsp_comp()\n");
        return;
	}
}

int start_at_rsp_rx(struct lte_modem_st * modem)
{
	modem->at_rsp_urb = usb_alloc_urb(0, GFP_KERNEL);
	if(!modem->at_rsp_urb)
	{
		printk("Can not allocate urb in start_at_rsp_rx()\n");
		return -1;
	}

	modem->at_rsp_buf = kmalloc(4096, GFP_KERNEL);
	if(!modem->at_rsp_buf)
	{
		printk("Can not allocate buffer in start_at_rsp_rx()\n");
		usb_free_urb(modem->at_rsp_urb);
		return -1;
	}

	//empty response string
	modem->at_rsp_buf[0] = 0;  

	usb_fill_bulk_urb(modem->at_rsp_urb, modem->usb_dev,  modem->at_rsp_pipe,
		              modem->at_rsp_buf, 4096, at_rsp_comp, modem);

	if(usb_submit_urb(modem->at_rsp_urb, GFP_KERNEL) < 0)
	{
		printk("submit urb failed in start_at_rsp_rx()\n");
		kfree(modem->at_rsp_buf);
		usb_free_urb(modem->at_rsp_urb);
		return -1;
	}
	printk("Async rx at response start...\n");
	return 0;

}

int stop_at_rsp_rx(struct lte_modem_st * modem)
{
	if(modem->at_rsp_urb)
	{
		usb_kill_urb(modem->at_rsp_urb);
		usb_free_urb(modem->at_rsp_urb);
	}
	if(modem->at_rsp_buf)
	{
		kfree(modem->at_rsp_buf);
	}
}

//------------------------------------------------------------------------------------ 
//              USB driver information
//------------------------------------------------------------------------------------

static struct usb_device_id lte_ids[] = 
{
#if SAMSUNG_LTE_SUPPORT == 1       
	{ USB_DEVICE(0x04e8, 0x689A) }, //Samsung B3730 LTE modem (storage mode)
	{ USB_DEVICE(0x04E8, 0x6889) }, //Samsung B3730 LTE modem
	{ USB_DEVICE(0x04E8, 0x6808) }, //Something wrong happened
#endif
#if LG_VL600_LTE_SUPPORT == 1       
	{ USB_DEVICE(0x1004, 0x61AA) }, //LG_VL600 LTE modem
#endif
	{ } /* Terminating entry */ 
};

MODULE_DEVICE_TABLE(usb, lte_ids);

static struct usb_driver lte_usb_driver = {
	.name = "LTE modem driver",
	.id_table = lte_ids,
	.probe = lte_probe,
	.disconnect = lte_disconnect,
};

//------------------------------------------------------------------------------------ 
//              Network device methods
//------------------------------------------------------------------------------------
static int lte_net_open (struct net_device *net)
{
	LTE_TRACE(("----> lte_net_open()\n"));

	//all thing's done, start packet transfer
	netif_start_queue(net);
	LTE_TRACE(("<---- lte_net_open()\n"));
	return 0;
}

static int lte_net_close (struct net_device *net)
{
	LTE_TRACE(("----> lte_net_close()\n"));
	//the first thing is to stop packet transfer 
	netif_stop_queue(net);

	LTE_TRACE(("<---- lte_net_close()\n"));
	return 0;
}

static int lte_net_start_xmit (struct sk_buff *skb, struct net_device *net_dev)
{
	struct lte_modem_st * modem;
	unsigned long flags;
//	LTE_TRACE(("----> lte_net_start_xmit()\n"));
//	printk("head room = %d\n", skb_headroom(skb));
//	lte_dump_buffer((unsigned char *)skb->data, skb->len, "---> Tx Packet");
	modem = (struct lte_modem_st *) netdev_priv(net_dev);
	spin_lock_irqsave(modem->spin_lock, flags);

	if(!modem)
	{
		printk("Serious error! Can't find modem in lte_net_start_xmit\n");
		spin_unlock_irqrestore(modem->spin_lock, flags);
		dev_kfree_skb_any(skb);
		return 0;
	}

	if(modem->disconnected) 
	{
		spin_unlock_irqrestore(modem->spin_lock, flags);
		printk("try to send packet when modem is disconnected\n");
		dev_kfree_skb_any(skb);
		return 0;
	}

	lte_send_to_usb(modem, skb);

	spin_unlock_irqrestore(modem->spin_lock, flags);
//	LTE_TRACE(("<---- lte_net_start_xmit()\n"));
	return 0;
}

#if 0
static int lte_net_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
	struct lte_ioctl_data_st *io_data;
	struct lte_modem_st * modem;
	int res;

	LTE_TRACE(("----> lte_net_ioctl()\n"));
    modem = (struct lte_modem_st *)netdev_priv((struct net_device *)dev);
	if(modem->disconnected) return -ENODEV;
    io_data = &modem->io_data;
	switch (cmd) 
	{
		case LTE_NET_READ:
			modem->io_data_valid = 0;
			copy_to_user(ifr->ifr_data, io_data, sizeof(struct lte_ioctl_data_st));
			break;

		case LTE_NET_WRITE:
			copy_from_user(io_data, ifr->ifr_data, sizeof(struct lte_ioctl_data_st));
			LTE_PRINT(("%s\n", io_data->buf));
			break;

		default:
			printk("Wrong I/O control code!!!!!\n");
			break;
	}
	LTE_TRACE(("<---- lte_net_ioctl()\n"));
	return 0;
}
#endif

int lte_netif_rx(struct lte_modem_st * modem, char * buf, int len)
{
	struct sk_buff * skb;

#if LTE_DRIVER_DUMP_TX_RX == 1
	lte_dump_buffer(buf, len, "<------------- in packet");
#endif //LTE_DRIVER_DUMP_TX_RX == 1

	skb = dev_alloc_skb( len + 2);
	if(!skb)
	{
		printk("Can't get sk buffer in samsung_lte_rx_process() 1\n");
		((struct net_device_stats *)modem->statistic)->rx_dropped++;
		return -1;
	}

	//let the IP header at 4 byte boundary
	skb_reserve(skb, 2);		

	//copy data to skb
	memcpy(skb_put(skb, len), buf, len);

#if 0 //dump ping reply for 168.95.1.1
	if(skb->data[30 - 4] == 0xA8 && skb->data[31 - 4] == 0x5F)
	{
		printk("<--- reply to ping 168.95.1.1\n");
//		lte_dump_buffer(skb->data, skb->len, "");
	}
#endif

	if(skb->data[30 - 4] == modem->dns_ip[0] && skb->data[31 - 4] == modem->dns_ip[1])
	{
//		printk("<--- Reply to ping DNS\n");
		modem->keep_alive_timer = 0;
//		lte_dump_buffer(skb->data, skb->len, "");
	}

//	lte_dump_buffer(skb->data, skb->len, "<-----in packet");
	skb->dev = modem->net_dev;
	skb->protocol = eth_type_trans(skb, modem->net_dev);
	skb->ip_summed = CHECKSUM_UNNECESSARY;
//	lte_dump_buffer(skb->data, skb->len, "<-----in packet");
	((struct net_device_stats *)modem->statistic)->rx_packets ++;
	((struct net_device_stats *)modem->statistic)->rx_bytes += skb->len;
	netif_rx(skb);
	return 0;
}

static struct net_device_stats * lte_net_get_stats(struct net_device *dev)
{
	struct lte_modem_st * modem;
	LTE_TRACE_P(("----> lte_net_get_stats()\n"));
	modem = (struct lte_modem_st *) netdev_priv(dev);
	LTE_TRACE_P(("<---- lte_net_get_stats()\n"));
	//return pointer of net_device_stats
	return modem->statistic;
}

static void lte_tx_timeout(struct net_device *dev)
{
	struct lte_modem_st * modem;
	printk("LTE tx timeout\n");
	modem = (struct lte_modem_st *) netdev_priv(dev);
	if (modem->current_tx_urb && ((struct urb *)modem->current_tx_urb)->status == -EINPROGRESS)
	{
		modem->current_tx_urb = 0;
		usb_unlink_urb(modem->current_tx_urb);
	}
}

static void lte_net_setup(struct net_device *dev)
{
	LTE_TRACE(("----> lte_net_setup()\n"));
	dev->open = lte_net_open;
	dev->stop = lte_net_close;
	dev->hard_start_xmit = lte_net_start_xmit;
	dev->tx_timeout = lte_tx_timeout;
	dev->watchdog_timeo = 2*HZ;
	dev->do_ioctl = 0;//lte_net_ioctl;
	dev->get_stats = lte_net_get_stats;
	//NOTE: %% is used to print %
	sprintf(dev->name, "%s%%d", NET_DEV_NAME);
	memset(dev->broadcast,0xFF, ETH_ALEN);
	dev->mtu                = 1360;
	dev->tx_queue_len = 100;   //this will be changed to modem->act_tx_buf_count, if LTE_USE_PRE_ALLOC_TX_BUF is 1
	LTE_TRACE(("<---- lte_net_setup()\n"));
}

void lte_send_ap_modem_power_recycle(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_power_recycle()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_MODEM_POWER_RECYCLE;
	lte_tell_app(event);
	return;
}

void lte_send_ap_modem_connected(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_modem_connected()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_MODEM_CONNECTED;
	lte_tell_app(event);
	return;
}

void lte_send_ap_modem_disconnected(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_modem_connected()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_MODEM_DISCONNECTED;
	lte_tell_app(event);
	return;
}

void lte_send_ap_hw_ver(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_hw_ver()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_HW_VERSION;
	strcpy(event->data.hw_ver, modem->hw_ver);
	lte_tell_app(event);
	return;
}

void lte_send_ap_sw_ver(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_sw_ver()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_SW_VERSION;
	strcpy(event->data.sw_ver, modem->sw_ver);
	lte_tell_app(event);
	return;
}

void lte_send_ap_mac(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_mac()\n");
		return;
	}

	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_MAC_ADDR;
	memcpy(event->data.mac, ((struct net_device *)modem->net_dev)->dev_addr, 6);
	lte_tell_app(event);
	return;
}

void lte_send_ap_bssid(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_bssid()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_BSSID;
	memcpy(event->data.bssid, modem->bssid, 6);
	lte_tell_app(event);
	return;
}

void lte_send_ap_link_status(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_link_status()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_LINK_STATUS;
	event->data.link_info.rssi = modem->rssi;
	event->data.link_info.cinr = modem->cinr;
	event->data.link_info.dbm = modem->dbm;
	event->data.link_info.link_status = modem->link_status;
	lte_tell_app(event);
	return;
}

void lte_send_ap_link_up(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_link_up()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_LINK_UP;
	lte_tell_app(event);
	return;
}

void lte_send_ap_link_down(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_link_down()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_LINK_DOWN;
	lte_tell_app(event);
	return;
}

void lte_send_ap_set_pin_invalid(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_set_pin_invalid()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_SET_PIN_INVALID;
	lte_tell_app(event);
	return;
}

void lte_send_ap_rx_packet(struct lte_modem_st * modem, unsigned char * buf, int len)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_rx_packet()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_PACKET;
	event->data.packet.len = len;
    memcpy(event->data.packet.body, buf, len);
	lte_tell_app(event);
	return;
}

void lte_send_ap_network_type(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_network_type()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_NETWORK_TYPE;
	event->data.network_type = modem->network_connected;
	lte_tell_app(event);
	return;
}

void lte_send_ap_get_config(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_get_config()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_GET_CONFIG;
	lte_tell_app(event);
	return;
}

void lte_send_ap_get_dns_ip(struct lte_modem_st * modem)
{
	struct lte_event_st * event;
	event = (struct lte_event_st *) kmalloc(sizeof(struct lte_event_st), GFP_ATOMIC);
	if(!event)
	{
		printk("Can not allocate event buffer in lte_send_ap_get_config()\n");
		return;
	}
	strcpy(event->if_name, ((struct net_device *)modem->net_dev)->name);
	event->type = LTE_GET_MY_DNS_IPS;
	lte_tell_app(event);
	return;
}

#if LTE_TEST_DAEMON == 1
static int test_count = 0;
void test_user_if(struct lte_modem_st * modem)
{
	unsigned char mac[6] = {0x00, 0x50, 0x7F, 0x01, 0x02, 0x03};
	unsigned char bssid[6] = {0x11, 0x22, 0x33, 0x44, 0x55, 0x66};
	unsigned char packet[4] = {0x55, 0xAA, 0x55, 0xAA};

//	if((test_count++ % 5) == 0)
	{
		strcpy(modem->hw_ver, "Hardware Ver 1.0");
		lte_send_ap_hw_ver(modem);

		strcpy(modem->sw_ver, "software Ver 2.3.4");
		lte_send_ap_sw_ver(modem);

		memcpy(((struct net_device *)modem->net_dev)->dev_addr, mac, 6);
		lte_send_ap_mac(modem);

		memcpy(modem->bssid, bssid, 6);
		lte_send_ap_bssid(modem, bssid);

		lte_send_ap_link_up(modem);

		modem->link_status = 2;
		modem->rssi = -65;
		modem->cinr = 30;
		modem->dbm = 24
		lte_send_ap_link_status(modem);

		lte_send_ap_rx_packet(modem, packet, 4);

		lte_send_ap_link_down(modem);
	}
	return;
}
#endif // LTE_TEST_DAEMON == 1

#if LTE_USE_KERNEL_TIMER == 1
//NOTE: no sleep is allowed in this function
static void lte_timer(unsigned long data)
{
	struct lte_modem_st * modem;
	LTE_TRACE_P(("----> lte_timer\n"));
    modem = (struct lte_modem_st *) data;

#if LTE_TEST_DAEMON == 1
	test_user_if(modem);
#endif

	if(modem->disconnected)
	{
		//modem is disconnected, don't queue delay work any more or the driver will crash
		return;
	}


	//restart timer if modem is still connected
    ((struct timer_list *)modem->timer)->function = lte_timer;
	((struct timer_list *)modem->timer)->data = (unsigned long)modem;
	((struct timer_list *)modem->timer)->expires = jiffies + 1 * HZ;
	add_timer(modem->timer);
	LTE_TRACE_P(("<---- lte_timer\n"));
}
#endif //LTE_USE_KERNEL_TIMER == 1

//NOTE: sleep is allowed in this function
static void lte_task_level_timer_exe(struct delayed_work * work)
{
	struct lte_modem_st * modem;

	LTE_TRACE_P(("----> lte_task_level_timer_exe()\n"));
	modem = (struct lte_modem_st * )usb_timer_to_modem(work);

	if(!modem)
	{
		LTE_TRACE_P(("<---- lte_task_level_timer_exe(), can't find modem object\n"));
		return;
	}

    modem->counter++;

	//NOTE: ping DNS from the driver directly will cause ARP table incorrect !!!
    //ping dns to keep alive is done by lte_d
	if(modem->link_up && (modem->counter % 3) == 0 )
	{
		//ask modem daemon for my IP and DNS IP
		if(!modem->my_ip_dns_ip_valid) lte_send_ap_get_dns_ip(modem);

		//ping DNS if we have IPs
//		if(modem->my_ip_dns_ip_valid) ping_dns(modem);
	}

#if LTE_TEST_DHCPC==1
	if(!modem->link_up)
	{
		modem->link_up = 1;
		modem->my_ip_dns_ip_valid = 1;
		lte_send_ap_link_up(modem);
	}
#endif

#if LTE_TEST_DAEMON == 1
	test_user_if(modem);
#endif

	if(!modem->ready)
	{
		if(modem->get_config) modem->get_config(modem);
		if(modem->report_info) modem->report_info(modem);
		modem->ready = 1;
	}


	if(modem->timer_exe)
	{
		modem->timer_exe(modem);
	}
	else
	{
		printk("Serious error! modem has no timer method\n");
	}

	if(modem->disconnected)
	{
		//modem is disconnected, don't queue delay work any more or the driver will crash
		return;
	}

	//re-queue delay work if modem is still connected
	queue_delayed_work(modem->work_q, work, 1*HZ);
	LTE_TRACE_P(("<---- lte_task_level_timer_exe()\n"));
	return;
}

void lte_tx_comp(void * urb)
{
	struct lte_modem_st * modem;

	if( ((struct urb *)urb)->status != 0)
	{
		printk("lte_tx_comp() with error = %d, act len = %d\n", 
			((struct urb *)urb)->status, ((struct urb *)urb)->actual_length);
	}

	modem = (struct lte_modem_st *) ((struct urb *)urb)->context;
#if LTE_USE_PRE_ALLOC_TX_BUF == 1
	modem->tx_buf_left++;
#else //LTE_USE_PRE_ALLOC_TX_BUF == 1

	usb_buffer_free(modem->usb_dev, 2048, ((struct urb *)urb)->transfer_buffer, ((struct urb *)urb)->transfer_dma);
	usb_free_urb(urb);
#endif //LTE_USE_PRE_ALLOC_TX_BUF == 1

	modem->current_tx_urb = 0;
	netif_wake_queue(modem->net_dev);
}

static void data_rx_comp(struct urb * urb)
{
	struct lte_modem_st * modem = (struct lte_modem_st *) urb->context;
//	printk("data_rx_comp() %d bytes, status = %d\n", urb->actual_length, urb->status);
	if(modem->disconnected || urb->status == -ESHUTDOWN/* || urb->status == -EPROTO*/)
	{
//		printk("data_rx_comp() stop resubmit urb\n");
		return;
	}

	if(urb->status == 0)
	{
		if(modem->rx_process)
		{
//			lte_dump_buffer(urb->transfer_buffer, urb->actual_length, "<--- rx buffer");
			modem->rx_process(modem, urb->transfer_buffer, urb->actual_length);
		}
		else
		{
			printk("Serious error! modem has no rx method\n");
		}
	}
	else
	{
        printk("data_rx_comp() error, status = %d\n", urb->status);		
	}

	if(usb_submit_urb(urb, GFP_ATOMIC) < 0)
	{
        printk("resubmit transfer failed in at_rsp_comp()\n");
        return;
	}
}

int start_data_rx(struct lte_modem_st * modem)
{
    int i, mask, one_count;

	if(modem->config_rx_buf)
	{
		modem->config_rx_buf(modem);
	}
	else
	{
		printk("Serious error! modem has no config rx buffer method\n");
		return -1;
	}

	if(modem->act_rx_buf_count > LTE_MAX_DATA_RX_BUF)
	{
		printk("Serious error! rx buffer count too large\n");
		return -1;
	}

	if((modem->rx_head_room & 3) != 0)
	{
		printk("Serious error! rx buffer header room make the IP header not at 4 byte boundary\n");
		return -1;
	}

	printk("Rx buffer count = %d, size = %d\n", modem->act_rx_buf_count, modem->rx_buf_size);
	for(i = 0; i < LTE_MAX_DATA_RX_BUF; i++)
	{
		modem->data_rx_urb[i] = 0;
		modem->data_rx_skb[i] = 0;
		modem->data_rx_dma_buf[i] = 0;
	}

	for(i = 0; i < modem->act_rx_buf_count; i++)
	{
		modem->data_rx_urb[i] = usb_alloc_urb(0, GFP_KERNEL);
		if(!modem->data_rx_urb[i])
		{
			printk("Can not allocate rx urb in start_data_rx()\n");
			goto fail;
		}

		if(modem->rx_use_dma_buf)
		{
			modem->data_rx_dma_buf[i] = usb_buffer_alloc(modem->usb_dev, modem->rx_buf_size, GFP_KERNEL | GFP_DMA, &((struct urb *)modem->data_rx_urb[i])->transfer_dma);
			if(!modem->data_rx_dma_buf[i])
			{
				printk("Can not allocate rx buffer in start_data_rx()\n");
				goto fail;
			}

			usb_fill_bulk_urb(modem->data_rx_urb[i], modem->usb_dev,  modem->data_in_pipe,
						modem->data_rx_dma_buf[i], modem->rx_buf_size, data_rx_comp, modem);
			((struct urb *)modem->data_rx_urb[i])->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		}
		else
		{
			modem->data_rx_skb[i] = dev_alloc_skb(modem->rx_buf_size + modem->rx_head_room + 2);
			if(!modem->data_rx_skb[i])
			{
				printk("Can not allocate rx skb in start_data_rx()\n");
				goto fail;
			}

			//let the IP header at 4 byte boundary and reserve some head room
			skb_reserve((struct sk_buff*)modem->data_rx_skb[i], modem->rx_head_room + 2);		

			usb_fill_bulk_urb(modem->data_rx_urb[i], modem->usb_dev,  modem->data_in_pipe,
						      ((struct sk_buff*) modem->data_rx_skb[i])->data, modem->rx_buf_size, data_rx_comp, modem);
		}

		if(usb_submit_urb(modem->data_rx_urb[i], GFP_KERNEL) < 0)
		{
			printk("submit rx urb failed in start_data_rx()\n");
			goto fail;
		}
	}

#if LTE_USE_PRE_ALLOC_TX_BUF == 1
	if(modem->act_tx_buf_count > LTE_MAX_DATA_TX_BUF)
	{
		printk("Serious error! tx buffer count too large\n");
		return -1;
	}

	mask = 1;
	one_count = 0;
	for(i = 0; i < 16; i++)
	{
		if(modem->act_tx_buf_count & mask) one_count++;
		mask = mask << 1;
		if(one_count > 1) 
		{
			printk("Serious error! tx buffer count not a power of 2\n");
			return -1;
		}
	}

	((struct net_device *)modem->net_dev)->tx_queue_len = modem->act_tx_buf_count;

	for(i = 0; i < LTE_MAX_DATA_TX_BUF; i++)
	{
		modem->data_tx_urb[i] = 0;
		modem->data_tx_dma_buf[i] = 0;
	}

	for(i = 0; i < modem->act_tx_buf_count; i++)
	{
 		modem->data_tx_urb[i] = usb_alloc_urb(0, GFP_KERNEL);
		if(!modem->data_tx_urb[i])
		{
			printk("Can not allocate tx urb in start_data_rx()\n");
			goto fail;
		}

		((struct urb *)modem->data_tx_urb[i])->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;

		modem->data_tx_dma_buf[i] = usb_buffer_alloc(modem->usb_dev, modem->tx_buf_size, GFP_KERNEL | GFP_DMA, 
			                        &((struct urb *)modem->data_tx_urb[i])->transfer_dma);
 		if(!modem->data_tx_dma_buf[i])
		{
			printk("Can not allocate tx buffer in start_data_rx()\n");
			goto fail;
		}
	}
#endif //LTE_USE_PRE_ALLOC_TX_BUF == 1

	printk("Async rx data start...\n");
	return 0;

fail:
	for(i = 0; i < LTE_MAX_DATA_RX_BUF; i++)
	{
		if(modem->data_rx_urb[i]) 
		{
			if(modem->data_rx_skb[i])
			{
				dev_kfree_skb_any((struct sk_buff*)modem->data_rx_skb[i]);
			}

			usb_kill_urb(modem->data_rx_urb[i]);
			if(modem->data_rx_dma_buf[i])
			{
				usb_buffer_free(modem->usb_dev, modem->rx_buf_size, modem->data_rx_dma_buf[i], ((struct urb *)modem->data_rx_urb[i])->transfer_dma);
			}

			usb_free_urb(modem->data_rx_urb[i]);
		}
	}

#if LTE_USE_PRE_ALLOC_TX_BUF == 1
	for(i = 0; i < LTE_MAX_DATA_TX_BUF; i++)
	{
		if(modem->data_tx_urb[i]) 
		{
			if(modem->data_tx_dma_buf[i])
			{
				usb_buffer_free(modem->usb_dev, modem->tx_buf_size, modem->data_tx_dma_buf[i], ((struct urb *)modem->data_tx_urb[i])->transfer_dma);
			}

			usb_free_urb(modem->data_tx_urb[i]);
		}
	}
#endif //LTE_USE_PRE_ALLOC_TX_BUF == 1
	return -1;
}

int stop_data_rx(struct lte_modem_st * modem)
{
	int i;

	for(i = 0; i < LTE_MAX_DATA_RX_BUF; i++)
	{
		if(modem->data_rx_urb[i]) 
		{
			if(modem->data_rx_skb[i]) 
			{
				dev_kfree_skb_any((struct sk_buff*)modem->data_rx_skb[i]);
			}

			usb_kill_urb(modem->data_rx_urb[i]);

			if(modem->data_rx_dma_buf[i])
			{
				usb_buffer_free(modem->usb_dev, modem->rx_buf_size, modem->data_rx_dma_buf[i], ((struct urb *)modem->data_rx_urb[i])->transfer_dma);
//				kfree(modem->data_rx_dma_buf[i]);
			}

			usb_free_urb(modem->data_rx_urb[i]);
		}
	}

#if LTE_USE_PRE_ALLOC_TX_BUF == 1
	for(i = 0; i < LTE_MAX_DATA_TX_BUF; i++)
	{
		if(modem->data_tx_urb[i]) 
		{
			if(modem->data_tx_dma_buf[i])
			{
				usb_buffer_free(modem->usb_dev, modem->tx_buf_size, modem->data_tx_dma_buf[i], ((struct urb *)modem->data_tx_urb[i])->transfer_dma);
			}

			usb_free_urb(modem->data_tx_urb[i]);
		}
	}
#endif //LTE_USE_PRE_ALLOC_TX_BUF == 1

}

static int storage_mode(unsigned short vid, unsigned short pid)
{
	unsigned long vid_pid;

	vid_pid = (vid << 16) | pid;
	switch(vid_pid)
	{
		case 0x04E8689A:
			return 1;

		default:
			return 0;
	}
}

static int switch_to_modem(unsigned short vid, unsigned short pid, struct usb_device * usb_dev)
{
	unsigned long vid_pid;

	vid_pid = (vid << 16) | pid;
	switch(vid_pid)
	{
		case 0x04E8689A:
			printk("Samsung B3730 storage mode, switch to modem mode\n");
			return samsung_lte_switch_to_modem(usb_dev);

		default:
			return 0;
	}
}

static int init_modem(unsigned short vid, unsigned short pid, struct lte_modem_st * modem)
{
	unsigned long vid_pid;

	vid_pid = (vid << 16) | pid;
	switch(vid_pid)
	{
		case 0x04E86889:
			printk("Samsung B3730 modem mode\n");
		    return init_samsumg_lte(modem);

#if LG_VL600_LTE_SUPPORT == 1
		case 0x100461AA:
			printk("LG VL600 modem mode\n");
		    return init_lg_vl600_lte(modem);
#endif
		default:
			return 0;
	}
}

static int lte_probe (struct usb_interface *usb_if, const struct usb_device_id *id)
{
	int res, vid, pid;
	struct usb_device * usb_dev;
	struct lte_modem_st * modem;
    struct net_device *net_dev = 0;

	LTE_TRACE(("----> lte_probe(), vid = 0x%04X pid = 0x%04X interface #%d altsetting #%d\n", 
		        id->idVendor, id->idProduct,
				usb_if->cur_altsetting->desc.bInterfaceNumber, 
				usb_if->cur_altsetting->desc.bAlternateSetting));
#if 0
	if((id->idVendor==0x04e8) && (id->idProduct==0x6808)){
		draytek_usb_power_recycle(500);
		return -ENODEV;
	}
#endif

	vid = id->idVendor;
	pid = id->idProduct;
	usb_dev = interface_to_usbdev(usb_if);

	if(storage_mode(vid, pid)) 
	{
		switch_to_modem(vid, pid, usb_dev);
		return 0;
	}


	modem = usb_dev_to_modem(usb_dev);
	if(modem)
	{
		LTE_TRACE(("<---- lte_probe(), modem already probed\n"));
		return 0;
	}

	//allocate net device and modem object together in alloc_etherdev()!!
	net_dev = alloc_etherdev(sizeof(struct lte_modem_st));
	if (!net_dev)
	{
		printk("Can't allocate LTE net device\n");
		return -ENOMEM;
	}
	lte_net_setup(net_dev);
	res = register_netdev(net_dev);
	if(res)
	{
		printk("Can't register LTE net device\n");
		free_netdev(net_dev);
		return -ENODEV;
	}

	modem = (struct lte_modem_st *)netdev_priv(net_dev);
	memset(modem, 0, sizeof(modem));
	modem->net_dev = net_dev;
    modem->usb_dev = usb_dev;
	modem->usb_if = usb_if;
	modem->vid = vid;
	modem->pid = pid;

#if LTE_USE_KERNEL_TIMER == 1
	modem->timer = kmalloc(sizeof(struct timer_list), GFP_ATOMIC);
	if(!modem->timer)
	{
		printk("modem can not allocate timer_list\n");
		goto fail;
	}
#endif

	modem->spin_lock = kmalloc(sizeof(spinlock_t), GFP_ATOMIC);
	if(!modem->spin_lock)
	{
		printk("modem can not allocate spin lock\n");
		goto fail;
	}

	spin_lock_init((spinlock_t *)modem->spin_lock);
	modem->statistic = kmalloc(sizeof(struct net_device_stats), GFP_ATOMIC);
	if(!modem->statistic)
	{
		printk("modem can not allocate net_device_stats\n");
		goto fail;
	}
    memset(modem->statistic, 0, sizeof(struct net_device_stats));
	modem->task_level_timer = kmalloc(sizeof(struct delayed_work), GFP_ATOMIC);
	if(!modem->task_level_timer)
	{
		printk("modem can not allocate task_level_timer\n");
		goto fail;
	}

	if(init_modem(vid, pid, modem) < 0)
	{
		printk("modem initialization failed\n");
		goto fail;
	}

	if(start_data_rx(modem) < 0)
	{
		printk("modem queue rx urb failed\n");
		return -1;	
	}

	//start 1 second task level timer
	modem->work_q = create_singlethread_workqueue("LTE_WORK_Q");
	if(!modem->work_q)
	{
		printk("Can't create work queue in lte_probe()\n");
		stop_at_rsp_rx(modem);
		stop_data_rx(modem);
		unregister_netdev(net_dev);
		free_netdev(net_dev);
		return -ENODEV;
	}
	INIT_DELAYED_WORK(((struct delayed_work*)(modem->task_level_timer)), (void (*)(struct work_struct * ))lte_task_level_timer_exe);
	queue_delayed_work(modem->work_q, modem->task_level_timer, 1*HZ);

#if LTE_USE_KERNEL_TIMER == 1
	//start 1 second kernel timer
	init_timer(modem->timer);
    ((struct timer_list *)modem->timer)->function = lte_timer;
	((struct timer_list *)modem->timer)->data = (unsigned long)modem;
	((struct timer_list *)modem->timer)->expires = jiffies + 1 * HZ;
	add_timer(modem->timer);
#endif

	//initialie ping packet template
	memcpy(modem->ping_packet, ping_template, sizeof(ping_template));
	
	//add to device list
	down_interruptible(&dev_list_lock);
	list_add(&modem->list, &dev_list);
	up(&dev_list_lock);

	//start to connect 2 seconds later
#if LTE_TEST_DHCPC==0 && LTE_TEST_DAEMON==0
	modem->connect_timer = 2;           
#endif

	LTE_TRACE(("<---- lte_probe()\n"));
	return 0;

fail:
	if(modem->timer) kfree(modem->timer);
	if(modem->statistic) kfree(modem->statistic);
	if(modem->spin_lock) kfree(modem->spin_lock);
	if(modem->task_level_timer) kfree(modem->task_level_timer);
	unregister_netdev(modem->net_dev);
	free_netdev(modem->net_dev);
	return -ENODEV;
}

static void lte_disconnect (struct usb_interface *usb_if)
{
	int vid, pid;
	struct usb_device * usb_dev;
	struct lte_modem_st * modem;

	//storage mode won't have a corresponding modem object in dev_list, we have to 
	//take information from the usb_if passed in first
	usb_dev = interface_to_usbdev(usb_if);
	vid = usb_dev->descriptor.idVendor;
	pid = usb_dev->descriptor.idProduct;
	LTE_TRACE(("----> lte_disconnect(), vid = 0x%04X pid = 0x%04X interface #%d altsetting #%d\n", 
		        vid, pid, usb_if->cur_altsetting->desc.bInterfaceNumber, 
				usb_if->cur_altsetting->desc.bAlternateSetting));

	if(storage_mode(vid, pid))
	{
		LTE_TRACE(("<---- lte_disconnect(), storage mode\n"));
		return;
	}


	//find the modem object for this USB instance
    modem = usb_dev_to_modem(usb_dev);
	if(!modem)
	{
		printk("<---- lte_disconnect(), modem already disconnected\n");
		return;
	}

	modem->disconnected = 1;
	netif_stop_queue(modem->net_dev);
	lte_send_ap_modem_disconnected(modem);

	//delay 1 second for daemon to finish its job
	lte_delay_ms(1000);          


#if LTE_TEST_DHCPC==1
	lte_send_ap_link_down(modem);
#endif

#if LTE_USE_KERNEL_TIMER == 1
	del_timer_sync(modem->timer);
	kfree(modem->timer);
#endif

	stop_at_rsp_rx(modem);
	stop_data_rx(modem);

	if(cancel_delayed_work(modem->task_level_timer) == 0)
	{
		flush_workqueue(modem->work_q);
	}

	destroy_workqueue(modem->work_q);


	unregister_netdev(modem->net_dev);
//	print_dev_list();

	down_interruptible(&dev_list_lock);
	list_del(&modem->list); 
	up(&dev_list_lock);
	kfree(modem->task_level_timer);
	kfree(modem->statistic);
	kfree(modem->spin_lock);
	//NOTE: modem will be freed as well after free_net_dev()
	free_netdev(modem->net_dev);
	LTE_TRACE(("<---- lte_disconnect()\n"));
	return;
}

static int lte_init(void)
{
	int res;

	LTE_TRACE(("----> lte_init()\n"));

    res = usb_register(&lte_usb_driver);
	if(res)
	{
		printk("Can't register LTE USB driver\n");
		return -ENODEV;
	}

    //register a character device for user mode application
	res = register_chrdev(MAJOR_NUM, "LteDriver", &lte_ch_fops);
	if (res < 0) {
		printk("LTE: register character device failed\n");
		return -ENODEV;
	}

	LTE_TRACE(("<---- lte_init()\n"));

	return 0;
}

static void lte_exit(void)
{
	LTE_TRACE(("----> lte_exit()\n"));
    unregister_chrdev(MAJOR_NUM, "LteDriver");
	usb_deregister(&lte_usb_driver);
	LTE_TRACE(("<---- lte_exit()\n"));
}

module_init(lte_init);
module_exit(lte_exit);
MODULE_AUTHOR("Frank Chien/Draytek");
MODULE_DESCRIPTION("LTE Driver");
MODULE_LICENSE("GPL");
