/* -- updatedd: dyndns.c --
 *
 * Copyright (C) 2002, 2003, 2004 Philipp Benner
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

// -------------------------------------------------------------------------
// include files
//
#include "config.h"
#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>

#include "base64encode.h"
#include "get_connection.h"

//#include <openssl/evp.h>

#include "dyndns.h"

// -------------------------------------------------------------------------
// Macro Defines
//

#if ENTERPRISE==1
#define MODEL_NAME "Enterprise Router"
#else
#define MODEL_NAME "Vigor3300 series"
#endif

#define DRAYTEK_SYSLOG_WAN_LOG 0
#define draytek_syslog(x, y) do {} while(0)
#define save_cli_values() do {} while(0)
#define get_cli_value_by_index(x,y,z) do {} while(0)
#define set_cli_value_by_index(x,y,z) do {} while(0)
#define get_cli_value(x,y,z) do {} while(0)

//#define DEBUG

#ifdef DEBUG
#define PDEBUG(x, args...) printf(x, ##args)
#else
#define PDEBUG(x, args...) do {} while(0)
#endif

#define EXPIRE_SEC    10*24*60*60        //ST, dyndns must update in 35days, I use 10 day as expired time
#define MAX_ARGS       10
#define DEFAULT_DELAY   3 //3 seconds
#define MAX_RECONNECT   3

// -------------------------------------------------------------------------
// global variable declarations
//
//presto -- 071304 -- <<
struct arguments DDNS_Args[MAX_ARGS];
//presto -- 071304 -- >>

struct    sigaction sig;
time_t start_timer;
// -------------------------------------------------------------------------
// external global variables and functions
void LogErr(char *fmt, ...);
/* Steven, Add End, 2004/04/22 */
void reset_master_oldip();


// -------------------------------------------------------------------------
// global funcion bodies
//

void sigusr1_handler(int signum)
{
    printf("into sigusr1_handler\n");
    get_arguments_from_cli(0);
}

void sigusr2_handler(int signum)
{
    printf("into sigusr2_handler\n");
    reset_master_oldip(); //fix damn backup bug
}

void reset_master_oldip()
{
    char if_num[8];
    int i;

    if_num[0] = '\0';

    get_cli_value_by_index("DDNS_PID", 1, if_num);

    if(strcmp(if_num, ""))
    {
        for(i = 0;i < MAX_ARGS;++i)
        {
            if(!strcmp(DDNS_Args[i].interface, if_num))
            {
                strcpy(DDNS_Args[i].oldIPaddress, "");
            }
        }
    }

    set_cli_value_by_index("DDNS_PID", 1, "");
}

void LogErr(char *fmt, ...)
{
    va_list ap;
    char    errmsg[BUFSIZE];

    va_start(ap, fmt);
    vsnprintf(errmsg, BUFSIZE-1, fmt, ap);
    va_end(ap);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, errmsg);
    printf("%s\n", errmsg);

    return;
}

void getip(char *p, char *device)
{
    char key[32];

    strcpy(p, "");

    sprintf(key, "NIF_WAN%s_IP_SETTINGS", device);

    get_cli_value_by_index(key, 0, p);

#if 0    
    int     fd;
    struct  sockaddr_in *sin;
    struct  ifreq ifr;
    struct  in_addr z;

    p[0] = 0;        // remove old address
    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        LogErr("Can't talk to kernel! (%d)\n", errno);
        return;
    }
    strcpy(ifr.ifr_name, device);
    
    if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0)  {
        LogErr("Can't get status for %s. (%d)\n", device, errno);
        close(fd);
        return;
    }
    
    if ((ifr.ifr_flags & IFF_UP) == 0)  {
        LogErr("Interface %s not active.\n", device);
        close(fd);
        return;
    }

    if (ioctl(fd, SIOCGIFADDR, &ifr) != 0) {
        LogErr("Can't get IP address for %s. (%d)\n", device, errno);
        //printf("Can't get IP address for %s. (%d)\n", device, errno);
        close(fd);
        return;
    }
    close(fd);
    sin = (struct sockaddr_in *)&ifr.ifr_addr;
    z = sin->sin_addr;
    strcpy(p, inet_ntoa(z));
#endif

    PDEBUG("! Our IP address is %s\n", p);
}


#define DDNS_PARAM_COUNT 11

void get_arguments_from_cli(int init)
{
    char key[32];
    int  i;
    char *value[DDNS_PARAM_COUNT];
    char value_tmp[DDNS_PARAM_COUNT][128];
    int  index_system, index_service_provider;

    for(i = 0;i < DDNS_PARAM_COUNT; ++i) {
        value[i] = value_tmp[i];
        value[i][0] = '\0';
    }

    for(i = 0;i < MAX_ARGS; ++i)
    {
        DDNS_Args[i].enable = 0;
        sprintf(key, "ddns_args_%d", i + 1);

        get_cli_value(key, value, DDNS_PARAM_COUNT);

        DDNS_Args[i].enable = atoi(value[0]);
        strcpy(DDNS_Args[i].interface, value[1]);
        index_service_provider = atoi(value[2]);
        DDNS_Args[i].service_provider = index_service_provider;
        index_system = atoi(value[3]);
        strcpy(DDNS_Args[i].system,   dd_system[index_system]);
        strcpy(DDNS_Args[i].hostname, value[4]);

        strcpy(DDNS_Args[i].login, value[5]);
        strcpy(DDNS_Args[i].password, value[6]);
        if(strcmp(value[7], "0") == 0)
            strcpy(DDNS_Args[i].wildcard, "OFF");
        else                
            strcpy(DDNS_Args[i].wildcard, "ON");

        if(strcmp(value[8], "0") == 0)
            strcpy(DDNS_Args[i].backmx, yn.no);
        else                
            strcpy(DDNS_Args[i].backmx, yn.yes);

        strcpy(DDNS_Args[i].mx, value[9]);

        DDNS_Args[i].update_function = Update_Function_Service_Provider[index_service_provider];
        //value[10] set to "" in cgi
        if(init || !strcmp(value[10], "")) 
        {
            strcpy(DDNS_Args[i].IPaddress, "");
            strcpy(DDNS_Args[i].oldIPaddress, "");
            set_cli_value_by_index(key, 10, "0");
        }
    }
}

int get_status(char *if_num)
{
    char key[32];
    char strStatus[8];

    sprintf(key, "NIF_STATUS%s", if_num);
    get_cli_value_by_index(key, 0, strStatus);
    if(!strcmp(strStatus, "1"))
        return 1;

    return 0;
}


int main(int argc, char *argv[])
{
    int    i, s, ret;
    char   if_name[8], cli_key[32];
    char   strStatus[8], strKey[32];
    time_t the_time;
    int    force_update=0;
    int    reconnect_count;
    const char *ptr = NULL;

    (void) time(&the_time);
    start_timer = the_time;

    sig.sa_flags = 0;
    sigemptyset(&sig.sa_mask);
    sig.sa_handler = SIG_IGN;
    sigaction(SIGPIPE,&sig,NULL);
    //sigaction(SIGUSR1,&sig,NULL);
    //sigaction(SIGUSR2,&sig,NULL);
    sig.sa_handler = (void *)sigusr1_handler;
    sigaction(SIGUSR1,&sig,NULL);
    sig.sa_handler = (void *)sigusr2_handler;
    sigaction(SIGUSR2,&sig,NULL);

    get_arguments_from_cli(1);

    //presto -- 071304 -- >>       
    while (1) 
    {
        (void)time(&the_time);
        if(the_time - EXPIRE_SEC > start_timer)
        {
            force_update = 1;
            start_timer = the_time;
        }

        for(i = 0;i < MAX_ARGS;++i)
        {
            //if(DDNS_Args[i].enable && strcmp(DDNS_Args[i].interface, "") != 0)
            if ( DDNS_Args[i].enable )
            {
                if(!get_status(DDNS_Args[i].interface))
                    continue;

                getip(DDNS_Args[i].IPaddress, DDNS_Args[i].interface);
                PDEBUG("! Last_IP_Addr = %s, IP = %s", DDNS_Args[i].oldIPaddress, DDNS_Args[i].IPaddress);

                if( ( strcmp(DDNS_Args[i].IPaddress, "") && strcmp(DDNS_Args[i].IPaddress, DDNS_Args[i].oldIPaddress) ) ||
                        force_update)
                {
                    sprintf(cli_key, "NIF_INFO%s", DDNS_Args[i].interface);
                    get_cli_value_by_index(cli_key, 0, if_name);

                    reconnect_count = 1;
reconnect:                    
                    s = get_connection(STR_Service_Provider[DDNS_Args[i].service_provider], PORT, &ptr, if_name);
                    if(s < 0) {
                        LogErr("%s: %s", ptr, STR_Service_Provider[DDNS_Args[i].service_provider]);
                        if(s == -1) { //-1 means gethostbyname failed, try to reconnect later
                            if(reconnect_count < MAX_RECONNECT) {
                                ++reconnect_count;    
                                sleep(1);
                                goto reconnect;
                            }
                        }
                    }
                    else if(s >= 0) //not connect failed and socket failed
                    {
retry:
                        //presto -- 083004 -- <<                        
                        //ret = update_dyndns(s, &DDNS_Args[i]);
                        ret = DDNS_Args[i].update_function(s, &DDNS_Args[i]);
                        //presto -- 083004 -- >>                        
                        if(ret == RET_OK) {
                            ret = check_server_msg(s, DDNS_Args[i].hostname);
                            if(ret == RET_OK) {
                                strcpy(DDNS_Args[i].oldIPaddress, DDNS_Args[i].IPaddress);
                            } 
                            else if(ret == RET_WRONG_USAGE) {
                                LogErr("message not from ddns server, retry");
                                sleep(DEFAULT_DELAY);
                                goto retry;
                            }
                        }
                        (void)close(s);

                        //presto -- 122904 -- <<
                        if(ret == RET_OK)
                            strcpy(strStatus, "1");
                        else
                            strcpy(strStatus, "2");    

                        sprintf(strKey, "ddns_args_%d", i + 1);
                        set_cli_value_by_index(strKey, 10, strStatus);
                        //presto -- 122904 -- >>
                    }
                    strcpy(DDNS_Args[i].IPaddress, "");
                }
            }
        } //for

        force_update = 0;
        sleep(DEFAULT_DELAY); //check again after 3 seconds
    } //while 
}

#define BUFLEN         4096
#define BUFFREE(name)  BUFLEN - strlen(name)

int update_dyndns(const int s, struct arguments *args)
{
    char *b64user;
    char b64user_buf[512];
    char message[BUFLEN];
    char login[128];
    char buffer[1024];

    sprintf(login, "%s:%s", args->login, args->password);

    b64user = b64user_buf;
    (void)memset(b64user, 0, 512);

    base64encode(login, b64user);
    (void)snprintf(message, BUFLEN,
            "GET /nic/update?system=%s&hostname=%s&wildcard=%s"
            "&backmx=%s",
            args->system, args->hostname,
            args->wildcard, args->backmx);

    if(strcmp(args->IPaddress, "") != 0) 
    {
        (void)strncat(message, "&myip=", BUFFREE(message));
        (void)strncat(message, args->IPaddress, BUFFREE(message));
    }

    if(strcmp(args->mx, "") != 0) 
    {
        (void)strncat(message, "&mx=", BUFFREE(message));
        (void)strncat(message, args->mx, BUFFREE(message));
    }

    (void)snprintf(buffer, 1024,
            " HTTP/1.0\r\n"
            "Host: %s\r\n"
            "Authorization: Basic %s\r\n"
            "Content-Type: application/x-www-form-urlencoded\r\n"
            "Cache-Control: no-cache\r\n"
            "User-Agent: %s\r\n\r\n",
            STR_Service_Provider[args->service_provider], 
            b64user, MODEL_NAME);

    (void)strncat(message, buffer, BUFLEN - 1 - strlen(message));

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if (write(s, message, strlen(message)) == -1) 
    {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}

int update_no_ip(const int s, struct arguments *args)
{
    return update_dyndns(s, args);
}

int update_dtdns(const int s, struct arguments *args)
{
    char message[BUFLEN];

    (void)snprintf(message, BUFLEN,
            "GET /api/autodns.cfm?id=%s&pw=%s&ip=%s HTTP/1.0\r\n"
            "Host: %s\r\nAccept: text/html, */*\r\n"
            "User-Agent: %s\r\n\r\n",
            args->hostname, args->password, args->IPaddress,
            STR_Service_Provider[args->service_provider],
            MODEL_NAME);

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if(write(s, message, strlen(message)) == -1) {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}


int update_changeip(const int s, struct arguments *args)
{
    char message[BUFLEN];
    char buffer[1024];

    (void)snprintf(message, BUFLEN,
            "GET /nic/update?u=%s&p=%s&hostname=%s&ip=%s&cmd=update&set=1&offline=0",
            args->login, args->password, args->hostname, args->IPaddress);

    (void)snprintf(buffer, 1024,
            " HTTP/1.1\r\n"
            "Host: %s\r\n"
            "Authorization: Basic %s\r\n"
            "Connection: close\r\n"
            "Pragma: no-cache\r\n\r\n",
            STR_Service_Provider[args->service_provider], 
            args->login);

    (void)strncat(message, buffer, BUFLEN - 1 - strlen(message));

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if(write(s, message, strlen(message)) == -1) {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}


int update_huagai(const int s, struct arguments *args)
{
    char message[BUFLEN];
    char *ptrHostname, *ptrZonename;
    char tmpHostname[128];
    char buffer[1024];

    strcpy(tmpHostname, args->hostname);

    ptrHostname = tmpHostname;
    ptrZonename = strchr(tmpHostname, '.');
    *ptrZonename = '\0';
    ++ptrZonename;

    (void)snprintf(message, BUFLEN,
            "GET /services/dyndns/update.aspx?username=%s&password=%s&zonename=%s&hostname=%s",
            args->login, args->password, ptrZonename, ptrHostname);

    (void)snprintf(buffer, 1024,
            " HTTP/1.1\r\n"
            "Content-Type: text/html\r\n"
            "Host: %s\r\n"
            "Accept: text/html, */*\r\n"
            "User-Agent: Mozilla/3.0 (compatible; Indy Library)\r\n\r\n",
            STR_Service_Provider[args->service_provider]);

    (void)strncat(message, buffer, BUFLEN - 1 - strlen(message));

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if(write(s, message, strlen(message)) == -1) {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}


int update_3322(const int s, struct arguments *args)
{
    char *b64user;
    char b64user_buf[512];
    char message[BUFLEN];
    char login[128];
    char buffer[1024];

    sprintf(login, "%s:%s", args->login, args->password);
    b64user = b64user_buf;
    (void)memset(b64user, 0, 512);

    base64encode(login, b64user);
    (void)snprintf(message, BUFLEN,
            "GET /dyndns/update?system=%s&hostname=%s&wildcard=%s"
            "&backmx=%s",
            args->system, args->hostname,
            args->wildcard, args->backmx);

    if(strcmp(args->IPaddress, "") != 0) 
    {
        (void)strncat(message, "&myip=", BUFFREE(message));
        (void)strncat(message, args->IPaddress, BUFFREE(message));
    }

    if(strcmp(args->mx, "") != 0) {
        (void)strncat(message, "&mx=", BUFFREE(message));
        (void)strncat(message, args->mx, BUFFREE(message));
    }

    (void)snprintf(buffer, 1024,
            " HTTP/1.0\r\n"
            "Host: %s\r\n"
            "Authorization: Basic %s\r\n"
            "Content-Type: application/x-www-form-urlencoded\r\n"
            "Cache-Control: no-cache\r\n"
            "User-Agent: %s\r\n\r\n",
            STR_Service_Provider[args->service_provider], 
            b64user, MODEL_NAME);

    (void)strncat(message, buffer, BUFLEN - 1 - strlen(message));

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if(write(s, message, strlen(message)) == -1) {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}


int update_dynamic_nameserver(const int s, struct arguments *args)
{
    return update_dyndns(s, args);
}


int update_vigorddns(const int s, struct arguments *args)
{
    char message[BUFLEN];
    char pwd[32];
    unsigned char md[16];
    char buf[80];
    int  i;
    char buffer[1024];

    *pwd = '\0';    
    strcat(pwd, "CNVigor");    
    strcat(pwd, args->password);
    // TODO
    //EVP_Digest(pwd, (unsigned long)strlen(pwd), md, NULL, EVP_md5(), NULL);

    memset(buf, 0 , sizeof(char) * 80);
    for (i=0; i<16; i++) 
    {
        sprintf(&(buf[i*2]),"%02x",md[i]);
    }

    (void)snprintf(message, BUFLEN,
            "GET /ddns/submit.php?txtUser=%s&txtPwd=%s&txtIP=%s&txtDomain=%s&Submit=SUBMIT",
            args->login, buf,
            args->IPaddress, args->hostname);

    (void)snprintf(buffer, 1024,
            " HTTP/1.1\r\n"
            "Host: %s\r\n"
            "Cache-Control: no-cache\r\n"
            "User-Agent: %s\r\n\r\n",
            STR_Service_Provider[args->service_provider], 
            MODEL_NAME);

    (void)strncat(message, buffer, BUFLEN - 1 - strlen(message));

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if(write(s, message, strlen(message)) == -1) {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}


int update_stratoddns(const int s, struct arguments *args)
{
    char *b64user;
    char b64user_buf[512];
    char message[BUFLEN];
    char login[128];
    char buffer[1024];

    sprintf(login, "%s:%s", args->login, args->password);
    b64user = b64user_buf;
    (void)memset(b64user, 0, 512);

    base64encode(login, b64user);
    (void)snprintf(message, BUFLEN,
            "GET /nic/update?system=%s&hostname=%s&wildcard=%s"
            "&backmx=%s",
            args->system, args->hostname,
            args->wildcard, args->backmx);

    if (strcmp(args->IPaddress, "") != 0) 
    {
        (void)strncat(message, "&myip=", BUFFREE(message));
        (void)strncat(message, args->IPaddress, BUFFREE(message));
    }

    if (strcmp(args->mx, "") != 0) 
    {
        (void)strncat(message, "&mx=", BUFFREE(message));
        (void)strncat(message, args->mx, BUFFREE(message));
    }

    (void)snprintf(buffer, 1024,
            " HTTP/1.0\r\n"
            "Host: %s\r\n"
            "Authorization: Basic %s\r\n"
            "Content-Type: application/x-www-form-urlencoded\r\n"
            "Cache-Control: no-cache\r\n"
            "User-Agent: %s\r\n\r\n",
            STR_Service_Provider[args->service_provider], 
            b64user, MODEL_NAME);

    (void)strncat(message, buffer, BUFLEN - 1 - strlen(message));

    printf("\n\nMessage:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            message);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, message);

    if(write(s, message, strlen(message)) == -1) {
        LogErr("write() failed");
        return RET_WARNING;
    }

    return RET_OK;
}


int read_socket_data(int s, char *buffer, int size)
{
    int    nBytes, nTotalBytes = 0;
    int    z;
    fd_set RdSet;
    struct timeval timeouts;

    timeouts.tv_sec = DEFAULT_TIMEOUT;
    timeouts.tv_usec = 0;

    memset(buffer, 0, size);

    while ( 1 ) 
    {
        FD_ZERO(&RdSet);
        FD_SET(s, &RdSet);
        z = select( s + 1, &RdSet, NULL, NULL, &timeouts);    
        if ( z == -1 ) {
            printf("read socket data, select failed, errno:%d\n", errno);
            return -1;
        } 
        else if ( z != 0 ) 
        {
            if ( FD_ISSET(s, &RdSet) ) {
                nBytes = read(s, buffer + nTotalBytes, size - nTotalBytes - 1);
                if (nBytes == -1) {
                    printf("read socket data, read failed, errno:%d\n", errno);
                    return -2;
                } 
                else if (nBytes == 0) {
                    buffer[nTotalBytes] = '\0';
                    return nTotalBytes;
                }

                nTotalBytes += nBytes;

                if ( nTotalBytes + 1 >= size ) {
                    nTotalBytes = size - 1;
                    buffer[nTotalBytes] = '\0';
                    return nTotalBytes;
                }            

            }

        } 
        else { 
            if ( nTotalBytes <= 0 ) {      
                return -3;
            } 
            else {
                buffer[nTotalBytes] = '\0';
                return nTotalBytes;                
            }
        }
    }

    return nTotalBytes;
}


int check_server_msg(int s, const char *hostname)
{
    int n;
    char server_msg[BUFSIZE], *ptr;

    /* get server_msg */
    n = read_socket_data(s, server_msg, BUFSIZE);
    if(n < 0) 
    {
        LogErr("wait server msg, err:%d", n);
        return RET_WARNING;
    }

    printf("\n\nServer message:"
            "\n--------------------------------------\n"
            "%s--------------------------------------\n\n",
            server_msg);

    draytek_syslog(DRAYTEK_SYSLOG_WAN_LOG, server_msg);

    if (strstr(server_msg, "HTTP/1.1 200 OK") || strstr(server_msg, "HTTP/1.0 200 OK") ||
        strstr(server_msg, "HTTP/1.1 200 Successful") || strstr(server_msg, "HTTP/1.0 200 Successful")) 
    {
        (void)strtok(server_msg, "\n");
        while((ptr = strtok(NULL, "\n")) != NULL) 
        {
            for(n=0; return_codes[n].code != NULL; n++) 
            {
                if(strstr(ptr, return_codes[n].code)) 
                {
                    LogErr("%s: %s",
                            hostname, return_codes[n].message);
                    if(return_codes[n].error == 1) {
                        return RET_ERROR;
                    } else {
                        return RET_OK;
                    }
                }
            }

            /* not reply from ddns server, send again */
            if(strstr(ptr, "Redirect") || strstr(ptr, "redirected"))
                return RET_WRONG_USAGE;
        }
    } 
    else if(strstr(server_msg, "401 Authorization Required")) 
    {
        LogErr("%s: wrong username or password", hostname);
    } 
    else 
    {
        LogErr("%s: Internal Server Error", hostname);
    }

    return RET_ERROR;
}

