/*
 *
 *  Copyright (C) 2007 Mindspeed Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <arpa/inet.h>

#include "libqos.h"

static unsigned long swap32(unsigned long x);
static void ipv4_format(unsigned char *ip, unsigned long x);
static void ipv6_format(unsigned char *ip, unsigned long *x);
#define IP4_STR_ADDR_LEN (16)
#define IP6_STR_ADDR_LEN (48)

/********************************************************************************
*		COMMON FUNCTIONS						*
*********************************************************************************/




/**
 * qos_open - opens the qos device file
 * @dev_name: name of the qos device file to be opened
 * @non_blocking: open mode flag
 *
 */
int qos_open (const char *dev_name, int non_blocking)
{
	int fd;

	//printf("qos_open with device %s\n", dev_name);

	if (non_blocking)
		fd = open (dev_name, O_RDWR | O_NONBLOCK, 0);
	else
		fd = open (dev_name, O_RDWR, 0);

	if (fd < 0)
		perror (dev_name);

	return fd;
}

/**
 * qos_close - closes the admittance or scheduler device file
 * @fd: file descriptor obtained with qos_open()
 *
 */
void qos_close (int fd)
{
	close(fd);
}


/**
 * qos_reset_adm - reset all shapers, policers and tcam entries of admittance blocks
 * @fd: file descriptor obtained with qos_open()
 *
 */
int qos_reset_adm (int fd, unsigned char *flag)
{
	int rc;

	rc = ioctl (fd, ADM_IOC_RESET, flag);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_RESET failed\n");
	}

	return rc;
}

/**
 * qos_reset_sch - reset all shapers of scheduler blocks
 * @fd: file descriptor obtained with qos_open()
 *
 */
int qos_reset_sch (int fd, unsigned char *flag)
{
	int rc;

	rc = ioctl (fd, SCH_IOC_RESET, flag);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_RESET failed\n");
	}

	return rc;
}


/**
 * qos_dumps_regs - dumps HW registers
 * @fd: file descriptor obtained with qos_open()
 *
 */
int qos_dump_regs (int fd)
{
	int rc;

	rc = ioctl (fd, ADM_IOC_GET_REGS, NULL);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_REGS failed\n");
	}

	return rc;
}






/********************************************************************************
*		ADMITTIANCE FUNCTIONS						*
*********************************************************************************/


/**
 * qos_adm_get_mode - return mode of the specified adm block
 * @fd: file descriptor obtained with qos_open()
 * @mode: returned data containing requested mode
 */
int qos_adm_get_mode (int fd, unsigned char *mode)
{
	int rc;

	rc = ioctl (fd, ADM_IOC_GET_MODE, mode);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_MODE failed\n");
	}

	return rc;
}


/**
 * qos_adm_set_mode - set mode to the specified adm block
 * @fd: file descriptor obtained with qos_open()
 * @mode: new mode to be set to the adm block
 */
int qos_adm_set_mode (int fd, unsigned char mode)
{
	int rc;

	if(mode > DISABLED_MODE)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "qos_adm_set_mode mode out of range\n");
		return -1;
	}
	rc = ioctl (fd, ADM_IOC_SET_MODE, &mode);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_MODE failed\n");
	}

	return rc;
}


/**
 * qos_adm_get_stats - get statistics from admittance block
 * @fd: file descriptor obtained with qos_open()
 * @mode: pointer to adm counters structure 
 */
int qos_adm_get_stats(int fd, struct _ADM_COUNTER *stats)
{
	int rc;
	rc = ioctl (fd, ADM_IOC_GET_ADM_STATS, stats);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_ADM_STATS failed\n");
	}

	return rc;
}


/**
 * qos_adm_dump_stats - print adm counters to standard output
 * @adm: index of the adm block
 * @stats: pointer to adm counters structure obtained with qos_adm_get_stats()
 */
void qos_adm_dump_stats(int adm_index, struct _ADM_COUNTER *stats)
{
	if(stats != NULL)
	{
		QOSLIB_PRINTF(QOSLIB_STATS, \
		"Admittance block %d counters:\n \
		port_byte                %ld\n \
		port_packet              %ld\n \
		reserved_byte            %ld\n \
		reserved_packet          %ld\n \
		managed_byte             %ld\n \
		managed_packet           %ld\n \
		packet_dropped           %ld\n \
		packet_dropped_error     %ld\n \
		packet_dropped_denied    %ld\n \
		packet_dropped_policer0  %ld\n \
		packet_dropped_policer1  %ld\n \
		packet_dropped_shaper    %ld\n \
		packet_dropped_managed   %ld\n \
		packet_dropped_unmanaged %ld\n\n",
		adm_index,
		stats->port_byte,
		stats->port_packet,
		stats->reserved_byte,
		stats->reserved_packet,
		stats->managed_byte,
		stats->managed_packet,
		stats->packet_dropped,
		stats->packet_dropped_error,
		stats->packet_dropped_denied,
		stats->packet_dropped_policer0,
		stats->packet_dropped_policer1,
		stats->packet_dropped_shaper,
		stats->packet_dropped_managed,
		stats->packet_dropped_unmanaged
		);
	}
}


/**
 * qos_adm_set_discard_conf - configure admittance block  discard mode
 * @fd: file descriptor obtained with qos_open()
 * @conf: discard configuration to be applied 
 */
int qos_adm_set_discard_conf (int fd, struct _ADM_DISCARD_CONF *conf)
{
	int rc;

	if((conf->queue_drop_min > conf->queue_drop_max) ||
		(conf->zone_prob[ADM_DROPZONE0] > conf->zone_prob[ADM_DROPZONE1]) ||
		(conf->zone_prob[ADM_DROPZONE1] > conf->zone_prob[ADM_DROPZONE2]) ||
		(conf->zone_prob[ADM_DROPZONE2] > conf->zone_prob[ADM_DROPZONE3]))
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "qos_adm_set_discard_conf() parameters out of range\n");
		return -1;
	}
 
	rc = ioctl (fd, ADM_IOC_SET_DISCARD_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_DISCARD_CONF failed\n");
	}

	return rc;
}


/**
 * qos_adm_set_flowctrl_conf - configure admittance block flow control mode
 * @fd: file descriptor obtained with qos_open()
 * @conf: flow control configuration to be applied 
 */
int qos_adm_set_flowctrl_conf (int fd, struct _ADM_FLOWCTRL_CONF *conf)
{
	int rc;
 
	//check configuration
	if(conf->off_tresh > conf->on_tresh)
		return -1;

	rc = ioctl (fd, ADM_IOC_SET_FLOWCTRL_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_FLOWCTRL_CONF failed\n");
	}

	return rc;
}

/**
 * qos_adm_set_shaper_conf - configure admittance block shaper mode
 * @fd: file descriptor obtained with qos_open()
 * @conf: shaper configuration to be applied 
 */
int qos_adm_set_shaper_conf (int fd, struct _ADM_SHAPER_CONF *conf)
{
	int rc;
 
	//TODO:check configuration
	//conf->rate
	//conf->max_credit
	//conf->init_burst
	//conf->overhead

	rc = ioctl (fd, ADM_IOC_SET_SHAPER_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_SHAPER_CONF failed\n");
	}

	return rc;
}

/**
 * qos_adm_get_shaper_conf - get shaper in admittance
 * @fd: file descriptor obtained with qos_open()
 * @conf: shaper configuration
 */
int qos_adm_get_shaper_conf (int fd, struct _ADM_SHAPER_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_GET_SHAPER_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_SHAPER_CONF failed\n");
	}

	return rc;
}


/**
 * qos_adm_add_camentry_top - add new entry in the classifier at first position
 * @fd: file descriptor obtained with qos_open()
 * @conf: entry configuration to be applied 
 */
int qos_adm_add_camentry_top (int fd, struct _ADM_TCAM_CONF *conf)
{
	conf->add_type = TCAM_ADD_TOP;
	return qos_adm_add_camentry (fd, conf);
}

/**
 * qos_adm_add_camentry_bottom - add new entry in the classifier at last position
 * @fd: file descriptor obtained with qos_open()
 * @conf: entry configuration to be applied 
 */
int qos_adm_add_camentry_bottom (int fd, struct _ADM_TCAM_CONF *conf)
{
	conf->add_type = TCAM_ADD_BOTTOM; 
	return qos_adm_add_camentry (fd, conf);
	
}

/**
 * qos_adm_add_camentry_before - add new entry in the classifier at before given index
 * @fd: file descriptor obtained with qos_open()
 * @conf: entry configuration to be applied 
 */
int qos_adm_add_camentry_before (int fd, struct _ADM_TCAM_CONF *conf, unsigned int index)
{
	conf->add_type = TCAM_ADD_BEFORE; 
	conf->position = index;
	return qos_adm_add_camentry (fd, conf);
}

/**
 * qos_adm_add_camentry - add new entry in the classifier
 * @fd: file descriptor obtained with qos_open()
 * @conf: entry configuration to be applied 
 */
int qos_adm_add_camentry (int fd, struct _ADM_TCAM_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_ADD_CAMENTRY, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_ADD_CAMENTRY failed\n");
	}

	return rc;
}


/**
 * qos_adm_remove_camentry - remove entry from the classifier
 * @fd: file descriptor obtained with qos_open()
 * @conf: entry identifier obtained with qos_adm_tcam_add()
 */
int qos_adm_remove_camentry (int fd, unsigned char index)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_REMOVE_CAMENTRY, &index);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_REMOVE_CAMENTRY failed\n");
	}

	return rc;
}

/**
 * qos_adm_change_camentry - update entry in the classifier
 * @fd: file descriptor obtained with qos_open()
 * @conf: entry configuration to be applied 
 */
int qos_adm_change_camentry (int fd, struct _ADM_TCAM_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_CHANGE_CAMENTRY, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_CHANGE_CAMENTRY failed\n");
	}

	return rc;
}


/**
 * qos_adm_set_camentry_state - enable/disable specified entry in classifier
 * @fd: file descriptor obtained with qos_open()
 * @conf: classifier entry state 
 */
int qos_adm_set_camentry_state (int fd, struct _ADM_TCAM_STATE *state)
{
	int rc;

	rc = ioctl (fd, ADM_IOC_SET_CAMENTRY_STATE, state);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_CAMENTRY_STATE failed\n");
	}

	return rc;
}

/**
 * qos_adm_get_tcam_entry - read tcam entry from admittance block
 * @fd: file descriptor obtained with qos_open()
 * @entry: pointer to the requested tcam entry. The requested entry->index should set before calling this function
 */
int qos_adm_get_camentry (int fd, struct _ADM_TCAM_CONF *conf)
{
	int rc;

	rc = ioctl (fd, ADM_IOC_GET_CAMENTRY, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_CAMENTRY failed\n");
	}

	return rc;
}

/**
 * qos_adm_dump_camentry - print tcam entry to standard output
 * entry: pointer to tcam entry structure obtained with qos_adm_tcam_get_entry()
 */
void qos_adm_dump_camentry(struct _ADM_TCAM_ENTRY *entry)
{
	unsigned char saddr_str[IP4_STR_ADDR_LEN];
	unsigned char daddr_str[IP4_STR_ADDR_LEN];
	unsigned char saddr_mask_str[IP4_STR_ADDR_LEN];
	unsigned char daddr_mask_str[IP4_STR_ADDR_LEN];

	unsigned char saddr6_str[IP6_STR_ADDR_LEN];
	unsigned char daddr6_str[IP6_STR_ADDR_LEN];
	unsigned char saddr6_mask_str[IP6_STR_ADDR_LEN];
	unsigned char daddr6_mask_str[IP6_STR_ADDR_LEN];

	unsigned long ip6mask[4];

	if(entry != NULL)
	{
		if(entry->ipfamily == TCAM_IPV4_FAMILY)
		{
			ipv4_format(saddr_str, entry->saddr[0]);
			ipv4_format(daddr_str, entry->daddr[0]);
			ipv4_format(saddr_mask_str, entry->saddr_mask);
			ipv4_format(daddr_mask_str, entry->daddr_mask);
		}
		else
		{
			ipv6_format(saddr6_str, entry->saddr);
			ipv6_format(daddr6_str, entry->daddr);
			memset((unsigned char*)ip6mask, 0xFF, 16);
			ip6mask[0] = entry->saddr_mask;
			ipv6_format(saddr6_mask_str, ip6mask);
			memset((unsigned char*)ip6mask, 0xFF, 16);
			ip6mask[0] = entry->daddr_mask;
			ipv6_format(daddr6_mask_str, ip6mask);
		}

		QOSLIB_PRINTF(1, \
		" \
		policer         %d\n \
		drop            %d\n \
		reserved        %d\n \
		etype           %x\n \
		vlan            %d\n \
		pppoe_ipv4      %x\n \
		pppoe_ipv6      %x\n \
		iptos           %x\n \
		ipproto         %d\n \
		saddr           %s\n \
		daddr           %s\n \
		sport_max       %d\n \
		sport_min       %d\n \
		dport_max       %d\n \
		dport_min       %d\n \
		multicast       %x\n \
		etype_mask      %x\n \
		vlan_mask       %x\n \
		saddr_mask      %s\n \
		daddr_mask      %s\n \
		sport_enable    %d\n \
		dport_enable    %d\n \
		iptos_mask      %lx\n \
		pppoe_mask      %lx\n \
		ipproto_mask    %lx\n \
		broadcast_mask  %lx\n \
		multicast_mask  %lx\n\n",
		entry->policer,
		entry->drop,
		entry->reserved,
		entry->etype,
		entry->vlan,
		entry->pppoe_ipv4,
		entry->pppoe_ipv6,
		entry->iptos,
		entry->ipproto,
		(entry->ipfamily == TCAM_IPV6_FAMILY) ? (unsigned char*)saddr6_str : (unsigned char*)saddr_str,
		(entry->ipfamily == TCAM_IPV6_FAMILY) ? (unsigned char*)daddr6_str : (unsigned char*)daddr_str,
		entry->sport_max,
		entry->sport_min,
		entry->dport_max,
		entry->dport_min,
		entry->multicast,
		entry->etype_mask,
		entry->vlan_mask,
		(entry->ipfamily == TCAM_IPV6_FAMILY) ? (unsigned char*)saddr6_mask_str : (unsigned char*)saddr_mask_str ,
		(entry->ipfamily == TCAM_IPV6_FAMILY) ? (unsigned char*)daddr6_mask_str : (unsigned char*)daddr_mask_str ,
		entry->sport_enable,
		entry->dport_enable,
		entry->iptos_mask,
		entry->pppoe_mask,
		entry->ipproto_mask,
		entry->broadcast_mask,
		entry->multicast_mask
		);
	}
}



/**
 * qos_adm_get_policer_conf - query policer entry
 * @fd: file descriptor obtained with qos_open()
 * @conf: policer configuration  
 */
int qos_adm_get_policer_conf (int fd, struct _ADM_POLICER_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_GET_POLICER_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_POLICER_CONF failed\n");
	}

	return rc;
}

/**
 * qos_adm_set_policer_conf - configures policer
 * @fd: file descriptor obtained with qos_open()
 * @conf: policer configuration to be applied 
 */
int qos_adm_set_policer_conf (int fd, struct _ADM_POLICER_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_SET_POLICER_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_POLICER_CONF failed\n");
	}

	return rc;
}



/**
 * qos_adm_get_lru_stats - get LRU statistics from admittance block
 * @fd: file descriptor obtained with qos_open()
 * @stats: pointer to lru stats structure 
 */
int qos_adm_get_lru_stats(int fd, struct _ADM_LRU_STATS *stats)
{
	int rc;
	rc = ioctl (fd, ADM_IOC_GET_LRU_STATS, stats);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_GET_LRU_STATS failed\n");
	}

	return rc;
}

/**
 * qos_adm_set_lru_conf - configures LRU 
 * @fd: file descriptor obtained with qos_open()
 * @conf: LRU configuration to be applied 
 */
int qos_adm_set_lru_conf (int fd, struct _ADM_LRU_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, ADM_IOC_SET_LRU_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "ADM_IOC_SET_LRU_CONF failed\n");
	}

	return rc;
}






/********************************************************************************
*		SCHEDULER FUNCTIONS						*
*********************************************************************************/

/**
 * qos_sch_get_stats - get statistics from scheduler block
 * @fd: file descriptor obtained with qos_open()
 * @mode: pointer to sch counters structure
 */
int qos_sch_get_stats(int fd, struct _SCH_COUNTER *stats)
{
	int rc;
	rc = ioctl (fd, SCH_IOC_GET_STATS, stats);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_GET_STATS failed\n");
	}

	return rc;
}


/**
 * qos_sch_dump_stats - print schcounters to standard output
 * @adm: index of the sch block
 * @stats: pointer to sch counters structure obtained with qos_adm_get_stats()
 */
void qos_sch_dump_stats(int sch_index, struct _SCH_COUNTER *stats)
{
	int q;

	if(stats != NULL)
	{	
		QOSLIB_PRINTF(QOSLIB_STATS, \
		"Scheduler block %d counters:\n \
		tx byte                %ld\n \
		tx packet              %ld\n \
		", \
		sch_index,
		stats->port_byte, \
		stats->port_packet \
		);
	
		for(q = 0; q < SCH_MAX_QUEUES; q++) 
		{	
			QOSLIB_PRINTF(QOSLIB_STATS, \
			"	queue %d:\n \
			byte in queue		%ld\n \
			packet in queue		%ld\n \
			queue idle time		%ld\n \
			", \
			q,
			stats->q_byte[q], \
			stats->q_packet[q], \
			stats->q_idle[q] \
			);
		}
	}
}

/**
 * qos_sch_set_shaper_state - enable/disable specified shaper in scheduler
 * @fd: file descriptor obtained with qos_open()
 * @state: shaper state 
 */
int qos_sch_set_shaper_state (int fd, struct _SCH_SHAPER_STATE *state)
{
	int rc;

	rc = ioctl (fd, SCH_IOC_SET_SHAPER_STATE, state);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_SET_SHAPER_STATE failed\n");
	}

	return rc;
}

/**
 * qos_sch_get_shaper_state - enable/disable specified shaper in scheduler
 * @fd: file descriptor obtained with qos_open()
 * @state: shaper state 
 */
int qos_sch_get_shaper_state (int fd, struct _SCH_SHAPER_STATE *state)
{
	int rc;

	rc = ioctl (fd, SCH_IOC_GET_SHAPER_STATE, state);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_SET_SHAPER_STATE failed\n");
	}

	return rc;
}

/**
 * qos_sch_set_shaper_conf - configures shaper in scheduler
 * @fd: file descriptor obtained with qos_open()
 * @conf: shaper configuration to be applied 
 */
int qos_sch_set_shaper_conf (int fd, struct _SCH_SHAPER_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, SCH_IOC_SET_SHAPER_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_SET_SHAPER_CONF failed\n");
	}

	return rc;
}

/**
 * qos_sch_get_shaper_conf - get shaper in scheduler
 * @fd: file descriptor obtained with qos_open()
 * @conf: shaper configuration
 */
int qos_sch_get_shaper_conf (int fd, struct _SCH_SHAPER_CONF *conf)
{	
	int rc;

	rc = ioctl (fd, SCH_IOC_GET_SHAPER_CONF, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_GET_SHAPER_CONF failed\n");
	}

	return rc;
}


/**
 * qos_sch_set_queue_weight - configures DWRR group queue weigth in scheduler
 * @fd: file descriptor obtained with qos_open()
 * @conf: weight configuration to be applied 
 */
int qos_sch_set_queue_weight (int fd, struct _SCH_QUEUE_WEIGHT *conf)
{	
	int rc;

	rc = ioctl (fd, SCH_IOC_SET_QUEUE_WEIGHT, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_SET_QUEUE_WEIGHT failed\n");
	}

	return rc;
}

/**
 * qos_sch_get_queue_weight - get DWRR group queue weigth in scheduler
 * @fd: file descriptor obtained with qos_open()
 * @conf: weight configuration
 */
int qos_sch_get_queue_weight (int fd, struct _SCH_QUEUE_WEIGHT *conf)
{	
	int rc;

	rc = ioctl (fd, SCH_IOC_GET_QUEUE_WEIGHT, conf);
	if (rc < 0)
	{
		QOSLIB_PRINTF (QOSLIB_ERR, "SCH_IOC_GET_QUEUE_WEIGHT failed\n");
	}

	return rc;
}


/**
 * ipv4_format -
 *
 *
 */
static void ipv4_format(unsigned char *ip, unsigned long x)
{
	unsigned long _x = swap32(x);	
	
	inet_ntop(AF_INET, &_x, ip, IP4_STR_ADDR_LEN);
	
	//printf("ipv4_format: from %08lx (%08lx) to %s\n", x, _x, ip);	
}


/**
 * ipv6_format -
 *
 *
 */
static void ipv6_format(unsigned char *ip, unsigned long *x)
{
	unsigned long _x[4] = {0}; 
	_x[0] = swap32(x[3]);
	_x[1] = swap32(x[2]);
	_x[2] = swap32(x[1]);
	_x[3] = swap32(x[0]);	
	
	inet_ntop(AF_INET6, _x, ip, IP6_STR_ADDR_LEN);
	
	//printf("ipv6_format: from %08lx (%08lx) to %s\n", x[0], _x[3], ip);	
}
/**
 * swap32 -
 *
 *
 */
static unsigned long swap32(unsigned long x)
{
	unsigned long _x = x;
	_x = ((_x & 0x000000FF) << 24) |
		((_x & 0x0000FF00) << 8) |
		((_x & 0x00FF0000) >> 8) |
		((_x & 0xFF000000) >> 24);	
	return _x;
}
