/*
 *
 *  Copyright (C) 2007 Mindspeed Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/version.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <net/sock.h>
#include <linux/timer.h>
#include <linux/time.h>

#include "qoscom.h"


MODULE_LICENSE("GPL");
MODULE_AUTHOR("Mindspeed Technologies");
MODULE_DESCRIPTION("Comcerto 1000 QoS Driver");

#define QOSCOM_VERSION	"$Name: qoscom_1_02_16 $"

static unsigned long swapbit(unsigned long x);
static unsigned char reduce_ip_mask(unsigned long ip_mask);
static unsigned long expand_ip_mask(unsigned long ip_mask);


/* ioctls functions  */
static int qos_open (struct inode *inode, struct file *file);
static int qos_release (struct inode *inode, struct file *file);
static int qos_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);

/* commons functions */
static int qos_init(void);
static void qos_dump_registers(unsigned char block);

/* admittance functions */
static int qos_adm_get_counters(struct _QOS_ADM *adm, struct _ADM_COUNTER *stats);
static void qos_adm_dump_counters(void);
static int qos_adm_set_mode(struct _QOS_ADM *adm, unsigned char mode);
static unsigned char qos_adm_get_mode(struct _QOS_ADM *adm);
static int qos_adm_set_discard_conf(struct _QOS_ADM *adm, struct _ADM_DISCARD_CONF *conf);
static int qos_adm_set_flowctrl_conf(struct _QOS_ADM *adm, struct _ADM_FLOWCTRL_CONF *conf);
static int qos_adm_set_shaper_conf(struct _QOS_ADM *adm, struct _ADM_SHAPER_CONF *conf);
static int qos_adm_get_shaper_conf(struct _QOS_ADM *adm, struct _ADM_SHAPER_CONF *conf);
static int qos_adm_add_camentry(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf);
static int qos_adm_remove_camentry(struct _QOS_ADM *this_adm, unsigned char tcam_entry_id);
static int qos_adm_change_camentry(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf);
static int qos_adm_set_camentry_state(struct _QOS_ADM *this_adm, struct _ADM_TCAM_STATE *conf);
static int qos_adm_get_camentry(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf);
static int qos_adm_add_camentry_to_slot(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf, unsigned char tcam_slot);
static int qos_adm_set_policer_conf(struct _QOS_ADM *this_adm, struct _ADM_POLICER_CONF *conf);
static int qos_adm_get_policer_conf(struct _QOS_ADM *this_adm, struct _ADM_POLICER_CONF *conf);
static int qos_adm_set_lru_conf(struct _QOS_ADM *this_adm, struct _ADM_LRU_CONF *conf);
static int qos_adm_get_lru_stats(struct _QOS_ADM *this_adm, struct _ADM_LRU_STATS *p);

/* scheduler functions */
static int qos_sch_get_counters(struct _QOS_SCH *this_sch, struct _SCH_COUNTER *p);
static void qos_sch_dump_counters(void);
static int qos_sch_set_shaper_state(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_STATE *conf);
static int qos_sch_get_shaper_state(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_STATE *conf);
static int qos_sch_set_shaper_conf(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_CONF *conf);
static int qos_sch_get_shaper_conf(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_CONF *conf);
static int qos_sch_set_queue_weight(struct _QOS_SCH *this_sch, struct _SCH_QUEUE_WEIGHT *conf);
static int qos_sch_get_queue_weight(struct _QOS_SCH *this_sch, struct _SCH_QUEUE_WEIGHT *conf);


static void get_shaper_rate_and_clock(unsigned long bps, unsigned long *p_clk_sel, unsigned long *p_fraction);
static void get_policer_rate_and_clock(unsigned long bps, unsigned long *p_clk_sel, unsigned long *p_fraction);
static unsigned long shaper_fraction_clk_to_rate(unsigned long fraction, unsigned long clk_sel);

static int qos_sch_global_reset(struct _QOS_SCH *this_sch);
static int qos_adm_global_reset(struct _QOS_ADM *this_adm);

/* admittance blocks main context */
static struct _QOS_ADM _adm[2];

/* scheduler blocks main context */
static struct _QOS_SCH _sch[2];

static char * adm_mode_string[] = {"DISCARD", "FLOW CONTROL", "DISABLED"};


#ifdef C1K_REG_SIM
unsigned long *FAKE_EMAC0_BASE;
unsigned long *FAKE_EMAC1_BASE;
unsigned long get_gemac_base(int emac)
{
	if(emac)
		return (unsigned long)FAKE_EMAC1_BASE;
	else
		return (unsigned long)FAKE_EMAC0_BASE;
}
#endif

/***************************  IOCTLs ******************************************/

/**
 * qos_ioctl -
 *
 *
 */
static int qos_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
        struct _QOS_ADM *this_adm = NULL;
	struct _ADM_COUNTER adm_stats;
	struct _ADM_LRU_STATS adm_lru_stats; struct _ADM_LRU_CONF adm_lru_conf;
	struct _ADM_DISCARD_CONF adm_discard_conf;
	struct _ADM_FLOWCTRL_CONF adm_flowctrl_conf;
	struct _ADM_SHAPER_CONF	adm_shaper_conf;
	struct _ADM_TCAM_CONF adm_tcam_conf; struct _ADM_TCAM_STATE adm_tcam_state;
	struct _ADM_POLICER_CONF adm_policer_conf;

	struct _QOS_SCH *this_sch = NULL;
	struct _SCH_COUNTER sch_stats;
	struct _SCH_SHAPER_CONF	sch_shaper_conf;
	struct _SCH_SHAPER_STATE sch_shaper_state;
	struct _SCH_QUEUE_WEIGHT sch_queue_weight;

	unsigned char mode, tcam_entry_id;
	int minor = MINOR (inode->i_rdev);
	int rc = 0;
 	
	if ((_IOC_TYPE (cmd) != ADM_IOC_TYPE) && (_IOC_TYPE (cmd) != SCH_IOC_TYPE))
        {
		QOS_PRINTK(QOS_ERR, "qos_ioctl() wrong IOC_TYPE");
                rc = -EINVAL;
                goto out;
        }

	if(file->private_data == NULL)
	{
		QOS_PRINTK(QOS_ERR, "qos_ioctl() private data is NULL\n");
		rc = -EINVAL;
		goto out;
	}

	if ((ADM0_DEVICE_MINOR_NUM == minor) || (ADM1_DEVICE_MINOR_NUM == minor))
	{
		this_adm = (struct _QOS_ADM *) file->private_data;
	}
	else if ((SCH0_DEVICE_MINOR_NUM == minor) || (SCH1_DEVICE_MINOR_NUM == minor))
	{
		this_sch = (struct _QOS_SCH *) file->private_data;
	}
	
	if (_IOC_TYPE (cmd) == ADM_IOC_TYPE)
	{
		switch (cmd)
		{
			case ADM_IOC_RESET:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_RESET\n");
				if(qos_adm_global_reset(this_adm))
				{
					QOS_PRINTK(QOS_ERR, "error resetting admittance block");
					rc = -EFAULT;
					goto out;
				}
				break;

			case ADM_IOC_SET_MODE:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_SET_MODE\n");
				if (copy_from_user(&mode, (unsigned char *)arg, sizeof(unsigned char)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				if(qos_adm_set_mode(this_adm, mode))
				{
					QOS_PRINTK(QOS_ERR, "error setting admittance mode");
					rc = -EFAULT;
					goto out;
				}
				break;

			case ADM_IOC_GET_MODE:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_MODE\n");
				mode = qos_adm_get_mode(this_adm);
				if (copy_to_user((unsigned char *)arg, &mode, sizeof(unsigned char)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case ADM_IOC_SET_DISCARD_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_DISCARD_CONF\n");
				if (copy_from_user(&adm_discard_conf, (struct _ADM_DISCARD_CONF *)arg, sizeof(struct _ADM_DISCARD_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
			
				if(qos_adm_set_discard_conf(this_adm, &adm_discard_conf))
				{
					rc = -EINVAL;
					goto out;
				}
				break;
	
			case ADM_IOC_SET_FLOWCTRL_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_FLOWCTRL_CONF\n");
				if (copy_from_user(&adm_flowctrl_conf, (struct _ADM_FLOWCTRL_CONF *)arg, sizeof(struct _ADM_FLOWCTRL_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				
				if(qos_adm_set_flowctrl_conf(this_adm, &adm_flowctrl_conf))
				{
					rc = -EINVAL;
					goto out;
				}
				break;
	
			case ADM_IOC_SET_SHAPER_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_SHAPER_CONF\n");
				if (copy_from_user(&adm_shaper_conf, (struct _ADM_SHAPER_CONF *)arg, sizeof(struct _ADM_SHAPER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_adm_set_shaper_conf(this_adm, &adm_shaper_conf))
				{
					rc = -EINVAL;
					goto out;
				}
				break;

			case ADM_IOC_GET_SHAPER_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_SET_SHAPER_CONF\n");
				if (copy_from_user(&adm_shaper_conf, (struct _ADM_SHAPER_CONF *)arg, sizeof(struct _ADM_SHAPER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				qos_adm_get_shaper_conf(this_adm, &adm_shaper_conf);
				if (copy_to_user((struct _ADM_SHAPER_CONF *)arg, &adm_shaper_conf, sizeof(struct _ADM_SHAPER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case ADM_IOC_ADD_CAMENTRY:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_ADD_CAMENTRY\n");
				if (copy_from_user(&adm_tcam_conf, (struct _ADM_TCAM_CONF *)arg, sizeof(struct _ADM_TCAM_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				switch(adm_tcam_conf.add_type)
				{
					case TCAM_ADD_TOP:
						break;

					case TCAM_ADD_BEFORE: 
						break;
		
					case TCAM_ADD_BOTTOM:
					default:
						if(qos_adm_add_camentry(this_adm, &adm_tcam_conf))
						{
							rc = -EINVAL;
							goto out;
						}
						break;
				}

				if (copy_to_user((struct _ADM_TCAM_CONF *)arg, &adm_tcam_conf, sizeof(struct _ADM_TCAM_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}		
				break;

			case ADM_IOC_CHANGE_CAMENTRY:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_CHANGE_CAMENTRY\n");
				if (copy_from_user(&adm_tcam_conf, (struct _ADM_TCAM_CONF *)arg, sizeof(struct _ADM_TCAM_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_adm_change_camentry(this_adm, &adm_tcam_conf))
				{
					rc = -EINVAL;
					goto out;
				}			
				break;
	
			case ADM_IOC_SET_CAMENTRY_STATE:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_SET_CAMENTRY_STATE\n");
				if (copy_from_user(&adm_tcam_state, (struct _ADM_TCAM_STATE *)arg, sizeof(struct _ADM_TCAM_STATE)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_adm_set_camentry_state(this_adm, &adm_tcam_state))
				{
					rc = -EINVAL;
					goto out;
				}
				break;

			case ADM_IOC_REMOVE_CAMENTRY:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_REMOVE_CAMENTRY\n");
				if (copy_from_user(&tcam_entry_id, (unsigned char *)arg, sizeof(unsigned char)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_adm_remove_camentry(this_adm, tcam_entry_id))
				{
					rc = -EINVAL;
					goto out;
				}
				break;

			case ADM_IOC_GET_CAMENTRY:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_CAMENTRY\n");
				if (copy_from_user(&adm_tcam_conf, (struct _ADM_TCAM_CONF *)arg, sizeof(struct _ADM_TCAM_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_CAMENTRY for entry %d on adm block %d\n", adm_tcam_conf.index,this_adm->index);
				qos_adm_get_camentry(this_adm, &adm_tcam_conf);

				if (copy_to_user((struct _ADM_TCAM_CONF *)arg, &adm_tcam_conf, sizeof(struct _ADM_TCAM_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;
	
			case ADM_IOC_SET_POLICER_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_SET_POLICER_CONF\n");
				if (copy_from_user(&adm_policer_conf, (struct _ADM_POLICER_CONF *)arg, sizeof(struct _ADM_POLICER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_adm_set_policer_conf(this_adm, &adm_policer_conf))
				{
					rc = -EINVAL;
					goto out;
				}	
				break;

			case ADM_IOC_GET_POLICER_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_POLICER_CONF\n");
				if (copy_from_user(&adm_policer_conf, (struct _ADM_POLICER_CONF *)arg, sizeof(struct _ADM_POLICER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_POLICER_CONF for policer %d on adm block %d\n", adm_policer_conf.index,this_adm->index);
				qos_adm_get_policer_conf(this_adm, &adm_policer_conf);

				if (copy_to_user((struct _ADM_POLICER_CONF *)arg, &adm_policer_conf, sizeof(struct _ADM_POLICER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;
	
			case ADM_IOC_SET_LRU_CONF:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_SET_LRU_CONF\n");
				if (copy_from_user(&adm_lru_conf, (struct _ADM_LRU_CONF *)arg, sizeof(struct _ADM_LRU_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_adm_set_lru_conf(this_adm, &adm_lru_conf))
				{
					rc = -EINVAL;
					goto out;
				}	
				break;
	
			case ADM_IOC_GET_LRU_STATS:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_LRU_STATS\n");
				qos_adm_get_lru_stats(this_adm, &adm_lru_stats);
				if (copy_to_user((struct _ADM_LRU_STATS *)arg, &adm_lru_stats, sizeof(struct _ADM_LRU_STATS)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case ADM_IOC_GET_ADM_STATS:
				QOS_PRINTK (QOS_IOCTL, "ADM_IOC_GET_ADM_STATS\n");
				qos_adm_get_counters(this_adm, &adm_stats);
				if (copy_to_user((struct _ADM_COUNTER *)arg, &adm_stats, sizeof(struct _ADM_COUNTER)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case ADM_IOC_GET_REGS:	
				QOS_PRINTK (QOS_IOCTL, "COMMON_IOC_GET_REGS\n");
				qos_dump_registers(QOS_ADM_BLOCK | QOS_SCH_BLOCK);
				break;

			default:
				QOS_PRINTK (QOS_ERR, "Unknown IOCTL\n");
				rc = -EINVAL;
				break;
		}
	}
	else
	{
		switch (cmd)
		{
			case SCH_IOC_RESET:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_RESET\n");
				if(qos_sch_global_reset(this_sch))
				{
					QOS_PRINTK(QOS_ERR, "error resetting scheduler block");
					rc = -EFAULT;
					goto out;
				}
				break;

			case SCH_IOC_GET_STATS:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_GET_STATS\n");
				qos_sch_get_counters(this_sch, &sch_stats);
				if (copy_to_user((struct _SCH_COUNTER *)arg, &sch_stats, sizeof(struct _SCH_COUNTER)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case SCH_IOC_SET_SHAPER_STATE:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_SET_SHAPER_STATE\n");
				if (copy_from_user(&sch_shaper_state, (struct _SCH_SHAPER_STATE *)arg, sizeof(struct _SCH_SHAPER_STATE)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_sch_set_shaper_state(this_sch, &sch_shaper_state))
				{
					rc = -EINVAL;
					goto out;
				}
				break;

			case SCH_IOC_GET_SHAPER_STATE:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_GET_SHAPER_STATE\n");
				if (copy_from_user(&sch_shaper_state, (struct _SCH_SHAPER_STATE *)arg, sizeof(struct _SCH_SHAPER_STATE)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				qos_sch_get_shaper_state(this_sch, &sch_shaper_state);
				if (copy_to_user((struct _SCH_SHAPER_STATE *)arg, &sch_shaper_state, sizeof(struct _SCH_SHAPER_STATE)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case SCH_IOC_SET_SHAPER_CONF:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_SET_SHAPER_CONF\n");
				if (copy_from_user(&sch_shaper_conf, (struct _SCH_SHAPER_CONF *)arg, sizeof(struct _SCH_SHAPER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_sch_set_shaper_conf(this_sch, &sch_shaper_conf))
				{
					rc = -EINVAL;
					goto out;
				}
				break;

			case SCH_IOC_GET_SHAPER_CONF:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_GET_SHAPER_CONF\n");
				if (copy_from_user(&sch_shaper_conf, (struct _SCH_SHAPER_CONF *)arg, sizeof(struct _SCH_SHAPER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				qos_sch_get_shaper_conf(this_sch, &sch_shaper_conf);
				if (copy_to_user((struct _SCH_SHAPER_CONF *)arg, &sch_shaper_conf, sizeof(struct _SCH_SHAPER_CONF)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			case SCH_IOC_SET_QUEUE_WEIGHT:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_SET_QUEUE_WEIGHT\n");
				if (copy_from_user(&sch_queue_weight, (struct _SCH_QUEUE_WEIGHT *)arg, sizeof(struct _SCH_QUEUE_WEIGHT)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}

				if(qos_sch_set_queue_weight(this_sch, &sch_queue_weight))
				{
					rc = -EINVAL;
					goto out;
				}
				break;

			case SCH_IOC_GET_QUEUE_WEIGHT:
				QOS_PRINTK (QOS_IOCTL, "SCH_IOC_GET_QUEUE_WEIGHT\n");
				if (copy_from_user(&sch_queue_weight, (struct _SCH_QUEUE_WEIGHT *)arg, sizeof(struct _SCH_QUEUE_WEIGHT)))
				{
					QOS_PRINTK(QOS_ERR, "error copying from user");
					rc = -EFAULT;
					goto out;
				}
				qos_sch_get_queue_weight(this_sch, &sch_queue_weight);
				if (copy_to_user((struct _SCH_QUEUE_WEIGHT *)arg, &sch_queue_weight, sizeof(struct _SCH_QUEUE_WEIGHT)))
				{
					QOS_PRINTK(QOS_ERR, "error copying to user");
					rc = -EFAULT;
					goto out;
				}
				break;

			default:
				QOS_PRINTK (QOS_ERR, "Unknown IOCTL\n");
				rc = -EINVAL;
				break;
		}
	}
out:
	return rc;

}


/**
 * qos_adm_global_reset -
 *
 *
 */
static int qos_adm_global_reset(struct _QOS_ADM *this_adm)
{
	unsigned char tcam_entry_id;

	QOS_PRINTK (QOS_IOCTL, "qos_adm_global_reset");

	/* policer 0 */
	ADM_SET_POLICER_FRAC_RATE(this_adm->index, 0, 0);
	ADM_SET_POLICER_MAX_CREDIT(this_adm->index, 0, 0);
	ADM_SET_POLICER_CLOCK(this_adm->index, 0, 0);
	ADM_SET_POLICER_ENABLE(this_adm->index, 0, 0);

	/* policer 1 */
	ADM_SET_POLICER_FRAC_RATE(this_adm->index, 1, 0);
	ADM_SET_POLICER_MAX_CREDIT(this_adm->index, 1, 0);
	ADM_SET_POLICER_CLOCK(this_adm->index, 1, 0);
	ADM_SET_POLICER_ENABLE(this_adm->index, 1, 0);

	/* shaper */
	ADM_SET_PORT_SHA_PKT_OVER(this_adm->index, 0);//sw wa - registers 0x4030 must be written prior to 0x4020
	ADM_SET_PORT_SHA_FRAC_RATE(this_adm->index, 0);
	ADM_SET_PORT_SHA_MAX_CREDIT(this_adm->index, 0);
	ADM_SET_PORT_SHA_CLOCK(this_adm->index, 0);
	ADM_SET_PORT_SHA_ENABLE(this_adm->index, 0);

	/* TCAM */
	for(tcam_entry_id = 0; tcam_entry_id <ADM_MAX_TCAM_ENTRIES ;tcam_entry_id++)
	{
		if(this_adm->cam_sw_table[tcam_entry_id].slot != ADM_TCAM_FREE)
		{
			memset(&this_adm->cam_sw_table[tcam_entry_id].entry, sizeof(struct _ADM_TCAM_ENTRY), 0);
			qos_adm_remove_camentry(this_adm, tcam_entry_id);
		}
	}
	
	/* LRU */
	ADM_SET_LRU_MASTER_DISABLE(this_adm->index);
	ADM_SET_LRU_STATE(this_adm->index, 0);
	ADM_SET_LRU_MASK(this_adm->index, 0xFFFFFFFF);

	return 0;
}

/**
 * qos_sch_global_reset -
 *
 *
 */
static int qos_sch_global_reset(struct _QOS_SCH *this_sch)
{
	QOS_PRINTK (QOS_IOCTL, "qos_sch_global_reset");
	
	SCH_SET_Q_CONTROL_CLOCK(this_sch->index, 7, 0);
	SCH_SET_Q_FRAC_RATE(this_sch->index, 7, 0);
	SCH_SET_Q_MAX_CREDIT(this_sch->index, 7, 0);
	SCH_SET_Q_CONTROL_ENABLE(this_sch->index, 7, 0);

	SCH_SET_Q_CONTROL_CLOCK(this_sch->index, 6, 0);
	SCH_SET_Q_FRAC_RATE(this_sch->index, 6, 0);
	SCH_SET_Q_MAX_CREDIT(this_sch->index, 6, 0);
	SCH_SET_Q_CONTROL_ENABLE(this_sch->index, 6, 0);

	SCH_SET_GROUP_CONTROL_CLOCK(this_sch->index, 0);
	SCH_SET_GROUP_FRAC_RATE(this_sch->index, 0);
	SCH_SET_GROUP_MAX_CREDIT(this_sch->index, 0);
	SCH_SET_GROUP_CONTROL_ENABLE(this_sch->index, 0);

	SCH_SET_PORT_CONTROL_CLOCK(this_sch->index, 0);
	SCH_SET_PORT_FRAC_RATE(this_sch->index, 0);
	SCH_SET_PORT_MAX_CREDIT(this_sch->index, 0);
	SCH_SET_PORT_CONTROL_ENABLE(this_sch->index, 0);

	return 0;
}

/**
 * qos_open -
 *
 *
 */
static int qos_open (struct inode *inode, struct file *file)
{
	int minor = MINOR (inode->i_rdev);

	QOS_PRINTK (QOS_IOCTL, "qos_open(%#lx, %#lx) minor = %d", (unsigned long)inode, (unsigned long)file, minor);

	if (ADM0_DEVICE_MINOR_NUM == minor)
	{
		QOS_PRINTK (QOS_IOCTL, "ADM0_DEVICE_MINOR_NUM opened");
		if(&_adm[ADM0] != NULL)
			file->private_data = (void *) &_adm[ADM0];
		else
			QOS_PRINTK (QOS_ERR, "qos_open ADM0 failed\n");
	}
	else if (ADM1_DEVICE_MINOR_NUM == minor)
	{
		QOS_PRINTK (QOS_IOCTL, "ADM1_DEVICE_MINOR_NUM opened");
		file->private_data = (void *) &_adm[ADM1];
	}
	else if (SCH0_DEVICE_MINOR_NUM == minor)
	{
		QOS_PRINTK (QOS_IOCTL, "SCH0_DEVICE_MINOR_NUM opened");
		file->private_data = (void *) &_sch[SCH0];
	}
	else if (SCH1_DEVICE_MINOR_NUM == minor)
	{
		QOS_PRINTK (QOS_IOCTL, "SCH1_DEVICE_MINOR_NUM opened");
		file->private_data = (void *) &_sch[SCH1];
	}
	else
	{
		QOS_PRINTK (QOS_ERR, "QoS device %d out of range", minor);
		return -ENODEV;
	}

	return 0;
}


/**
 * qos_release -
 *
 *
 */
static int qos_release (struct inode *inode, struct file *file)
{
	QOS_PRINTK (QOS_IOCTL, "qos_release(%#lx, %#lx)", (unsigned long)inode, (unsigned long)file);

	file->private_data = NULL;

	return 0;
}


/********************************************************************************
*		ADMITTANCE FUNCTIONS						*
*********************************************************************************/


/**
 * qos_adm_set_policer_conf -
 *
 *
 */
static int qos_adm_set_policer_conf(struct _QOS_ADM *this_adm, struct _ADM_POLICER_CONF *conf)
{
	int rc = 0;
	unsigned long bps;
	unsigned long clk_sel, fraction;

	QOS_PRINTK (QOS_POLICER, "policer %d rate %ld max_credit %ld init_burst %ld state %d", conf->index, conf->rate, conf->max_credit, conf->init_burst, conf->state);

	if(conf->index > ADM_MAX_POLICER_ENTRIES)
	{
		QOS_PRINTK (QOS_ERR, "qos_adm_set_policer_conf(): policer %d is out of range", conf->index);
		rc = 1;
	}
	else 
	{
		if(conf->rate)
		{
			bps = conf->rate;
			get_policer_rate_and_clock(bps, &clk_sel, &fraction);
			printk("bps=%ld, clk_sel=%ld, fraction=0x%lx\n", bps, clk_sel, fraction);
		}
		else
		{
			fraction = 0;
			clk_sel	= 15;
		}
		
		ADM_SET_POLICER_FRAC_RATE(this_adm->index, conf->index, fraction);
		ADM_SET_POLICER_MAX_CREDIT(this_adm->index, conf->index, conf->max_credit);
		ADM_SET_POLICER_CLOCK(this_adm->index, conf->index, clk_sel);
		ADM_SET_POLICER_ENABLE(this_adm->index, conf->index, conf->state);
		QOS_PRINTK (QOS_POLICER, "POLICER (%p) index %d state is %d", ADM_POL_FRAC_RATE(this_adm->index, conf->index),conf->index, conf->state);

	}

	return rc;
}

/**
 * qos_adm_get_policer_conf -
 *
 *
 */
static int qos_adm_get_policer_conf(struct _QOS_ADM *this_adm, struct _ADM_POLICER_CONF *conf)
{
	int rc = 0;
	
	if(conf->index > ADM_MAX_POLICER_ENTRIES)
	{
		QOS_PRINTK (QOS_ERR, "qos_adm_get_policer_conf(): policer %d is out of range", conf->index);
		rc = 1;
	}
	else 
	{
		conf->rate = ADM_GET_POLICER_FRAC_RATE(this_adm->index, conf->index);
		conf->max_credit = ADM_GET_POLICER_MAX_CREDIT(this_adm->index, conf->index);
		conf->init_burst = ADM_GET_POLICER_INIT_BURST(this_adm->index, conf->index);
		conf->state = ADM_GET_POLICER_ENABLE(this_adm->index, conf->index);
	}
	return rc;
}


/**
 * qos_adm_set_camentry_state -
 *
 *
 */
static int qos_adm_set_camentry_state(struct _QOS_ADM *this_adm, struct _ADM_TCAM_STATE *conf)
{
	
	if((conf->index > ADM_MAX_TCAM_ENTRIES) || (this_adm->cam_sw_table[conf->index].slot == ADM_TCAM_FREE))
	{
		QOS_PRINTK (QOS_ERR, "camentry %d not found", conf->index);
		return QOS_TCAM_INDEX_ERR;
	}
	else
	{

		QOS_PRINTK (QOS_TCAM_DEBUG, "camentry %d state set to %d", conf->index, conf->state);
		
		ADM_SET_TCAM_ENABLE(this_adm->index, this_adm->cam_sw_table[conf->index].slot, conf->state);

		return QOS_OK;
	}
}


/**
 * qos_adm_get_camentry -
 *
 *
 */
static int qos_adm_get_camentry(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf)
{
	int rc = 0;
	unsigned char entry;
	
	if(conf->index > ADM_MAX_TCAM_ENTRIES)
	{
		QOS_PRINTK (QOS_ERR, "camentry %d is out of range", conf->index);
		rc = 1;
	}
	else 
	{
		entry = this_adm->cam_sw_table[conf->index].slot;
		if(entry < ADM_MAX_TCAM_ENTRIES)
		{
			conf->state = ADM_GET_TCAM_ENABLE(this_adm->index, entry);
			conf->entry.policer = ADM_GET_TCAM_POLICER(this_adm->index, entry); 
			conf->entry.drop = ADM_GET_TCAM_DROP(this_adm->index, entry); 
			conf->entry.reserved = ADM_GET_TCAM_RSVD(this_adm->index, entry);
			conf->entry.etype = ADM_GET_TCAM_ETHERTYPE(this_adm->index, entry);
			conf->entry.vlan = ADM_GET_TCAM_VLANID(this_adm->index, entry);
			conf->entry.pppoe_ipv4 = ADM_GET_TCAM_PPPOEPROTO_IP4(this_adm->index, entry);
			conf->entry.pppoe_ipv6 = ADM_GET_TCAM_PPPOEPROTO_IP6(this_adm->index, entry);
			conf->entry.iptos = ADM_GET_TCAM_IPTOS(this_adm->index, entry);
			conf->entry.ipproto = ADM_GET_TCAM_IPPROTO(this_adm->index, entry);
			conf->entry.ipfamily = this_adm->cam_sw_table[conf->index].entry.ipfamily;
			if(conf->entry.ipfamily == TCAM_IPV4_FAMILY)
			{
				conf->entry.saddr[0] = ADM_GET_TCAM_IPSRC(this_adm->index, entry);
				conf->entry.saddr[0] = swapbit(conf->entry.saddr[0]);
				conf->entry.daddr[0] = ADM_GET_TCAM_IPDST(this_adm->index, entry);
				conf->entry.daddr[0] = swapbit(conf->entry.daddr[0]);
			}
			else
			{
				memcpy((unsigned char*)conf->entry.saddr, (unsigned char*)this_adm->cam_sw_table[conf->index].entry.saddr, 16);
				memcpy((unsigned char*)conf->entry.daddr, (unsigned char*)this_adm->cam_sw_table[conf->index].entry.daddr, 16);
			}
			conf->entry.sport_max = ADM_GET_TCAM_L4_SPORT_MAX(this_adm->index, entry);
			conf->entry.sport_min = ADM_GET_TCAM_L4_SPORT_MIN(this_adm->index, entry);
			conf->entry.dport_max = ADM_GET_TCAM_L4_DPORT_MAX(this_adm->index, entry);
			conf->entry.dport_min = ADM_GET_TCAM_L4_DPORT_MIN(this_adm->index, entry);
			conf->entry.multicast = ADM_GET_TCAM_MCAST(this_adm->index, entry);
			conf->entry.etype_mask = ADM_GET_TCAM_ETHERTYPE_MASK(this_adm->index, entry);
			conf->entry.vlan_mask = ADM_GET_TCAM_VLAN_MASK(this_adm->index, entry);

			conf->entry.saddr_mask = ADM_GET_TCAM_SADDR_MASK(this_adm->index, entry);
			conf->entry.saddr_mask = expand_ip_mask(conf->entry.saddr_mask);
			conf->entry.daddr_mask = ADM_GET_TCAM_DADDR_MASK(this_adm->index, entry);
			conf->entry.daddr_mask = expand_ip_mask(conf->entry.daddr_mask);

			conf->entry.sport_enable = ADM_GET_TCAM_SPORT_MASK(this_adm->index, entry);
			conf->entry.dport_enable = ADM_GET_TCAM_DPORT_MASK(this_adm->index, entry);
			conf->entry.pppoe_mask =ADM_GET_TCAM_PPPOE_MASK(this_adm->index, entry);
			conf->entry.iptos_mask = ADM_GET_TCAM_IPTOS_MASK(this_adm->index, entry);
			conf->entry.pppoe_mask = ADM_GET_TCAM_PPPOE_MASK(this_adm->index, entry);
			conf->entry.ipproto_mask = ADM_GET_TCAM_IPPROTO_MASK(this_adm->index, entry);
			conf->entry.broadcast_mask = ADM_GET_TCAM_BCAST_MASK(this_adm->index, entry);
			conf->entry.multicast_mask = ADM_GET_TCAM_MCAST_MASK(this_adm->index, entry);
		
			QOS_PRINTK(QOS_TCAM_DEBUG, \
			"qos_adm_get_camentry() %d state %d @ %p\n\
			policer         %d\n \
			drop            %d\n \
			etype           %x\n \
			vlan            %x\n \
			pppoe_ipv4      %x\n \
			pppoe_ipv6      %x\n \
			iptos           %x\n \
			ipproto         %x\n \
			ipfamily	%x\n \
			saddr           %lx\n \
			daddr           %lx\n \
			sport_max       %d\n \
			sport_min       %d\n \
			dport_max       %d\n \
			dport_min       %d\n \
			multicast       %x\n \
			etype_mask      %x\n \
			vlan_mask       %x\n \
			saddr_mask      %lx\n \
			daddr_mask      %lx\n \
			sport_enable    %d\n \
			dport_enable    %d\n \
			iptos_mask      %lx\n \
			pppoe_mask      %lx\n \
			ipproto_mask    %lx\n \
			broadcast_mask  %lx\n \
			multicast_mask  %lx\n\n",
			entry, conf->state,
			ADM_GET_TCAM_ENTRY(this_adm->index, entry),
			conf->entry.policer,
			conf->entry.drop,
			conf->entry.etype,
			conf->entry.vlan,
			conf->entry.pppoe_ipv4,
			conf->entry.pppoe_ipv6,
			conf->entry.iptos,
			conf->entry.ipproto,
			conf->entry.ipfamily,
			conf->entry.saddr[0],
			conf->entry.daddr[0],
			conf->entry.sport_max,
			conf->entry.sport_min,
			conf->entry.dport_max,
			conf->entry.dport_min,
			conf->entry.multicast,
			conf->entry.etype_mask,
			conf->entry.vlan_mask,
			conf->entry.saddr_mask,
			conf->entry.daddr_mask,
			conf->entry.sport_enable,
			conf->entry.dport_enable,
			conf->entry.iptos_mask,
			conf->entry.pppoe_mask,
			conf->entry.ipproto_mask,
			conf->entry.broadcast_mask,
			conf->entry.multicast_mask
			);
		}
	}

	return rc;
}

/**
 * qos_adm_remove_camentry -
 *
 *
 */
static int qos_adm_remove_camentry(struct _QOS_ADM *this_adm, unsigned char tcam_entry_id)
{
	struct _ADM_TCAM_CONF tcam_conf;
	unsigned int tcam_slot;

	if(this_adm->cam_sw_table[tcam_entry_id].slot == ADM_TCAM_FREE)
	{
		QOS_PRINTK (QOS_ERR, "camentry %d already free\n", tcam_entry_id);
		return QOS_TCAM_INDEX_ERR;
	}
	else
	{ 	
		/* get hw slot associated to the entry to be removed */
		tcam_slot = this_adm->cam_sw_table[tcam_entry_id].slot;

		/* this id can be used again */
		this_adm->cam_sw_table[tcam_entry_id].slot = ADM_TCAM_FREE;

		/*clear entry in sw table */
		memset(&this_adm->cam_sw_table[tcam_entry_id].entry, sizeof(struct _ADM_TCAM_ENTRY), 0);

		/* do not move up once lastest entry or end of array are reached */
		while((tcam_slot < (ADM_MAX_TCAM_ENTRIES - 1)) && 
			(this_adm->cam_hw_table[tcam_slot + 1].slot != ADM_TCAM_FREE)  && 
			(tcam_entry_id  < (ADM_MAX_TCAM_ENTRIES - 1)))
		{
			/* disable entry n + 1 before moving it */
			ADM_SET_TCAM_ENABLE(this_adm->index, tcam_slot + 1, 0);

			/* copy entry n + 1 to entry n */
			tcam_conf.index = tcam_entry_id + 1;
			qos_adm_get_camentry(this_adm, &tcam_conf);
			qos_adm_add_camentry_to_slot(this_adm, &tcam_conf, tcam_slot);

			/* enable entry n */
			ADM_SET_TCAM_ENABLE(this_adm->index, tcam_slot, 1);
		
			this_adm->cam_hw_table[tcam_slot].slot = this_adm->cam_hw_table[tcam_slot + 1].slot;
			this_adm->cam_sw_table[tcam_entry_id + 1].slot = tcam_slot;

			tcam_slot++; tcam_entry_id++;
		}
	}

	/* one entry less in the list thus the last one must be marked as free */
	this_adm->cam_hw_table[tcam_slot].slot = ADM_TCAM_FREE;

	ADM_SET_TCAM_ENABLE(this_adm->index, tcam_slot, 0);

	ADM_SET_TCAM_POLICER(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_DROP(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_RSVD(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_ETHERTYPE(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_VLANID(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_PPPOEPROTO_IP4(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_PPPOEPROTO_IP6(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_IPTOS(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_IPPROTO(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_IPSRC(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_IPDST(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_L4_SPORT(this_adm->index, tcam_slot, 0, 0);
	ADM_SET_TCAM_L4_DPORT(this_adm->index, tcam_slot, 0, 0);
	ADM_SET_TCAM_MCAST(this_adm->index, tcam_slot, 0);

	ADM_SET_TCAM_ETHERTYPE_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_VLAN_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_SADDR_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_DADDR_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_SPORT_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_DPORT_MASK(this_adm->index, tcam_slot, 0); 
	ADM_SET_TCAM_IPTOS_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_PPPOE_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_IPPROTO_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_MCAST_MASK(this_adm->index, tcam_slot, 0);
	ADM_SET_TCAM_BCAST_MASK(this_adm->index, tcam_slot, 0);

	QOS_PRINTK (QOS_TCAM_DEBUG, "camentry id %d @ slot %d cleared\n",tcam_entry_id, tcam_slot);

	return QOS_OK;
}

/**
 * qos_adm_add_camentry -
 * This function allocate a new entry in the TCAM array and set the conf->index to a unique
 * value corresponding to the identifier associated to the TCAM slot used in the hardware. The identifer and the hw slot 
 * number may be different. This software identifier is to be used by the user space application for any requests concerning 
 * the CAM entry being configured here.
 */
static int qos_adm_add_camentry(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf)
{
	unsigned char i;
	unsigned char free_handle = ADM_TCAM_FREE;
	unsigned char free_slot = ADM_TCAM_FREE;
	unsigned long *tmpU32 = NULL;

	/* go through both sw and hw cam entry number and find a free room */
	for(i = 0; i < ADM_TCAM_MAX_ENTRIES; i++) 
	{
		if((this_adm->cam_sw_table[i].slot == ADM_TCAM_FREE) && (free_handle == ADM_TCAM_FREE))
			free_handle = i; /* this is the indentifer associated to the the tcam entry */	
		if((this_adm->cam_hw_table[i].slot == ADM_TCAM_FREE) && (free_slot == ADM_TCAM_FREE))
			free_slot = i; /* this is where the tcam entry is located in the hardware */
	}

	if((free_handle == ADM_TCAM_FREE) || (free_slot == ADM_TCAM_FREE))
	{
		QOS_PRINTK (QOS_ERR, "no more room for new cam entry free_handle = %d free_slot = %d\n", free_handle, free_slot);
		return QOS_TCAM_NO_MORE_ROOM;
	}
	else
	{
		this_adm->cam_sw_table[free_handle].slot = free_slot; /* sw to hw identifer mapping */
		this_adm->cam_hw_table[free_slot].slot = free_handle; /* hw to sw identifier mapping */

		conf->index = free_handle; /* this handle is will be used by user space application */

	 	tmpU32 = (unsigned long *)ADM_GET_TCAM_ENTRY(this_adm->index, free_slot);
		QOS_PRINTK(QOS_TCAM_DEBUG, "qos_adm_add_camentry() adm %d entry %d is at %p\n", this_adm->index, free_slot, (unsigned long*)tmpU32);
		QOS_PRINTK(QOS_TCAM_DEBUG, \
		"\n\
		policer         %d\n \
		drop            %d\n \
		etype           %x\n \
		vlan            %x\n \
		pppoe_ipv4      %x\n \
		pppoe_ipv6      %x\n \
		iptos           %x\n \
		ipproto         %x\n \
		ipfamily	%x\n \
		saddr           %lx\n \
		daddr           %lx\n \
		sport_max       %d\n \
		sport_min       %d\n \
		dport_max       %d\n \
		dport_min       %d\n \
		multicast       %x\n \
		etype_mask      %x\n \
		vlan_mask       %x\n \
		saddr_mask      %lx\n \
		daddr_mask      %lx\n \
		sport_enable    %d\n \
		dport_enable    %d\n \
		iptos_mask      %lx\n \
		pppoe_mask      %lx\n \
		ipproto_mask    %lx\n \
		broadcast_mask  %lx\n \
		multicast_mask  %lx\n\n",
		conf->entry.policer,
		conf->entry.drop,
		conf->entry.etype,
		conf->entry.vlan,
		conf->entry.pppoe_ipv4,
		conf->entry.pppoe_ipv6,
		conf->entry.iptos,
		conf->entry.ipproto,
		conf->entry.ipfamily,
		conf->entry.saddr[0],
		conf->entry.daddr[0],
		conf->entry.sport_max,
		conf->entry.sport_min,
		conf->entry.dport_max,
		conf->entry.dport_min,
		conf->entry.multicast,
		conf->entry.etype_mask,
		conf->entry.vlan_mask,
		conf->entry.saddr_mask,
		conf->entry.daddr_mask,
		conf->entry.sport_enable,
		conf->entry.dport_enable,
		conf->entry.iptos_mask,
		conf->entry.pppoe_mask,
		conf->entry.ipproto_mask,
		conf->entry.broadcast_mask,
		conf->entry.multicast_mask
		);

		/* only complete mask are supported, so overwrite any mask if not 0 (1bit length masks and 
		ip src/dst masks are not overwritten */
		if(conf->entry.etype_mask)
				conf->entry.etype_mask = 0xFFFF;
		if(conf->entry.ipproto_mask)
				conf->entry.ipproto_mask = 0xFF;
		if(conf->entry.iptos_mask)
				conf->entry.iptos_mask = 0xFF;
		if(conf->entry.vlan_mask)
				conf->entry.vlan_mask = 0xFFFF;

		/* now let's configure tcam block using the hw slot number */
		ADM_SET_TCAM_ENABLE(this_adm->index, free_slot, 0);

		ADM_SET_TCAM_POLICER(this_adm->index, free_slot, conf->entry.policer);
		ADM_SET_TCAM_DROP(this_adm->index, free_slot, conf->entry.drop);
		ADM_SET_TCAM_RSVD(this_adm->index, free_slot,conf->entry.reserved);
		ADM_SET_TCAM_ETHERTYPE(this_adm->index, free_slot, conf->entry.etype);
		ADM_SET_TCAM_VLANID(this_adm->index, free_slot, conf->entry.vlan);
		ADM_SET_TCAM_PPPOEPROTO_IP4(this_adm->index, free_slot, conf->entry.pppoe_ipv4);
		ADM_SET_TCAM_PPPOEPROTO_IP6(this_adm->index, free_slot, conf->entry.pppoe_ipv6);
		ADM_SET_TCAM_IPTOS(this_adm->index, free_slot, conf->entry.iptos);
		ADM_SET_TCAM_IPPROTO(this_adm->index, free_slot, conf->entry.ipproto);
		ADM_SET_TCAM_IPSRC(this_adm->index, free_slot, swapbit(conf->entry.saddr[0]));
		ADM_SET_TCAM_IPDST(this_adm->index, free_slot, swapbit(conf->entry.daddr[0]));
		ADM_SET_TCAM_L4_SPORT(this_adm->index, free_slot, conf->entry.sport_min, conf->entry.sport_max);
		ADM_SET_TCAM_L4_DPORT(this_adm->index, free_slot, conf->entry.dport_min, conf->entry.dport_max);
		ADM_SET_TCAM_MCAST(this_adm->index, free_slot, conf->entry.multicast);
		ADM_SET_TCAM_ETHERTYPE_MASK(this_adm->index, free_slot, conf->entry.etype_mask);
		ADM_SET_TCAM_VLAN_MASK(this_adm->index, free_slot, conf->entry.vlan_mask);
		ADM_SET_TCAM_SADDR_MASK(this_adm->index, free_slot, reduce_ip_mask(conf->entry.saddr_mask));
		ADM_SET_TCAM_DADDR_MASK(this_adm->index, free_slot, reduce_ip_mask(conf->entry.daddr_mask));
		ADM_SET_TCAM_SPORT_MASK(this_adm->index, free_slot, conf->entry.sport_enable);
		ADM_SET_TCAM_DPORT_MASK(this_adm->index, free_slot, conf->entry.dport_enable); 
		ADM_SET_TCAM_IPTOS_MASK(this_adm->index, free_slot, conf->entry.iptos_mask);
		ADM_SET_TCAM_PPPOE_MASK(this_adm->index, free_slot, conf->entry.pppoe_mask);
		ADM_SET_TCAM_IPPROTO_MASK(this_adm->index, free_slot, conf->entry.ipproto_mask);
		ADM_SET_TCAM_MCAST_MASK(this_adm->index, free_slot, conf->entry.multicast_mask);
		ADM_SET_TCAM_BCAST_MASK(this_adm->index, free_slot, conf->entry.broadcast_mask);

		ADM_SET_TCAM_ENABLE(this_adm->index, free_slot, 1);

		/* save tcam entry (inlcuding extra parameters) in the sw table */
		memcpy(&this_adm->cam_sw_table[free_handle].entry, &conf->entry, sizeof(struct _ADM_TCAM_ENTRY));

		QOS_PRINTK(QOS_TCAM_DEBUG, "after write to regs\n");
		for(i = 0; i < ADM_TCAM_ENTRY_SIZE / 4; i++)
			QOS_PRINTK(QOS_TCAM_DEBUG, "%p = %8lx\n",&tmpU32[i], tmpU32[i]);
	}
#if 0
	// for debug purpose only list all entries
	QOS_PRINTK(QOS_TCAM_DEBUG,"qos_adm_add_camentry:\n");	
	for(i = 0; i < ADM_TCAM_MAX_ENTRIES; i++) 
	{
		QOS_PRINTK(QOS_TCAM_DEBUG,"this_adm->cam_sw_table[%d] = %d ", i, this_adm->cam_sw_table[i].slot);
		QOS_PRINTK(QOS_TCAM_DEBUG,"this_adm->cam_hw_table[%d] = %d \n", i, this_adm->cam_hw_table[i].slot);
	}
#endif
	return QOS_OK;
}


/**
 * qos_adm_add_camentry_to_slot -
 *
 *
 */
static int qos_adm_add_camentry_to_slot(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf, unsigned char tcam_slot)
{
	if(tcam_slot >= ADM_MAX_TCAM_ENTRIES)
	{
		QOS_PRINTK (QOS_ERR, "slot %d out of range\n", tcam_slot);
		return QOS_TCAM_INDEX_ERR;
	}
	else
	{
		ADM_SET_TCAM_POLICER(this_adm->index, tcam_slot, conf->entry.policer);
		ADM_SET_TCAM_DROP(this_adm->index, tcam_slot, conf->entry.drop);
		ADM_SET_TCAM_RSVD(this_adm->index, tcam_slot,conf->entry.reserved);
		ADM_SET_TCAM_ETHERTYPE(this_adm->index, tcam_slot, conf->entry.etype);
		ADM_SET_TCAM_VLANID(this_adm->index, tcam_slot, conf->entry.vlan);
		ADM_SET_TCAM_PPPOEPROTO_IP4(this_adm->index, tcam_slot, conf->entry.pppoe_ipv4);
		ADM_SET_TCAM_PPPOEPROTO_IP6(this_adm->index, tcam_slot, conf->entry.pppoe_ipv6);
		ADM_SET_TCAM_IPTOS(this_adm->index, tcam_slot, conf->entry.iptos);
		ADM_SET_TCAM_IPPROTO(this_adm->index, tcam_slot, conf->entry.ipproto);
		ADM_SET_TCAM_IPSRC(this_adm->index, tcam_slot, swapbit(conf->entry.saddr[0]));
		ADM_SET_TCAM_IPDST(this_adm->index, tcam_slot, swapbit(conf->entry.daddr[0]));
		ADM_SET_TCAM_L4_SPORT(this_adm->index, tcam_slot, conf->entry.sport_min, conf->entry.sport_max);
		ADM_SET_TCAM_L4_DPORT(this_adm->index, tcam_slot, conf->entry.dport_min, conf->entry.dport_max);
		ADM_SET_TCAM_MCAST(this_adm->index, tcam_slot, conf->entry.multicast);

		ADM_SET_TCAM_ETHERTYPE_MASK(this_adm->index, tcam_slot, conf->entry.etype_mask);
		ADM_SET_TCAM_VLAN_MASK(this_adm->index, tcam_slot, conf->entry.vlan_mask);
		ADM_SET_TCAM_SADDR_MASK(this_adm->index, tcam_slot, reduce_ip_mask(conf->entry.saddr_mask));
		ADM_SET_TCAM_DADDR_MASK(this_adm->index, tcam_slot, reduce_ip_mask(conf->entry.daddr_mask));
		ADM_SET_TCAM_SPORT_MASK(this_adm->index, tcam_slot, conf->entry.sport_enable);
		ADM_SET_TCAM_DPORT_MASK(this_adm->index, tcam_slot, conf->entry.dport_enable); 
		ADM_SET_TCAM_IPTOS_MASK(this_adm->index, tcam_slot, conf->entry.iptos_mask);
		ADM_SET_TCAM_PPPOE_MASK(this_adm->index, tcam_slot, conf->entry.pppoe_mask);
		ADM_SET_TCAM_IPPROTO_MASK(this_adm->index, tcam_slot, conf->entry.ipproto_mask);
		ADM_SET_TCAM_MCAST_MASK(this_adm->index, tcam_slot, conf->entry.multicast_mask);
	}

	return QOS_OK;
}

/**
 * qos_adm_change_camentry -
 *
 *
 */
static int qos_adm_change_camentry(struct _QOS_ADM *this_adm, struct _ADM_TCAM_CONF *conf)
{
	unsigned int tcam_slot;

	if(this_adm->cam_sw_table[conf->index].slot == ADM_TCAM_FREE)
	{
		QOS_PRINTK (QOS_ERR, "camentry %d not found\n", conf->index);
		return QOS_TCAM_INDEX_ERR;
	}
	else
	{
		tcam_slot = this_adm->cam_sw_table[conf->index].slot;
		ADM_SET_TCAM_POLICER(this_adm->index, tcam_slot, conf->entry.policer);
		ADM_SET_TCAM_DROP(this_adm->index, tcam_slot, conf->entry.drop);
		ADM_SET_TCAM_RSVD(this_adm->index, tcam_slot,conf->entry.reserved);
		ADM_SET_TCAM_ETHERTYPE(this_adm->index, tcam_slot, conf->entry.etype);
		ADM_SET_TCAM_VLANID(this_adm->index, tcam_slot, conf->entry.vlan);
		ADM_SET_TCAM_PPPOEPROTO_IP4(this_adm->index, tcam_slot, conf->entry.pppoe_ipv4);
		ADM_SET_TCAM_PPPOEPROTO_IP6(this_adm->index, tcam_slot, conf->entry.pppoe_ipv6);
		ADM_SET_TCAM_IPTOS(this_adm->index, tcam_slot, conf->entry.iptos);
		ADM_SET_TCAM_IPPROTO(this_adm->index, tcam_slot, conf->entry.ipproto);
		ADM_SET_TCAM_IPSRC(this_adm->index, tcam_slot, swapbit(conf->entry.saddr[0]));
		ADM_SET_TCAM_IPDST(this_adm->index, tcam_slot, swapbit(conf->entry.daddr[0]));
		ADM_SET_TCAM_L4_SPORT(this_adm->index, tcam_slot, conf->entry.sport_min, conf->entry.sport_max);
		ADM_SET_TCAM_L4_DPORT(this_adm->index, tcam_slot, conf->entry.dport_min, conf->entry.dport_max);
		ADM_SET_TCAM_MCAST(this_adm->index, tcam_slot, conf->entry.multicast);

		ADM_SET_TCAM_ETHERTYPE_MASK(this_adm->index, tcam_slot, conf->entry.etype_mask);
		ADM_SET_TCAM_VLAN_MASK(this_adm->index, tcam_slot, conf->entry.vlan_mask);
		ADM_SET_TCAM_SADDR_MASK(this_adm->index, tcam_slot, reduce_ip_mask(conf->entry.saddr_mask));
		ADM_SET_TCAM_DADDR_MASK(this_adm->index, tcam_slot, reduce_ip_mask(conf->entry.daddr_mask));
		ADM_SET_TCAM_SPORT_MASK(this_adm->index, tcam_slot, conf->entry.sport_enable);
		ADM_SET_TCAM_DPORT_MASK(this_adm->index, tcam_slot, conf->entry.dport_enable); 
		ADM_SET_TCAM_IPTOS_MASK(this_adm->index, tcam_slot, conf->entry.iptos_mask);
		ADM_SET_TCAM_PPPOE_MASK(this_adm->index, tcam_slot, conf->entry.pppoe_mask);
		ADM_SET_TCAM_IPPROTO_MASK(this_adm->index, tcam_slot, conf->entry.ipproto_mask);
		ADM_SET_TCAM_MCAST_MASK(this_adm->index, tcam_slot, conf->entry.multicast_mask);

		return QOS_OK;
	}
}


/**
 * qos_adm_set_discard_conf -
 *
 *
 */
static int qos_adm_set_discard_conf(struct _QOS_ADM *this_adm, struct _ADM_DISCARD_CONF *conf)
{
	int rc = 0;
	unsigned long qfull, qmin, qmax = 0;
	
	if(conf->enable == 0)
	{
		QOS_PRINTK (QOS_INFO, "ADM_SET_CTRL_DISCARD_DISABLE\n");
	}
	else
	{
		ADM_SET_CONF_PROB0(this_adm->index, conf->zone_prob[0]);
		ADM_SET_CONF_PROB1(this_adm->index, conf->zone_prob[1]);
		ADM_SET_CONF_PROB2(this_adm->index, conf->zone_prob[2]);
		ADM_SET_CONF_PROB3(this_adm->index, conf->zone_prob[3]);

		qfull = ADM_GET_Q_FULL_THRESHOLD(this_adm->index);
		qmax = ((conf->queue_drop_max * qfull) / 100);
		// need at least 16 buffers for reserved traffic (packets in flight in GEM fifo)
		if ( qmax > (qfull - 16)) 
			qmax = qfull - 16;
		qmin = (conf->queue_drop_min * qfull) / 100;
		// need at least 4 buffers for the 4 zones (1 per zone)
		if (qmin > (qmax - 4))
			qmin = qmax - 4; 

		ADM_SET_Q_DROP_MIN(this_adm->index, qmin);
		ADM_SET_Q_DROP_MAX(this_adm->index, qmax);

		QOS_PRINTK (QOS_INFO, "ADM_SET_CTRL_DISCARD_ENABLE (qfull %lu qmax %lu qmin %lu)\n", qfull, qmax, qmin);
	}

	return rc;
}


/**
 * qos_adm_set_flowctrl_conf -
 *
 *
 */
static int qos_adm_set_flowctrl_conf(struct _QOS_ADM *this_adm, struct _ADM_FLOWCTRL_CONF *conf)
{
	int rc = 0;
	unsigned long qfull, threshon, threshoff = 0;

	if(conf->enable == 0)
	{
		QOS_PRINTK (QOS_INFO, "ADM_SET_CTRL_FLOWCTRL_DISABLE\n");
	}
	else
	{
		qfull = ADM_GET_Q_FULL_THRESHOLD(this_adm->index);
		threshon = (conf->on_tresh * qfull) / 100;
		// need at least 20 buffers for reserved traffic (16 packets in flight in GEM fifo + 4 packets on the line)
		if ( threshon > (qfull - 20)) 
			threshon = qfull - 20;
		
		threshoff = (conf->off_tresh * qfull) / 100;
		if (threshoff >= threshon)
			threshoff = threshon - 1; 

		ADM_SET_FLOWCTRL_ON_TRESHOLD(this_adm->index, threshon);
		ADM_SET_FLOWCTRL_OFF_TRESHOLD(this_adm->index, threshoff);
		ADM_SET_DEPTH_PAUSE_INT_TIMER(this_adm->index, conf->pause_timer);

		QOS_PRINTK (QOS_INFO, "ADM_SET_CTRL_FLOWCTRL_ENABLE\n");
	}

	return rc;
}


/**
 * qos_adm_set_shaper_conf -
 *
 *
 */
static int qos_adm_set_shaper_conf(struct _QOS_ADM *this_adm, struct _ADM_SHAPER_CONF *conf)
{
	int rc = 0;
	unsigned long bps;
	unsigned long clk_sel, fraction;

	if(conf->enable == 0)
	{
		ADM_SET_PORT_SHA_ENABLE(this_adm->index, 0);
		QOS_PRINTK (QOS_SHAPER, "qos_adm_set_shaper_conf: ADM_SET_PORT_SHA_DISABLE\n");
	}
	else
	{
		if(conf->rate)
		{
			bps = conf->rate;
			get_shaper_rate_and_clock(bps, &clk_sel, &fraction);
			QOS_PRINTK (QOS_SHAPER, "qos_adm_set_shaper_conf: bps=%ld, clk_sel=%ld, fraction=%ld\n", bps, clk_sel, fraction);
		}
		else
		{
			fraction = 0;
			clk_sel	= 15;
		}

		ADM_SET_PORT_SHA_PKT_OVER(this_adm->index, conf->overhead); //sw wa - registers 0x4030 must be written prior to 0x4020
		ADM_SET_PORT_SHA_FRAC_RATE(this_adm->index, fraction);
		ADM_SET_PORT_SHA_MAX_CREDIT(this_adm->index, conf->max_credit);
		ADM_SET_PORT_SHA_CLOCK(this_adm->index, clk_sel);
		ADM_SET_PORT_SHA_ENABLE(this_adm->index, 1);
		QOS_PRINTK (QOS_SHAPER, "qos_adm_set_shaper_conf: ADM_SET_PORT_SHA_ENABLE\n");

	}

	return rc;
}

/**
 * qos_adm_get_shaper_conf -
 *
 *
 */
static int qos_adm_get_shaper_conf(struct _QOS_ADM *this_adm, struct _ADM_SHAPER_CONF *conf)
{
	int rc = QOS_OK;
	unsigned long clk, fraction;

	fraction = ADM_GET_PORT_SHA_FRAC_RATE(this_adm->index);
	clk = ADM_GET_PORT_SHA_CLOCK(this_adm->index);
	conf->rate = shaper_fraction_clk_to_rate(fraction, clk);
	conf->max_credit = ADM_GET_PORT_SHA_MAX_CREDIT(this_adm->index);
	conf->overhead = ADM_GET_PORT_SHA_PKT_OVER(this_adm->index);
	conf->enable = ADM_GET_PORT_SHA_ENABLE(this_adm->index);

	QOS_PRINTK (QOS_SHAPER,"qos_adm_get_shaper_conf: ADM%d fraction=%ld, clk=%ld, bps=%ld\n", this_adm->index, fraction, clk, conf->rate);

	return rc;
}




/**
 * qos_adm_set_mode -
 *
 *
 */
static int qos_adm_set_mode(struct _QOS_ADM *this_adm, unsigned char mode)
{
	int rc = 0;

	if(mode >= ADM_MAX_MODE)
	{
		rc = 1;
	}
	else
	{
		if(mode == DISABLED_MODE) {
			QOS_PRINTK (QOS_INFO, "ADM%d BLOCK DISABLED\n", this_adm->index);
			ADM_SET_CONF_MODE_DISCARD(this_adm->index);
			ADM_SET_CTRL_ADM_DISABLE(this_adm->index);
		}
		else {
			if(mode == DISCARD_MODE) {
				QOS_PRINTK (QOS_INFO, "ADM%d DISCARD_MODE or SHAPED_MODE\n", this_adm->index);
				ADM_SET_CONF_MODE_DISCARD(this_adm->index);
			}
			else {
				QOS_PRINTK (QOS_INFO, "ADM%d FLOWCTRL_MODE\n", this_adm->index);
				ADM_SET_CONF_MODE_FLOWCTRL(this_adm->index);
			}
			ADM_SET_CTRL_ADM_ENABLE(this_adm->index);
			QOS_PRINTK (QOS_INFO, "ADM%d BLOCK ENABLED\n", this_adm->index);
		}
	}

	return rc;
}


/**
 * qos_adm_get_mode -
 *
 *
 */
static unsigned char qos_adm_get_mode(struct _QOS_ADM *this_adm)
{
	return ADM_GET_CONF_MODE(this_adm->index);
}



/**
 * qos_adm_set_lru_conf -
 *
 *
 */
static int qos_adm_set_lru_conf(struct _QOS_ADM *this_adm, struct _ADM_LRU_CONF *conf)
{
	int rc = 0;

	if(conf->period)
	{
		ADM_SET_LRU_MASTER_ENABLE(this_adm->index);
		ADM_SET_LRU_PERIOD(this_adm->index, conf->period);
		ADM_SET_LRU_MASK(this_adm->index, conf->mask);
		ADM_SET_LRU_STATE(this_adm->index, 1);
	}
	else
	{
		ADM_SET_LRU_MASTER_DISABLE(this_adm->index);
		ADM_SET_LRU_STATE(this_adm->index, 0);
	}
	return rc;
}


/**
 * qos_adm_get_lru_stats -
 *
 *
 */
static int qos_adm_get_lru_stats(struct _QOS_ADM *this_adm, struct _ADM_LRU_STATS *p)
{
	int size = 0;
	unsigned char tcam_index;

	if (p) 
	{
		size = sizeof(struct _ADM_LRU_STATS);
		p->entry = ADM_GET_LRU_ENTRY(this_adm->index);
		p->count = ADM_GET_LRU_ENTRY_HIT(this_adm->index);
		
		for(tcam_index = 0; tcam_index < ADM_MAX_TCAM_ENTRIES; tcam_index++)
			p->hit_count[tcam_index] = (unsigned char)(ADM_GET_LRU_HIT_GROUP(this_adm->index, tcam_index) >> ((tcam_index & 3) * 8));
	}

	return size;
}



/**
 * qos_adm_get_counters -
 *
 *
 */
static int qos_adm_get_counters(struct _QOS_ADM *this_adm, struct _ADM_COUNTER *p)
{
	int size = 0;

	memset(p, 0, sizeof(struct _ADM_COUNTER));

	if (p) 
	{
		size = sizeof(struct _ADM_COUNTER);
		memcpy(p, this_adm->counter, size);
	}

	return size;
}

/**
 * qos_adm_dump_counters -
 *
 *
 */
static void qos_adm_dump_counters(void)
{
	int i;
	
	QOS_PRINTK (QOS_STAT, "qos_adm_dump_counters() adm0 stat @%p adm1 stats @%p\n", _adm[0].counter, _adm[1].counter);

	for(i = ADM0; i <= ADM1; i++) 
	{
		ADM_SET_CTRL_SNAPSHOT_ENABLE(i);
		QOS_PRINTK(QOS_STAT, \
		" \
		Admittance block %d counters:\n \
		port_byte                %ld\n \
		port_packet              %ld\n \
		reserved_byte            %ld\n \
		reserved_packet          %ld\n \
		managed_byte             %ld\n \
		managed_packet           %ld\n \
		packet_dropped           %ld\n \
		packet_dropped_error     %ld\n \
		packet_dropped_denied    %ld\n \
		packet_dropped_policer0  %ld\n \
		packet_dropped_policer1  %ld\n \
		packet_dropped_shaper    %ld\n \
		packet_dropped_managed   %ld\n \
		packet_dropped_unmanaged %ld\n \
		", \
		i, \
		_adm[i].counter->port_byte, \
		_adm[i].counter->port_packet, \
		_adm[i].counter->reserved_byte, \
		_adm[i].counter->reserved_packet, \
		_adm[i].counter->managed_byte, \
		_adm[i].counter->managed_packet, \
		_adm[i].counter->packet_dropped, \
		_adm[i].counter->packet_dropped_error, \
		_adm[i].counter->packet_dropped_denied, \
		_adm[i].counter->packet_dropped_policer0, \
		_adm[i].counter->packet_dropped_policer1, \
		_adm[i].counter->packet_dropped_shaper, \
		_adm[i].counter->packet_dropped_managed, \
		_adm[i].counter->packet_dropped_unmanaged \
		);
	}

}


/********************************************************************************
*		SCHEDULER FUNCTIONS						*
*********************************************************************************/

/**
 * qos_sch_get_counters -
 *
 *
 */
static int qos_sch_get_counters(struct _QOS_SCH *this_sch, struct _SCH_COUNTER *p)
{
	int i;

	if (p) 
	{
		p->port_byte = SCH_GET_PORT_BYTE_CNT(this_sch->index);
		p->port_packet = SCH_GET_PORT_PKT_CNT(this_sch->index);
		for(i = 0; i < SCH_MAX_QUEUES; i++)
		{
			QOS_PRINTK(QOS_STAT, "SCH_GET_Q_BYTE_IN_Q%d @%p = %ld", i ,SCH_Q_BYTE_IN_Q(this_sch->index,i), SCH_GET_Q_BYTE_IN_Q(this_sch->index, i));
			p->q_byte[i] = SCH_GET_Q_BYTE_IN_Q(this_sch->index, i);
			p->q_packet[i] = SCH_GET_Q_PKT_IN_Q(this_sch->index, i);
			p->q_idle[i] = SCH_GET_Q_IDLE(this_sch->index, i);
		}
		
		return sizeof(struct _SCH_COUNTER);
	}
	else
	{
		return 0;
	}
}

/**
 * qos_sch_dump_counters -
 *
 *
 */
static void qos_sch_dump_counters(void)
{
	int s, q;
	struct _SCH_COUNTER sch_stats;

	for(s = SCH0; s <= SCH1; s++) 
	{
		memset(&sch_stats, 0, sizeof(struct _SCH_COUNTER));
		qos_sch_get_counters(&_sch[s], &sch_stats);

		QOS_PRINTK(QOS_STAT, \
		" \
		Scheduler block %d counters:\n \
		tx byte                %ld\n \
		tx packet              %ld\n \
		", \
		s,
		sch_stats.port_byte, \
		sch_stats.port_packet \
		);
	
		for(q = 0; q < SCH_MAX_QUEUES; q++) 
		{	
			QOS_PRINTK(QOS_STAT, \
			" \
			Scheduler block %d counters for queue %d:\n \
			byte in queue		%ld\n \
			packet in queue		%ld\n \
			queue idle time		%ld\n \
			", \
			s,
			q,
			sch_stats.q_byte[q], \
			sch_stats.q_packet[q], \
			sch_stats.q_idle[q] \
			);
		}
	}
}

/**
 * qos_sch_set_shaper_state -
 *
 *
 */
static int qos_sch_set_shaper_state(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_STATE *conf)
{
	int rc = QOS_OK;	

	switch(conf->shaper)
	{
		case SHAPER_Q7:
			SCH_SET_Q_CONTROL_ENABLE(this_sch->index,7,conf->state);
			break;
		case SHAPER_Q6:
			SCH_SET_Q_CONTROL_ENABLE(this_sch->index,6,conf->state);
			break;
		case SHAPER_GROUP:
			SCH_SET_GROUP_CONTROL_ENABLE(this_sch->index, conf->state);
			break;
		case SHAPER_PORT:
			SCH_SET_PORT_CONTROL_ENABLE(this_sch->index, conf->state);
			break;
		default:
			QOS_PRINTK (QOS_IOCTL, "unknown shaper index");
			rc = -EINVAL;
			break;
	}
	return rc;
}

/**
 * qos_sch_get_shaper_state -
 *
 *
 */
static int qos_sch_get_shaper_state(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_STATE *conf)
{
	int rc = QOS_OK;

	switch(conf->shaper)
	{
		case SHAPER_Q7:
			conf->state = SCH_GET_Q_CONTROL_ENABLE(this_sch->index,7);
			break;
		case SHAPER_Q6:
			conf->state = SCH_GET_Q_CONTROL_ENABLE(this_sch->index,6);
			break;
		case SHAPER_GROUP:
			conf->state = SCH_GET_GROUP_CONTROL_ENABLE(this_sch->index);
			break;
		case SHAPER_PORT:
			conf->state = SCH_GET_PORT_CONTROL_ENABLE(this_sch->index);
			break;
		default:
			QOS_PRINTK (QOS_IOCTL, "unknown shaper index");
			rc = -EINVAL;
			break;
	}

	return rc;
}

/**
 * get_policer_rate_and_clock -
 *
 * cf. c1000 HW Admittance Specs v14 for rate calculation details
 */
static void get_policer_rate_and_clock(unsigned long pps, unsigned long *p_clk_sel, unsigned long *p_fraction)
{
	unsigned long cycles_per_packet;
	unsigned long frac_packet_rate;
	unsigned long clk_sel;
	unsigned long fraction;
	unsigned long clock = (REFCLK/256);

	cycles_per_packet = clock / pps;
	QOS_PRINTK (QOS_POLICER,"cycles_per_packet=%ld, pps=%ld\n", cycles_per_packet, pps);

	/* From Specs: Clock select paramter 0 and 1 are not suported in the Policer block. Starting clk_sel at 0x0010 */
	for(clk_sel = 2; clk_sel < 16; clk_sel++)
	{
		if((1 << clk_sel) >= cycles_per_packet)
			break;
	}

	if(clk_sel == 16)
	{
		clk_sel = 15;
		fraction = 0;
	}
	else
	{
		QOS_PRINTK (QOS_POLICER,"(1 << clk_sel)=%d\n", (1 << clk_sel));
		if(cycles_per_packet == 0) cycles_per_packet = 1;
		frac_packet_rate = ((1 << clk_sel) * 100) / cycles_per_packet;
		QOS_PRINTK (QOS_POLICER,"rate=%ld  -> %ld\n", frac_packet_rate, (frac_packet_rate - 100));
		fraction = (((frac_packet_rate - 100) * 256) / 100);
		QOS_PRINTK (QOS_POLICER,"fraction=%ld clk_sel=%ld\n", fraction, clk_sel);
		if(fraction > 255)
			fraction = 255;
	}

	*p_clk_sel = clk_sel;
	*p_fraction = fraction;
}

/**
 * get_shaper_rate_and_clock -
 *
 * cf. c1000 HW Scheduler Specs v14 for rate calculation details (section Shaper Block)
 */
static void get_shaper_rate_and_clock(unsigned long bps, unsigned long *p_clk_sel, unsigned long *p_fraction)
{
	unsigned long byte_rate;
	unsigned long cycles_per_byte;
	unsigned long clk_sel;
	unsigned long frac_byte_rate;
	unsigned long fraction;

	byte_rate = bps >> 3;
	cycles_per_byte = ((REFCLK * 10) / byte_rate);
	QOS_PRINTK (QOS_SHAPER,"cycles_per_byte_x10=%ld byte_rate =%ld\n", cycles_per_byte,byte_rate);
	for(clk_sel = 0; clk_sel < 16; clk_sel++)
	{
		if(((1 << clk_sel) * 10) >= cycles_per_byte)
			break;
	}
	if(clk_sel == 16)
	{
		clk_sel = 15;
		fraction = 0;
	}
	else
	{
		QOS_PRINTK (QOS_SHAPER,"(1 << clk_sel)=%d\n", (1 << clk_sel));
		if(cycles_per_byte == 0) cycles_per_byte = 1;
		frac_byte_rate = (((1 << clk_sel) * 1000) * 10) / cycles_per_byte;
		QOS_PRINTK (QOS_SHAPER,"rate=%ld  -> %ld\n", frac_byte_rate, (frac_byte_rate - 1000));
		if(frac_byte_rate > 1000)
			fraction = (((frac_byte_rate - 1000) * 256) / 1000);
		else
			fraction = 0;
		QOS_PRINTK (QOS_SHAPER,"fraction=%ld clk_sel=%ld\n", fraction, clk_sel);
		if(fraction > 255)
			fraction = 255;
	}

	*p_clk_sel = clk_sel;
	*p_fraction = fraction;
}

/**
 * shaper_fraction_clk_to_rate -
 *
 * cf. c1000 HW Scheduler Specs v14 for rate calculation details (section Shaper Block)
 */
static unsigned long shaper_fraction_clk_to_rate(unsigned long fraction, unsigned long clk_sel)
{
	unsigned long long rate = 0;

	if(clk_sel != 0) {
		/* without floating point value must be majored (e.g x1000) in order to keep accuracy */
		if(!fraction) 
			fraction=1;
		rate = ((((fraction+256) * 1000)/256) * ((((REFCLK / 100)/(1 << clk_sel)) * 8) / 10));
		QOS_PRINTK (QOS_SHAPER,"shaper_fraction_clk_to_rate fraction=%lu clk_sel=%lu\n", fraction, (unsigned long)1 << clk_sel);
		QOS_PRINTK (QOS_SHAPER,"(((fraction+256)*1000)/256)=%ld, (((REFCLK/100)/(1<<clk_sel))*8)/10)=%ld\n",(((fraction+256) * 1000)/256),(((REFCLK / 100) / (1 << clk_sel)) * 8)/10);
	}
	return (unsigned long)rate;
}

/**
 * qos_sch_set_shaper_conf -
 *
 *
 */
static int qos_sch_set_shaper_conf(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_CONF *conf)
{
	int rc = 0;
	unsigned long bps;
	unsigned long clk_sel, fraction;

	if((conf->rate) && (conf->state))
	{
		bps = conf->rate;
		get_shaper_rate_and_clock(bps, &clk_sel, &fraction);
		QOS_PRINTK (QOS_SHAPER,"qos_sch_set_shaper_conf: bps=%ld, clk_sel=%ld, fraction=0x%lx\n", bps, clk_sel, fraction);
	}
	else
	{
		fraction = 0;
		clk_sel	= 0;
	}

	QOS_PRINTK (QOS_SHAPER,"qos_sch_set_shaper_conf SCH %d shaper %d state %d\n", this_sch->index, conf->shaper,conf->state);

	switch(conf->shaper)
	{
		case SHAPER_Q7:
			SCH_SET_Q_CONTROL_CLOCK(this_sch->index, 7, clk_sel);
			SCH_SET_Q_FRAC_RATE(this_sch->index, 7, fraction);
			SCH_SET_Q_MAX_CREDIT(this_sch->index, 7, conf->max_credit);
			SCH_SET_Q_CONTROL_ENABLE(this_sch->index, 7, conf->state);
			break;
		case SHAPER_Q6:
			SCH_SET_Q_CONTROL_CLOCK(this_sch->index, 6, clk_sel);
			SCH_SET_Q_FRAC_RATE(this_sch->index, 6, fraction);
			SCH_SET_Q_MAX_CREDIT(this_sch->index, 6, conf->max_credit);
			SCH_SET_Q_CONTROL_ENABLE(this_sch->index, 6, conf->state);
			break;
		case SHAPER_GROUP:
			SCH_SET_GROUP_CONTROL_CLOCK(this_sch->index, clk_sel);
			SCH_SET_GROUP_FRAC_RATE(this_sch->index, fraction);
			SCH_SET_GROUP_MAX_CREDIT(this_sch->index, conf->max_credit);
			SCH_SET_GROUP_CONTROL_ENABLE(this_sch->index, conf->state);
			break;
		case SHAPER_PORT:
			SCH_SET_PORT_CONTROL_CLOCK(this_sch->index, clk_sel);
			SCH_SET_PORT_FRAC_RATE(this_sch->index, fraction);
			SCH_SET_PORT_MAX_CREDIT(this_sch->index, conf->max_credit);
			SCH_SET_PORT_CONTROL_ENABLE(this_sch->index, conf->state);
			break;
		default:
			QOS_PRINTK (QOS_ERR, "unknown shaper index");
			rc = -EINVAL;
			break;
	}
	return rc;
}

/**
 * qos_sch_get_shaper_conf -
 *
 *
 */
static int qos_sch_get_shaper_conf(struct _QOS_SCH *this_sch, struct _SCH_SHAPER_CONF *conf)
{
	int rc = QOS_OK;
	unsigned long clk, fraction;

	//printk("qos_sch_get_shaper_conf SCH %d shaper %d\n", this_sch->index, conf->shaper);

	switch(conf->shaper)
	{
		case SHAPER_Q7:
			fraction = SCH_GET_Q_FRAC_RATE(this_sch->index, 7);
			clk = SCH_GET_Q_CONTROL_CLOCK(this_sch->index, 7);
			conf->rate = shaper_fraction_clk_to_rate(fraction, clk);
			conf->max_credit = SCH_GET_Q_MAX_CREDIT(this_sch->index, 7);
			conf->init_burst = SCH_GET_Q_INIT_BURST(this_sch->index, 7);
			conf->state = SCH_GET_Q_CONTROL_ENABLE(this_sch->index, 7);
			break;
		case SHAPER_Q6:
			fraction = SCH_GET_Q_FRAC_RATE(this_sch->index, 6);
			clk = SCH_GET_Q_CONTROL_CLOCK(this_sch->index, 6);
			conf->rate = shaper_fraction_clk_to_rate(fraction, clk);
			conf->max_credit = SCH_GET_Q_MAX_CREDIT(this_sch->index, 6);
			conf->init_burst = SCH_GET_Q_INIT_BURST(this_sch->index, 6);
			conf->state = SCH_GET_Q_CONTROL_ENABLE(this_sch->index, 6);
			break;
		case SHAPER_GROUP:
			fraction = SCH_GET_GROUP_FRAC_RATE(this_sch->index);
			clk = SCH_GET_GROUP_CONTROL_CLOCK(this_sch->index);
			conf->rate = shaper_fraction_clk_to_rate(fraction, clk);
			conf->max_credit = SCH_GET_GROUP_MAX_CREDIT(this_sch->index);
			conf->init_burst = SCH_GET_GROUP_INIT_BURST(this_sch->index);
			conf->state = SCH_GET_GROUP_CONTROL_ENABLE(this_sch->index);
			
			break;
		case SHAPER_PORT:
			fraction = SCH_GET_PORT_FRAC_RATE(this_sch->index);
			clk = SCH_GET_PORT_CONTROL_CLOCK(this_sch->index);
			conf->rate = shaper_fraction_clk_to_rate(fraction, clk);
			conf->max_credit = SCH_GET_PORT_MAX_CREDIT(this_sch->index);
			conf->init_burst = SCH_GET_PORT_INIT_BURST(this_sch->index);
			conf->state = SCH_GET_PORT_CONTROL_ENABLE(this_sch->index);
			break;
		default:
			QOS_PRINTK (QOS_ERR, "unknown shaper index");
			rc = -EINVAL;
			break;
	}

	return rc;
}


/**
 * qos_sch_set_queue_weight -
 *
 *
 */
static int qos_sch_set_queue_weight(struct _QOS_SCH *this_sch, struct _SCH_QUEUE_WEIGHT *conf)
{
	int rc = QOS_OK;
	
	QOS_PRINTK (QOS_SHAPER,"qos_sch_set_queue_weight SCH %d queue %d\n", this_sch->index, conf->queue);

	if((conf->queue > QUEUE_0) && (conf->queue < QUEUE_6))
	{
		SCH_SET_Q_DWRR_WEIGHT(this_sch->index, conf->queue, conf->weight);
	}
	else
	{
		QOS_PRINTK (QOS_ERR, "can't assigned weight to queue %d", conf->queue);
		rc = -EINVAL;
	}
	return rc;
}

/**
 * qos_sch_get_queue_weight -
 *
 *
 */
static int qos_sch_get_queue_weight(struct _QOS_SCH *this_sch, struct _SCH_QUEUE_WEIGHT *conf)
{
	int rc = QOS_OK;

	QOS_PRINTK (QOS_SHAPER,"qos_sch_get_queue_weight SCH %d queue %d\n", this_sch->index, conf->queue);

	if((conf->queue > QUEUE_0) && (conf->queue < QUEUE_6))
	{
		conf->weight = SCH_GET_Q_DWRR_WEIGHT(this_sch->index, conf->queue);
	}
	else
	{
		QOS_PRINTK (QOS_ERR, "can't get weight for queue %d", conf->queue);
		rc = -EINVAL;
	}

	return rc;
}
/********************************************************************************
*		COMMON FUNCTIONS						*
*********************************************************************************/


/**
 * swapbit -
 *
 *
 */
static unsigned long swapbit(unsigned long x)
{
	unsigned long r = x;
	int s = (sizeof(x) * 8) -1;
	
	for(x >>= 1; x; x >>= 1)
	{
		r <<= 1;
		r |= x & 1;
		s--;
	}
	r <<= s;
	return r;
}


/**
 * reduce_ip_mask -
 *
 *
 */
static unsigned char reduce_ip_mask(unsigned long ip_mask)
{
	unsigned char reduced_ip_mask = 0;
	unsigned char num_bit = (sizeof(ip_mask) * 8);
	
	//printk("reduce_ip_mask: from %08lx", ip_mask);

	while(num_bit)
	{
		if(!(ip_mask & 1))
			reduced_ip_mask++;
		num_bit--; ip_mask >>= 1;
	}

	if(reduced_ip_mask >= 0x20)
		reduced_ip_mask = 0x3F;

	//printk(" to %d\n", reduced_ip_mask);

	return reduced_ip_mask;
}


/**
 * expand_ip_mask -
 *
 *
 */
static unsigned long expand_ip_mask(unsigned long ip_mask)
{
	unsigned long expanded_ip_mask = 0xFFFFFFFF;

	if(ip_mask == 0x3F)
		ip_mask = 0x20;

	expanded_ip_mask <<= ip_mask;

	//printk("expand_ip_mask: from %d to %08lx\n", ip_mask, expanded_ip_mask);

	return expanded_ip_mask;
}


/**
 * qos_dump_registers -
 *
 *
 */
static void qos_dump_registers(unsigned char block)
{
	int adm, sch, i;

	if(block & QOS_ADM_BLOCK)
	{
		QOS_PRINTK(QOS_REGS_DUMP, "**********C1K HW QoS ADM Registers**********\n");

		for(adm = ADM0; adm <= ADM1; adm++) 
		{
			QOS_PRINTK(QOS_REGS_DUMP, \
			"\nAdmittance block %d registers:\n \
\tADM_STATUS_REG		(%p) = %08lx\n \
\tADM_PKT_DEQUEUED		(%p) = %08lx\n \
\tADM_CONF_REG			(%p) = %08lx\n \
\tADM_CTRL_REG			(%p) = %08lx\n\n \
\tADM_DEPTH_Q_DEPTH		(%p) = %08lx\n \
\tADM_DEPTH_AVG_Q_DEPTH	   	(%p) = %08lx\n \
\tADM_DEPTH_Q_FULL_THRESHOLD 	(%p) = %08lx\n \
\tADM_DEPTH_Q_DROP_MAX		(%p) = %08lx\n \
\tADM_DEPTH_Q_DROP_MIN	   	(%p) = %08lx\n \
\tADM_DEPTH_PAUSE_INT_TIMER	(%p) = %08lx\n \
\tADM_DEPTH_DECAY_TIMER		(%p) = %08lx\n \
\tADM_DEPTH_DROP_RANDOM		(%p) = %08lx\n\n \
\tADM_PORT_SHA_FRAC_RATE	(%p) = %08lx\n \
\tADM_PORT_SHA_MAX_CREDIT	(%p) = %08lx\n \
\tADM_PORT_SHA_CREDIT		(%p) = %08lx\n \
\tADM_PORT_SHA_CTRL		(%p) = %08lx\n \
\tADM_PORT_SHA_PKT_OVER		(%p) = %08lx\n \
			", 
			adm, 
			ADM_STATUS_REG(adm), *(ADM_STATUS_REG(adm)), 
			ADM_PKT_DEQUEUED_REG(adm), *ADM_PKT_DEQUEUED_REG(adm), 
			ADM_CONF_REG(adm), *ADM_CONF_REG(adm), 
			ADM_CTRL_REG(adm), *ADM_CTRL_REG(adm), 
			ADM_DEPTH_Q_DEPTH(adm), *ADM_DEPTH_Q_DEPTH(adm),
			ADM_DEPTH_AVG_Q_DEPTH(adm), *ADM_DEPTH_AVG_Q_DEPTH(adm),
			ADM_DEPTH_Q_FULL_THRESHOLD(adm), *ADM_DEPTH_Q_FULL_THRESHOLD(adm),
			ADM_DEPTH_Q_DROP_MAX(adm), *ADM_DEPTH_Q_DROP_MAX(adm),
			ADM_DEPTH_Q_DROP_MIN(adm), *ADM_DEPTH_Q_DROP_MIN(adm),
			ADM_DEPTH_PAUSE_INT_TIMER(adm), *ADM_DEPTH_PAUSE_INT_TIMER(adm),
			ADM_DEPTH_DECAY_TIMER(adm), *ADM_DEPTH_DECAY_TIMER(adm),
			ADM_DEPTH_DROP_RANDOM(adm), *ADM_DEPTH_DROP_RANDOM(adm),
			ADM_PORT_SHA_FRAC_RATE(adm), *ADM_PORT_SHA_FRAC_RATE(adm),
			ADM_PORT_SHA_MAX_CREDIT(adm), *ADM_PORT_SHA_MAX_CREDIT(adm),
			ADM_PORT_SHA_CREDIT(adm), *ADM_PORT_SHA_CREDIT(adm),
			ADM_PORT_SHA_CTRL(adm), *ADM_PORT_SHA_CTRL(adm),
			ADM_PORT_SHA_PKT_OVER(adm), *ADM_PORT_SHA_PKT_OVER(adm)
			);

			QOS_PRINTK(QOS_REGS_DUMP, "\nAdmittance block %d TCAM:", adm);

			for(i=0; i< ADM_TCAM_MAX_ENTRIES; i++)
			{
				if(*ADM_TCAM_VALID(adm, i)) {
					QOS_PRINTK(QOS_REGS_DUMP, \
					"\tTCAM %d [%p]\n \
\tADM_TCAM_VALID        = %08lx\tADM_TCAM_STATE        = %08lx\tADM_TCAM_ETHERTYPE    = %08lx\n \
\tADM_TCAM_VLANID       = %08lx\tADM_TCAM_PPPOEPROTO   = %08lx\tADM_TCAM_IPTOS        = %08lx\n \
\tADM_TCAM_IPPROTO      = %08lx\tADM_TCAM_IPSRC        = %08lx\tADM_TCAM_IPDST        = %08lx\n \
\tADM_TCAM_L4_SPORT     = %08lx\tADM_TCAM_L4_DPORT     = %08lx\tADM_TCAM_MCAST        = %08lx\n \
\tADM_TCAM_MASK_REG0    = %08lx\tADM_TCAM_MASK_REG1    = %08lx\tADM_TCAM_MASK_REG2    = %08lx\n \
					",
					i,
					ADM_GET_TCAM_ENTRY(adm, i),
					*ADM_TCAM_VALID(adm, i),
					*ADM_TCAM_STATE(adm, i),
					*ADM_TCAM_ETHERTYPE(adm,i),
					*ADM_TCAM_VLANID(adm,i),
					*ADM_TCAM_PPPOEPROTO(adm,i),
					*ADM_TCAM_IPTOS(adm,i),
					*ADM_TCAM_IPPROTO(adm,i),
					*ADM_TCAM_IPSRC(adm,i),
					*ADM_TCAM_IPDST(adm,i),
					*ADM_TCAM_L4_SPORT(adm,i),
					*ADM_TCAM_L4_DPORT(adm,i),
					*ADM_TCAM_MCAST(adm,i),
					*ADM_TCAM_MASK_REG0(adm,i),
					*ADM_TCAM_MASK_REG1(adm,i),
					*ADM_TCAM_MASK_REG2(adm,i)
					);
				} else {
					QOS_PRINTK(QOS_REGS_DUMP, "\tTCAM %d [%p]\n\tnot used\n ", i, ADM_GET_TCAM_ENTRY(adm, i));
				}
			}

			QOS_PRINTK(QOS_REGS_DUMP, \
			"\nAdmittance block %d LRU Registers:\n \
\tADM_LRU_ENTRY       (%p) = %08lx\n \
\tADM_LRU_ENTRY_MASK  (%p) = %08lx\n \
\tADM_LRU_HIT_CNT_0_3 (%p) = %08lx\n \
\tADM_LRU_HIT_CNT_4_7 (%p) = %08lx\n \
\tADM_LRU_TIMER       (%p) = %08lx\n \
\tADM_LRU_PERIOD      (%p) = %08lx\n \
\tADM_LRU_CTRL        (%p) = %08lx\n \
			",
			adm,
			ADM_LRU_ENTRY(adm), *ADM_LRU_ENTRY(adm),
			ADM_LRU_ENTRY_MASK(adm), *ADM_LRU_ENTRY_MASK(adm),
			ADM_LRU_HIT_CNT_0_3(adm),*ADM_LRU_HIT_CNT_0_3(adm),
			ADM_LRU_HIT_CNT_4_7(adm),*ADM_LRU_HIT_CNT_4_7(adm),
			ADM_LRU_TIMER(adm), *ADM_LRU_TIMER(adm),
			ADM_LRU_PERIOD(adm), *ADM_LRU_PERIOD(adm),
			ADM_LRU_CTRL(adm), *ADM_LRU_CTRL(adm)
			);
		}
	}
	
	if(block & QOS_SCH_BLOCK)
	{
		QOS_PRINTK(QOS_REGS_DUMP, "**********C1K HW QoS SCH Registers**********");

		for(sch = SCH0; sch <= SCH1; sch++) 
		{
			QOS_PRINTK(QOS_REGS_DUMP, \
			"\nScheduler block %d registers:\n \
\tSCH_STATUS_REG    (%p) = %08lx\n \
\tSCH_CTRL_REG      (%p) = %08lx\n \
\tSCH_PKT_QUEUED    (%p) = %08lx\n \
\tSCH_PORT_BYTE_CNT (%p) = %08lx\n \
\tSCH_PORT_PKT_CNT  (%p) = %08lx\n \
\tSCH_PKT_OVERHEAD  (%p) = %08lx\n \
			", 
			sch, 
			SCH_STATUS_REG(sch), *SCH_STATUS_REG(sch), 
			SCH_CTRL_REG(sch), *SCH_CTRL_REG(sch), 
			SCH_PKT_QUEUED(sch), *SCH_PKT_QUEUED(sch), 
			SCH_PORT_BYTE_CNT(sch), *SCH_PORT_BYTE_CNT(sch), 
			SCH_PORT_PKT_CNT(sch), *SCH_PORT_PKT_CNT(sch), 
			SCH_PKT_OVERHEAD(sch), *SCH_PKT_OVERHEAD(sch) 
			);
		}
	}
}


/**
 * qos_init -
 *
 *
 */
static int qos_init(void)
{
	int i;

	QOS_PRINTK(QOS_INFO, "qos_init()\n");

	memset(_adm, 0, sizeof(struct _QOS_ADM) * 2);
	memset(_sch, 0, sizeof(struct _QOS_SCH) * 2);

	/* needed to retrieve block index through IOCTLs calls */
	_adm[0].index = ADM0;
	_adm[1].index = ADM1;
	_sch[0].index = SCH0;
	_sch[1].index = SCH1;

	/* initialize TCAM entries table */
	for(i = 0; i < ADM_TCAM_MAX_ENTRIES; i++) 
	{
		_adm[0].cam_sw_table[i].slot = ADM_TCAM_FREE;
		_adm[1].cam_sw_table[i].slot = ADM_TCAM_FREE;
		_adm[0].cam_hw_table[i].slot = ADM_TCAM_FREE;
		_adm[1].cam_hw_table[i].slot = ADM_TCAM_FREE;
	}

#ifdef C1K_REG_SIM
#define FAKE_EMAC_SZ	0x8180
	if ((FAKE_EMAC0_BASE = kmalloc (FAKE_EMAC_SZ, GFP_KERNEL)) == NULL)
	{
		QOS_PRINTK(QOS_ERR, "cannot allocate memory for fake registers()\n");
		return -1;
	}
	if ((FAKE_EMAC1_BASE = kmalloc (FAKE_EMAC_SZ, GFP_KERNEL)) == NULL)
	{
		QOS_PRINTK(QOS_ERR, "cannot allocate memory for fake registers()\n");
		return -1;
	}
	QOS_PRINTK(QOS_INIT, "FAKE_GEMAC0_BASE @ %p FAKE_GEMAC1_BASE @ %p\n", FAKE_EMAC0_BASE, FAKE_EMAC1_BASE);
	memset(FAKE_EMAC0_BASE, 0, FAKE_EMAC_SZ);
	memset(FAKE_EMAC1_BASE, 0, FAKE_EMAC_SZ);
#endif

	/* Map HW counters */
	_adm[ADM0].counter = (struct _ADM_COUNTER *)(adm_statis(ADM0));
	_adm[ADM1].counter = (struct _ADM_COUNTER *)(adm_statis(ADM1));

	/* The default configuration of the Ethernet admittance block is to run in discard mode, with no
	proportional dropping, dropping only when the receive queue is full. The shaper is disabled, and the
	classifier is disabled, resulting in all traffic being treated as Unmanaged.*/
#if 0
	/* For debufg purpose only, let's admittance accepted errored packets */
	ADM_SET_CONF_ADMITERR(0);
#endif

	return 0;
}

/**
 *  qos_proc_info -
 *
 *
 */
int qos_proc_info_adm(char* page, char** start, off_t off, int count, int *eof, void* data)
{
	struct _QOS_ADM *this_adm = (struct _QOS_ADM *)data;
	struct _ADM_LRU_STATS lru_stats;
	int adm = this_adm->index;
	int tcam, adm_state, len = 0;

	len += sprintf(page+len, "\n");
	len += sprintf(page+len, "Comcerto 1000 HW QoS Stats for ADM %d:\n\n", adm);

	ADM_SET_CTRL_SNAPSHOT_ENABLE(adm);
	
	adm_state = ADM_GET_CTRL_ADM_ENABLE(adm);
	if(adm_state)
		len += sprintf(page+len, "mode %s\n", adm_mode_string[ADM_GET_CONF_MODE(adm)]); /* DISCARD_MODE or FLOWCTRL_MODE */
	else
		len += sprintf(page+len, "mode %s\n", adm_mode_string[2]);/* DISABLED_MODE */
	len += sprintf(page+len, "port bytes %lu\n", _adm[adm].counter->port_byte);
	len += sprintf(page+len, "port packets %lu\n", _adm[adm].counter->port_packet);
	len += sprintf(page+len, "reserved bytes %lu\n", _adm[adm].counter->reserved_byte);
	len += sprintf(page+len, "reserved packets %lu\n", _adm[adm].counter->reserved_packet);
	len += sprintf(page+len, "managed bytes %lu\n", _adm[adm].counter->managed_byte);
	len += sprintf(page+len, "managed packets %lu\n", _adm[adm].counter->managed_packet);
	len += sprintf(page+len, "dropped %lu\n", _adm[adm].counter->packet_dropped);
	len += sprintf(page+len, "dropped upon errors %lu\n", _adm[adm].counter->packet_dropped_error);
	len += sprintf(page+len, "dropped denied %lu\n", _adm[adm].counter->packet_dropped_denied);
	len += sprintf(page+len, "dropped policer0 %lu\n", _adm[adm].counter->packet_dropped_policer0);
	len += sprintf(page+len, "dropped policer1 %lu\n", _adm[adm].counter->packet_dropped_policer1);
	len += sprintf(page+len, "dropped qfull %lu\n", _adm[adm].counter->packet_dropped_qfull);
	len += sprintf(page+len, "dropped shaped %lu\n", _adm[adm].counter->packet_dropped_shaper);
	len += sprintf(page+len, "dropped managed %lu\n", _adm[adm].counter->packet_dropped_managed);
	len += sprintf(page+len, "dropped unmanaged %lu\n", _adm[adm].counter->packet_dropped_unmanaged);

	qos_adm_get_lru_stats(&_adm[adm], &lru_stats);
	len += sprintf(page+len, "LRU TCAM entry: %u (%u)\n", lru_stats.entry, lru_stats.count);
	len += sprintf(page+len, "TCAM hit count:\n");
	for(tcam = 0; tcam < ADM_MAX_TCAM_ENTRIES; tcam++)
		len += sprintf(page+len, "[%u]=%u ", tcam, lru_stats.hit_count[tcam]);

	len += sprintf(page+len, "\n");

	return len;
}

/**
 * qos_proc_info_sch -
 *
 */
int qos_proc_info_sch(char* page, char** start, off_t off, int count, int *eof, void* data)
{
	struct _QOS_SCH *this_sch = (struct _QOS_SCH *)data;
	int sch = this_sch->index;
	int q;
	struct _SCH_COUNTER sch_stats;
	int len = 0;

	len += sprintf(page+len, "\n");
	len += sprintf(page+len, "Comcerto 1000 HW QoS Stats for SCH %d:\n\n", sch);

	memset(&sch_stats, 0, sizeof(struct _SCH_COUNTER));
	SCH_SET_SNAPSHOT(sch);
	qos_sch_get_counters(&_sch[sch], &sch_stats);

	len += sprintf(page+len, "port bytes %lu\n", sch_stats.port_byte);
	len += sprintf(page+len, "port packets %lu\n", sch_stats.port_packet); 
	
	for(q = 0; q < SCH_MAX_QUEUES; q++) 
	{	
		len += sprintf(page+len, "\nqueue %u: ", q);
		len += sprintf(page+len, "bytes %lu ", sch_stats.q_byte[q]);
		len += sprintf(page+len, "packets %lu ", sch_stats.q_packet[q]);
		len += sprintf(page+len, "idle time %lu ", sch_stats.q_idle[q]);
	}

	len += sprintf(page+len, "\n");

	return len;
}

struct file_operations qos_fops = {
	.owner =	THIS_MODULE,
	.open =		qos_open,
	.release =	qos_release,
	.ioctl = 	qos_ioctl,
};


/**
 * qos_module_init -
 *
 */
static int __init qos_module_init(void)
{
	int rc = 0;

	QOS_PRINTK(QOS_INIT, "Initializing Comcerto 1000 QoS Driver version %s\n", QOSCOM_VERSION);

	if(qos_init() < 0)
	{
		QOS_PRINTK(QOS_ERR, "QOS: qos init failed\n");
	
		return -1;
	}

	rc = register_chrdev (QOS_DEVICE_MAJOR_NUM, QOS_DRIVER_NAME, &qos_fops);
	if (rc)
	{
		QOS_PRINTK (QOS_ERR, "unable to register QoS char device (%d %s)", QOS_DEVICE_MAJOR_NUM, QOS_DRIVER_NAME);
		return -1;
	}

	/* Create /proc/qos entry */
	if(proc_mkdir ("qos", 0)) 
	{
		if(proc_mkdir ("qos/wan", 0)) 
		{
			create_proc_read_entry("qos/wan/adm", 0, 0, qos_proc_info_adm, (void*)&_adm[ADM0]); //ADM0
			create_proc_read_entry("qos/wan/sch", 0, 0, qos_proc_info_sch, (void*)&_sch[SCH0]);//SCH0
		}
		if(proc_mkdir ("qos/lan", 0)) 
		{
			create_proc_read_entry("qos/lan/adm", 0, 0, qos_proc_info_adm, (void*)&_adm[ADM1]); //ADM1
			create_proc_read_entry("qos/lan/sch", 0, 0, qos_proc_info_sch, (void*)&_sch[SCH1]);//SCH1
		}
	}

	return 0;
}


/**
 * qos_module_exit -
 *
 */
static void __exit qos_module_exit(void)
{
	char s[30];
	QOS_PRINTK(QOS_INIT, "Unloading Comcerto 1000 QoS Driver\n");

#ifdef C1K_REG_SIM
	kfree(FAKE_EMAC0_BASE);
	kfree(FAKE_EMAC1_BASE);
#endif

	/* Remove /proc/qos entry */
	snprintf(s, 30, "qos/wan/adm");
	remove_proc_entry(s, NULL);

	snprintf(s, 30, "qos/wan/sch");
	remove_proc_entry(s, NULL);

	snprintf(s, 30, "qos/wan");
	remove_proc_entry(s, NULL);

	snprintf(s, 30, "qos/lan/adm");
	remove_proc_entry(s, NULL);

	snprintf(s, 30, "qos/lan/sch");
	remove_proc_entry(s, NULL);

	snprintf(s, 30, "qos/lan");
	remove_proc_entry(s, NULL);

	remove_proc_entry("qos", NULL);

	unregister_chrdev (QOS_DEVICE_MAJOR_NUM, QOS_DRIVER_NAME);
}

module_init(qos_module_init);
module_exit(qos_module_exit);
