#!/bin/sh

# Command:
# /usr/sbin/flush_route_cache.sh [call_from] [device_name] [addr_subnet] [mark2]

# Usage:
# This command will flush route cache which in-interface or out-interface is [device_name],
#   AND src-address or dst-address is in [addr_subnet].
# If [device_name] is NULL, flush route cache of ALL interfaces.

# Parameters:
# [call_from]	: For debug use. To check where call this script.
# [device_name]	: Format is lan-lan1, wan-wan2. Use "0" or "all" if not specified.
# [addr_subnet]	: Format is 192.168.1.0/24, 11.22.33.44/32. Use "0" or "0/0" if not specified.
# [mark2]		: Format is mark2/mask (ex: 0x1a/0xff). Keep empty if not specified.

# Examples:
# /usr/sbin/flush_route_cache.sh DEBUG wan-wan1
#     this command will flush route cache of interface wan1.
# /usr/sbin/flush_route_cache.sh DEBUG 0 11.22.33.44/32
#     this command will flush route cache which src-address or dst-address is 11.22.33.44

# Notice:
# /usr/sbin/flush_route_cache.sh DEBUG
#     this command will flush ALL route cache.
# /usr/sbin/flush_route_cache.sh DEBUG 0 0
#     this command will NOT do any flush action. DO NOT use this!

# Trace and Debug:
# To enable echo in following DBG_PRINT() function.

CALL_FROM=$1
DEVICE_NAME=$2
ADDR_SUBNET=$3
MARK2=$4

#--------------------
uptime=`cat /proc/uptime`
[ "$uptime" ] && uptime=${uptime%%.*} || uptime=0

# Enable echo in this function when debug
DBG_PRINT() {
	[ $uptime -lt 600 ] && return

	#echo "[flush_route_cache.sh][`date +%Y/%m/%d-%H:%M:%S`] $1" > /dev/console
	#[ "$DEVICE_NAME" ] || echo "[`date +%Y/%m/%d-%H:%M:%S`] $1" >> /tmp/dbg_log/flush_route_cache.log
	return
}
#--------------------

#No input value, flush all
DBG_PRINT "Call from $CALL_FROM, device:$DEVICE_NAME subnet:$ADDR_SUBNET mark2:$MARK2"
[ "$CALL_FROM" -a "$DEVICE_NAME" ] || {
	/usr/sbin/ip route flush cache
	exit 0
}

#Get ifindex
if [ "$DEVICE_NAME" == "0" -o "$DEVICE_NAME" == "all" ]; then
	ifidx=0
else
	ifpath="/sys/devices/virtual/net/$DEVICE_NAME"
	[ -d $ifpath ] && ifidx=`cat "$ifpath/ifindex"` || ifidx=0
	[ "$ifidx" ] || ifidx=0
fi

#Get address subnet
ip_addr=${ADDR_SUBNET%%/*}
if [ "$ip_addr" -a "$ip_addr" != "0" ]; then
	[ "$ip_addr" == "$ADDR_SUBNET" ] && addr_len=32 || addr_len=${ADDR_SUBNET##*/}
	[ "$addr_len" -a "$addr_len" != "0" ] || addr_len=32
	ip_addr=`echo $ip_addr |sed 's/\./ /g'`
	addr_32=
	for octet in $ip_addr ; do
		addr_32=$addr_32`printf %02x $octet`
	done
else
	addr_32=0
	addr_len=0
fi

mk2=${MARK2%%/*}
if [ "$mk2" ]; then
	[ "$mk2" = "$MARK2" ] && mk2_mask="ffffffff" || mk2_mask=${MARK2##*/}
	[ "$mk2" = "0x" ] && mk2=0 || mk2=`echo $mk2 |sed s/^0x//g`
	[ -z "$mk2_mask" -o "$mk2_mask" = "0x" ] && mk2_mask=0 || mk2_mask=`echo $mk2_mask |sed s/^0x//g`
else
	mk2=0
	mk2_mask=0
fi

#Final check
DBG_PRINT "Get $DEVICE_NAME ifindex:$ifidx addr:0x$addr_32/$addr_len mark2:$mk2/$mk2_mask"
[ $ifidx -le 0 -a "$addr_32" = "0" -a "$mk2_mask" = "0" ] && exit 0

#Flush route cache
/usr/sbin/ip route flush cache $ifidx-$addr_32-$addr_len-$mk2-$mk2_mask
