/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "MLD_Proxy.h"
#include "MLD_MN.h"
#include "MLD_forward.h"
#include "MLD_membership.h"
#include <sys/time.h>

/*   VARIABLES   */
FILE *log_file;
int debug_lvl = VERBOSE;
int mc_mode = MC_ASM;
char int_lan[16] = "eth0";
int int_lan_id;
int int_adhoc_id;
int nb_tunnel = 1;
int first_index = 0;
char prefixe_tunnel[16];
int fwd_scope = MC_GLOBAL;
int mn_src = TRUE;
int immediate_report_v1 = TRUE;
int entry_validity = ENTRY_VAL;
/** The Forwarding Table */
forward_t *fwd_tab = NULL;
/** The MN Table */
MN_t *mn_tab = NULL;
/** The Membership Table */
membership_t member_tab;

/** Socket used to join multicast groups on behalf of mobile nodes, i.e. on the LAN interface */
int sock_lan;
/** Socket to receive MLD reports from the mobile nodes */
int sock_mld;
/** Socket used to forward multicast data to the mobile nodes */
int sock_fwd;	/* used to forward with PF_PACKET and SOCK_DGRAM (give sockaddr_ll, not ethernet header) */

/* for detection of duplicated packets ... by Guillaume Chelius from INRIA-ARES */
#define SIG_SIZE 4
#define PK_TIMEOUT 2

void clean_duplicate(struct timeval *tv);
int check_duplicate(const u_char *buffer, int size);
u_int16_t udp_check( int count, u_int16_t* addr );

typedef struct _pk_sig_t
{
  char sig[SIG_SIZE]; // Packet signature
  struct timeval tv; // Packet reception date
  struct _pk_sig_t *next;
} pk_sig_t;

pk_sig_t  *pk_sigs;

/* end duplicated packets detection */

void read_conf_file() 
{
	FILE * f;
	char buffer[256], option[32], val[16];
	char lan_addr[INET6_ADDRSTRLEN];

	strcpy( int_adhoc, "eth1");

	/* the default name for the configuration file is MLD_Proxy.cfg */
	if ( (f = fopen("MLD_Proxy.cfg", "r")) != NULL )
	{
		 while ( fgets(buffer, 255,f) != '\0')				/* reads one line */
		 {
			 if((buffer[0] != '#')&&(buffer[0]!='\n'))		/* if the line is an option line */
			 {
				sscanf( buffer, "%s = %s", option, val);	/* a line with an option is like option <sapce> = <space> value */
				if( !strcmp(option,"DEBUG_LVL") )
				{
					debug_lvl = atoi(val);
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"MLD_Proxy is in VERBOSE mode.\n");
					fprintf(log_file,"MLD_Proxy is in debug level %d.\n", debug_lvl);
				}
				else if( !strcmp(option,"MC_MODE") )
				{
					if( !strcmp(val,"ASM") )
						mc_mode = MC_ASM;
					else if( !strcmp(val,"SSM") )
						mc_mode = MC_SSM;
					else if( !strcmp(val,"BOTH") )
						mc_mode = MC_BOTH;
					else
						mc_mode = MC_BOTH;
					
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"MLD_Proxy works in %s multicast mode.\n", val);
					fprintf(log_file,"MLD_Proxy works in %s multicast mode.\n", val);
				}
				else if( !strcmp(option,"INT_LAN") )
				{
					strcpy(int_lan,val);
					if( (int_lan_id = if_nametoindex(int_lan)) == 0 )
					{
						fprintf(stderr, "Cannot get index of LAN interface\n");
						exit(-1);
					}
										
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"LAN interface is %s (id %d)\n", int_lan, int_lan_id);
					fprintf(log_file,"LAN interface is %s (id %d)\n", int_lan, int_lan_id);
					
					if( get_v6_addr( int_lan_id, IPV6_GLOBAL, &lan_global_addr ) == 0 )
					{
						if( (inet_ntop( AF_INET6, &lan_global_addr, lan_addr, INET6_ADDRSTRLEN)) == NULL)
						{	
							fprintf(stderr,"Problem with global address of LAN interface\n");
							fprintf(log_file,"Problem with global address of LAN interface\n");
							exit(-1);
						}
					
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Global address of LAN interface is %s\n", lan_addr);
						fprintf(log_file,"Global address of LAN interface is %s\n", lan_addr);
					}
					else
					{
						fprintf(stderr,"Cannot get global address of LAN interface\n");
						fprintf(log_file,"Cannot get global address of LAN interface\n");
						exit(-1);
					}
					
				}
				else if( !strcmp(option,"INT_ADHOC") )
				{
					strcpy(int_adhoc,val);
					if( (int_adhoc_id = if_nametoindex(int_adhoc)) == 0 )
					{
						fprintf(stderr, "Cannot get index of ADHOC interface\n");
						exit(-1);
					}
										
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"ADHOC interface is %s (id %d)\n", int_adhoc, int_adhoc_id);
					fprintf(log_file,"ADHOC interface is %s (id %d)\n", int_adhoc, int_adhoc_id);
				}
				else if( !strcmp(option, "FWD_SCOPE") )
				{
					if( !strcmp(val,"LINK") )
					{
						fwd_scope = MC_LINK;
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"MLD_Proxy forwards data for all scopes.\n");
						fprintf(log_file,"MLD_Proxy forwards data for all scopes.\n");
					}
					else if( !strcmp(val,"SITE") )
					{
						fwd_scope = MC_SITE;
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"MLD_PROXY forwards data for SITE LOCAL and GLOBAL scopes.\n");
						fprintf(log_file,"MLD_PROXY forwards data for SITE LOCAL and GLOBAL scopes.\n");
					}
					else 
					{
						fwd_scope = MC_GLOBAL;
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"MLD_Proxy forwards data for GLOBAL scopes.\n");
						fprintf(log_file,"MLD_Proxy forwards data for GLOBAL scopes.\n");
					}
				}
				else if( !strcmp(option,"ALLOW_SRC") )
				{
					mn_src = atoi(val);
					
					if(mn_src == FALSE)
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Ad hoc nodes are not allowed be multicast sources.\n");
						fprintf(log_file,"Ad hoc nodes are not allowed be multicast sources.\n");
					}
					else
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"Ad hoc nodes are allowed to be multicast sources.\n");
						fprintf(log_file,"Ad hoc nodes are allowed to be multicast sources.\n");
					}
				}
				else if( !strcmp(option,"IMMEDIATE_REPORT_V1") )
				{
					immediate_report_v1 = atoi(val);
					
					if(immediate_report_v1 == FALSE)
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"MLD_Proxy will not send an extra MLDv1 REPORT after receiving a MLDv2 REPORT.\n");
						fprintf(log_file,"MLD_Proxy will not send an extra MLDv1 REPORT after receiving a MLDv2 REPORT.\n");
					
					}
					else
					{
						if(debug_lvl == VERBOSE)
							fprintf(stderr,"MLD_Proxy will send an extra MLDv1 REPORT after receiving a MLDv2 REPORT.\n");
						fprintf(log_file,"MLD_Proxy will send an extra MLDv1 REPORT after receiving a MLDv2 REPORT.\n");
					}
				}
				else if( !strcmp(option,"ENTRY_VAL") )
				{
					entry_validity = atoi(val);
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"An entry in the forwarding table lasts for %d seconds without any update.\n", entry_validity);
					fprintf(log_file,"An entry in the forwarding table lasts for %d seconds without any update.\n", entry_validity);
				}
			 }
		}
	}
	else
	{
		perror("MLD_Proxy.cfg");
		exit(-1);
	}
	fclose(f);
}

int get_v6_addr( int if_index, int scope, struct in6_addr * v6_addr )
{
	/* this function returns the address (v6_addr) of */
	/* scope (scope) for interface (if_index) */
	/* see #define for scope values */
	
	FILE *f;

	char addr[8][5];
	char devname[20];
	int if_id, plen, addr_scope, dad_stat;

	char s_addr[INET6_ADDRSTRLEN];
	
        if ((f = fopen("/proc/net/if_inet6", "r")) != NULL)
	{
		while (fscanf(f, "%4s%4s%4s%4s%4s%4s%4s%4s %02x %02x %02x %02x %20s\n",
			addr[0], addr[1], addr[2], addr[3], 
			addr[4], addr[5], addr[6], addr[7],
			&if_id, &plen, &addr_scope, &dad_stat, devname) != EOF) 
		{
			if ( (if_id == if_index) && (addr_scope == scope) )
			{
				sprintf(s_addr, "%s:%s:%s:%s:%s:%s:%s:%s", 
					addr[0], addr[1], addr[2], addr[3], 
					addr[4], addr[5], addr[6], addr[7]);
				if (inet_pton(AF_INET6, s_addr, v6_addr) <= 0) 
				{
					fclose(f);
					return -1;
				}
				else 
				{
					fclose(f);
					return 0;
				}
				break;
			}
		}
		fclose(f);
		return -1;
	}
	else 
	{
		perror("/proc/net/if_inet6");
		return -1;
	}
}

/* periodically check the timers */
void check(int sig)
{
	alarm(CHECK_TIMER_PERIOD);
	// printf("Check ...\n");
	check_timers(&fwd_tab);
	check_forward(&fwd_tab,&member_tab);
	fflush(log_file);
}

/**
 * The handler called if the daemon is stopped
 * @param sig The signal identifier
*/
void handler_quit(int sig)
{
	if( sig == SIGINT )
	{
		fprintf(log_file,"MLD_Proxy Stopped...\n\n");
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"MLD_Proxy Stopped...\n\n");
	//	
		free_forwarded_group(&fwd_tab);
		free_MN(&mn_tab);
		free_membership(&member_tab);
		fclose(log_file);
		exit(0);
	}
}
/**
 * The handler called if stat printing is asked
 * @param sig The signal identifier
*/
void handler_stat(int sig)
{
	if( sig == SIGUSR1 )
	{
		fprintf(stderr,"\033[0;34m\n#####   MLD_Proxy Statistics   #####\n\n\033[0m");
		print_stat(fwd_tab, mn_tab);
		fprint_stat(fwd_tab, mn_tab);
		fprintf(stderr,"\n\n");
	}
}


/**
 * Initialisation of the socket on the lan interface
 * @return TRUE if success, FALSE otherwise
 */
int init_sock_lan()
{
	int on = 1;	/* to turn on PF_PACKET */
		
	if( (sock_lan = socket( AF_INET6, SOCK_RAW, IPPROTO_UDP )) == -1)
	{
		fprintf(log_file,"init_sock_lan : Problem with socket creation\n");
		fprintf(stderr,"init_sock_lan : Problem with socket creation\n");
		return FALSE;
	}
	
	if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_PKTINFO, &on, sizeof(on) ) == -1)
	{
		fprintf(log_file,"init_sock_lan : Problem with setsockopt IPV6_PKTINFO\n");
		fprintf(stderr,"init_sock_lan : Problem with setsockopt IPV6_PKTINFO\n");
		return FALSE;
	}
	
	if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_HOPLIMIT, &on, sizeof(on) ) == -1)
	{
		fprintf(log_file,"init_sock_lan : Problem with setsockopt IPV6_HOPLIMIT\n");
		fprintf(stderr,"init_sock_lan : Problem with setsockopt IPV6_HOPLIMIT\n");
		return FALSE;
	}

	return TRUE;
}
/**
 * Initialisation of the socket used to receive MLD Reports
 * @return TRUE if success, FALSE otherwise
 */
int init_sock_mld()
{
	struct ipv6_mreq imr;			/* specify IPv6 Address in setsockopt */
	struct icmp6_filter filter;		/* filter for ICMP6 messages : we only wanna receive MLD messages */
	MN_t * tmp_mn = mn_tab;
	int on = 1;	/* to turn on PF_PACKET */
	int no = 0;
	int offset = 2;
	
	/* socket creation */
	if( (sock_mld = socket( AF_INET6, SOCK_RAW, IPPROTO_ICMPV6 )) == -1)
	{
		fprintf(stderr,"init_sock_mld : Problem with socket creation\n");
		fprintf(log_file,"init_sock_mld : Problem with socket creation\n");
		return FALSE;
	}
	
	/* Turn on PF_PACKET on socket */
	if( setsockopt( sock_mld, IPPROTO_IPV6, IPV6_PKTINFO, &on, sizeof(on) ) == -1)
	{
		fprintf(stderr,"init_sock_mld : Problem with setsockopt IPV6_PKTINFO\n");
		fprintf(log_file,"init_sock_mld : Problem with setsockopt IPV6_PKTINFO\n");
		return FALSE;
	}
	
	if( setsockopt( sock_mld, IPPROTO_IPV6, IPV6_MULTICAST_LOOP, &no, sizeof(int) ) == -1)
	{
		fprintf(stderr,"init_sock_mld : Problem with setsockopt IPV6_MULTICAST_LOOP\n");
		fprintf(log_file,"init_sock_mld : Problem with setsockopt IPV6_MULTICAST_LOOP\n");
		return FALSE;
	}
	
	/* compute checksum when sending icmp6 message */
	if ( setsockopt( sock_mld, IPPROTO_IPV6, IPV6_CHECKSUM, &offset, sizeof(offset)) < 0)
        { 
		fprintf(stderr,"init_sock_mld : Problem with setsockopt IPV6_CHECKSUM\n");
		fprintf(log_file,"init_sock_mld : Problem with setsockopt IPV6_CHECKSUM\n");
		return FALSE;
        }
	
	/* JOIN All MLDv2 Routers Address on tunnel interfaces */
	if( (inet_pton(AF_INET6, "ff02::16", &imr.ipv6mr_multiaddr)) == -1)
	{
		fprintf(stderr,"init_sock_mld : Problem when joining ff02::16\n");
		fprintf(log_file,"init_sock_mld : Problem when joining ff02::16\n");
		return FALSE;
	}	
		
	/* we must join on each interface to the tunnels */
	while(tmp_mn != NULL)
	{
		if( (imr.ipv6mr_interface = tmp_mn->tunnel_id) == 0)
		{
			fprintf(stderr,"init_sock_mld : Problem with ADHOC interface %s\n", tmp_mn->tunnel_name);
			fprintf(log_file,"init_sock_mld : Problem with ADHOC interface %s\n", tmp_mn->tunnel_name);
			return FALSE;
		}
	
		if( setsockopt( sock_mld, IPPROTO_IPV6, IPV6_JOIN_GROUP,(char *)&imr, sizeof(struct ipv6_mreq) ) == -1 )
		{      
			fprintf(stderr,"init_sock_mld : Problem to join ff02::16 on ADHOC if %s\n", tmp_mn->tunnel_name);
			fprintf(log_file,"init_sock_mld : Problem to join fff02::16 on ADHOC if %s\n", tmp_mn->tunnel_name);
			return FALSE;
		}
		tmp_mn = tmp_mn->next;
	}

	/* Set the ICMP filter on the socket to avoid receiving other messages than the MLD ones */
	ICMP6_FILTER_SETBLOCKALL( &filter );							/* all messages are denied */
	ICMP6_FILTER_SETPASS( ICMP6_MEMBERSHIP_QUERY, &filter );		/* we explicitely authorize the desired messages */
	ICMP6_FILTER_SETPASS( ICMP6_MEMBERSHIP_REPORT, &filter );
	ICMP6_FILTER_SETPASS( ICMP6_MEMBERSHIP_REDUCTION, &filter );
	ICMP6_FILTER_SETPASS( ICMPV6_MLD2_REPORT, &filter );
	ICMP6_FILTER_SETPASS( ICMPV6_MLDv2_REPORT, &filter );
	ICMP6_FILTER_SETPASS( ICMP6_SPEC_MSG, &filter );
	
	
	/* apply the filter on the socket */
	if( setsockopt(sock_mld, IPPROTO_ICMPV6, ICMP6_FILTER, &filter, sizeof(filter)) < 0) {
	
		perror("\nSetsockopt(ICMP6_FILTER)");
		exit(-1);
	}

	
	return TRUE;
}
/**
 * Initialisation of the socket used to forward data to the mobile nodes
 * @return TRUE if success, FALSE otherwise
 */
int init_sock_fwd()
{	
	if( (sock_fwd = socket( PF_PACKET, SOCK_DGRAM, htons(ETH_P_IPV6) )) == -1)
	{
		fprintf(stderr,"init_sock_fwd : Problem with socket creation\n");
		fprintf(log_file,"init_sock_fwd : Problem with socket creation\n");
		return FALSE;
	}

	return TRUE;
}


/**
 * Function called if a MLD message is received on sock_mld
 * @return TRUE on succes,FALSE otherwise
 */
int mld_rcv()
{
	char new_buffer[DATA_BUFFER_SIZE],buffer[DATA_BUFFER_SIZE];		/* buffer */
	int length;							/* lenght of received packet */
	char from_addr[INET6_ADDRSTRLEN];	/* address for sender */
	char group_addr[INET6_ADDRSTRLEN];	/* multicast address in MLD msg */
	struct ipv6_mreq grp_data;
	int nb_grp;							/* number of group records in MLDv2 report */
	MN_t * tmp_mn = mn_tab;
	int if_name = 0;		/* when trying to find an if name, says if successful or not*/
	struct in6_addr forbid_ff0e_1;
	time_t cur_time;	/* for logging */
	
	/* Packet info */
	struct in6_pktinfo extra_info;
	struct sockaddr_in6 host_info;
	/* Pointers to headers in buffer */
	struct icmp6_hdr *icmp;
	struct mld6_v1_hdr *mld1;
	struct mld6_v2_hdr *mld2;
	struct mld6_v2_query *query2;
	char * offset;
	int new_size = 0;
	/* ra is the complete HOP-by-HOP router alert option */
	char ra[8] = { IPPROTO_ICMPV6, 0, IPV6_TLV_PADN, 0,
			IPV6_TLV_ROUTERALERT, 2, 0, 0 };
	
	/* this is used to get packet information : dst address and interface */
	struct msghdr msgh;
	struct cmsghdr *cmsgh;
	struct iovec iovector;
	char control[128];
	struct ip6_hdr * ip6h;
	struct sockaddr_ll ll;
		
	ip6h = (struct ip6_hdr *)buffer;
	offset = (buffer + sizeof(struct ip6_hdr) );
		
	memset( &msgh, 0, sizeof(struct msghdr) );
	msgh.msg_iov = &iovector;
	msgh.msg_iovlen = 1;
	msgh.msg_control = control;
	msgh.msg_controllen = 128;
	msgh.msg_name = &host_info;
	msgh.msg_namelen = sizeof(struct sockaddr_in6);
	
	memset( &iovector, 0, sizeof(struct iovec) );
	iovector.iov_base = offset;
	iovector.iov_len = DATA_BUFFER_SIZE;
	
	/* get message */
	if( (length = recvmsg( sock_mld, &msgh, MSG_WAITALL)) <= 0)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"mld_rcv : Problem with recvfrom.\n");
		fprintf(log_file,"mld_rcv : Problem with recvfrom.\n");
		return FALSE;
	}
	
	for( cmsgh = CMSG_FIRSTHDR(&msgh); cmsgh != NULL; cmsgh = CMSG_NXTHDR(&msgh,cmsgh) )
	{
		/* find the PKTINFO in ancillary data and copy it to extra_info */
		if( (cmsgh->cmsg_level == IPPROTO_IPV6) && (cmsgh->cmsg_type == IPV6_PKTINFO) )
		{
			memcpy( &extra_info, (struct in6_pktinfo *)CMSG_DATA(cmsgh), 
			sizeof(struct in6_pktinfo) );
			break;
		}
	}
	
	/* address resolution */
	if( (inet_ntop( AF_INET6, &(host_info.sin6_addr.s6_addr), from_addr, INET6_ADDRSTRLEN)) == NULL)
	{	
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"mld_rcv : Problem with sender address resolution\n");
		fprintf(log_file,"mld_rcv : Problem with sender address resolution\n");
		return FALSE;
	}
	
	/* set pointers */
	icmp = (struct icmp6_hdr *) offset;
	mld1 = (struct mld6_v1_hdr *) (offset);
	mld2 = (struct mld6_v2_hdr *) (offset + sizeof(struct icmp6_hdr) );
	query2 = (struct mld6_v2_query *) (offset);
	
	cur_time = time(NULL);
	fprintf( log_file, "\n--- %s", ctime(&cur_time));
	fprintf( log_file, "Received %d bytes sent by %s ", length, from_addr);
	fprintf( log_file, "(from if %d ", extra_info.ipi6_ifindex );
	
	if(debug_lvl == VERBOSE)
	{
		fprintf( stderr, "\n--- %s", ctime(&cur_time));
		fprintf(stderr,"Received %d bytes sent by %s ", length, from_addr);
		fprintf(stderr,"(from if %d ", extra_info.ipi6_ifindex );
	}
			
	if_name = extra_info.ipi6_ifindex;
	
	if( if_name == int_adhoc_id )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"--> ADHOC)\n");
		fprintf(log_file,"--> ADHOC)\n");
	}
		
	if( if_name == int_lan_id)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"--> LAN)\n");
		fprintf(log_file,"--> LAN)\n");
	}
	
	if( if_name != int_adhoc_id && if_name != int_lan_id )
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"--> ?)\n");
		fprintf(log_file,"--> ?)\n");
	}
	
	if(debug_lvl == VERBOSE)
		fprintf(stderr,"ICMP type = %d, code = %d ", icmp->icmp6_type, icmp->icmp6_code);
	fprintf(log_file,"ICMP type = %d, code = %d ", icmp->icmp6_type, icmp->icmp6_code);
		
	tmp_mn = mn_tab;
	
	switch( icmp->icmp6_type )
	{
		case(ICMP6_MEMBERSHIP_QUERY) :
		{
			if( extra_info.ipi6_ifindex != int_lan_id )
			{
				if(debug_lvl == VERBOSE)
					fprintf(stderr, "\nQuery was received from ADHOC network ... and is therefore not forwarded\n");
				fprintf(log_file, "\nQuery was received from ADHOC network ... and is therefore not forwarded\n");
				break;
			}
		
			inet_ntop( AF_INET6, &(mld1->mld6_addr), group_addr, INET6_ADDRSTRLEN);
			
			/* ip6h->ip6_vfc */
			ip6h->ip6_flow = htonl(0x60000000);
			ip6h->ip6_hlim = 0x01;
			ip6h->ip6_src = host_info.sin6_addr;
			ip6h->ip6_dst = extra_info.ipi6_addr;
		
			if( length == V1_QUERY_SIZE)
			{
				if(debug_lvl == VERBOSE)
				{
					fprintf(stderr,"--> MLD v1 General Query\n");
					fprintf(stderr,"Address (in MLD header) : %s\n", group_addr);
				}
				fprintf(log_file,"--> MLD v1 Query\n");
				fprintf(log_file,"Address (in MLD header) : %s\n", group_addr);
								
				ip6h->ip6_nxt = IPPROTO_HOPOPTS;
				ip6h->ip6_plen = htons(32);
				memset( new_buffer,0,DATA_BUFFER_SIZE);
				memcpy( new_buffer,ip6h,sizeof( struct ip6_hdr));
				memcpy( new_buffer+sizeof( struct ip6_hdr),ra,8);
				memcpy( new_buffer+sizeof( struct ip6_hdr)+8,mld1,sizeof(struct mld6_v1_hdr) );
				new_size = (sizeof(struct mld6_v1_hdr) + sizeof(struct ip6_hdr) +8);
			}
			
			if( length == V2_QUERY_SIZE)
			{
				if(debug_lvl == VERBOSE)
				{
					fprintf(stderr,"--> MLD v2 General Query\n");
					fprintf(stderr,"Address (in MLD header) : %s - Num Src %d\n",group_addr,query2->nb_src);
				}
				fprintf(log_file,"--> MLD v2 Query\n");
				fprintf(log_file,"Address (in MLD header) : %s - Num Src %d\n",group_addr,query2->nb_src);
				
				ip6h->ip6_nxt = IPPROTO_HOPOPTS;
				ip6h->ip6_plen = htons(36);
				memset( new_buffer,0,DATA_BUFFER_SIZE);
				memcpy( new_buffer,ip6h,sizeof( struct ip6_hdr));
				memcpy( new_buffer+sizeof( struct ip6_hdr),ra,8);
				memcpy( new_buffer+sizeof( struct ip6_hdr)+8,query2,sizeof(struct mld6_v2_query) );
				new_size = (sizeof(struct mld6_v2_query) + sizeof(struct ip6_hdr) +8);
			}				
			
			if( length != V1_QUERY_SIZE && length != V2_QUERY_SIZE )
			{
				/*
				if(debug_lvl == VERBOSE)
				{
					fprintf(stderr,"--> MLD v2 Specific Query (not supported yet)\n");
					fprintf(log_file,"--> MLD v2 Specific Query (not supported yet)\n");
				}*/
			
				break;
			}
			
			ll.sll_family = AF_PACKET;
			ll.sll_protocol = htons(ETH_P_IPV6);
			ll.sll_hatype = ARPHRD_TUNNEL6;
			ll.sll_pkttype = PACKET_MULTICAST;
			ll.sll_halen = 6;
	
			ll.sll_addr[0] = 0x33;
			ll.sll_addr[1] = 0x33;
			ll.sll_addr[2] = extra_info.ipi6_addr.in6_u.u6_addr8[12];
			ll.sll_addr[3] = extra_info.ipi6_addr.in6_u.u6_addr8[13];
			ll.sll_addr[4] = extra_info.ipi6_addr.in6_u.u6_addr8[14];
			ll.sll_addr[5] = extra_info.ipi6_addr.in6_u.u6_addr8[15];
			ll.sll_addr[6] = 0x00;
			ll.sll_addr[7] = 0x00;
	
			length = sizeof(struct sockaddr_ll);
			
			if( check_duplicate( (char *)ip6h, (ntohs(ip6h->ip6_plen)+sizeof(struct ip6_hdr)) ) == -1)
				break;
			
			/* backup last query */
			memcpy( last_query, new_buffer, new_size);
			last_query_size = new_size;
			memcpy( &last_query_ll, &ll, sizeof(struct sockaddr_ll));
			
			while( tmp_mn != NULL )
			{
				ll.sll_ifindex = tmp_mn->tunnel_id;
				if( sendto( sock_fwd, new_buffer, new_size, 0, (struct sockaddr *)&ll, length ) == -1 )
				{
					if(debug_lvl == VERBOSE)
						fprintf(stderr,"Problem while forwarding Query to %d\n",tmp_mn->tunnel_id);
					fprintf(log_file,"Problem while forwarding Query to %d\n",tmp_mn->tunnel_id);	
				}
				
				tmp_mn = tmp_mn->next;
			}
			break;
		}
			
		case(ICMP6_MEMBERSHIP_REPORT) :
		{
			if( member_tab.oif_id == extra_info.ipi6_ifindex)
			{
				/* we do not use MLD message sent by the HA */
				return FALSE;
			}	
			inet_ntop( AF_INET6, &(mld1->mld6_addr), group_addr, INET6_ADDRSTRLEN);
			if(debug_lvl == VERBOSE)
			{
				fprintf(stderr,"--> MLD v1 Report\n");
				fprintf(stderr,"Address : %s\n", group_addr);
			}
			fprintf(log_file,"--> MLD v1 Report\n");
			fprintf(log_file,"Address : %s\n", group_addr);
			
			/* verify scope */
			
			/* we don't allow ff0e::1 as it is used by Qolyester and the autoconfiguration daemon */
			inet_pton(AF_INET6, "ff0e::1", &forbid_ff0e_1);
			if( IN6_ARE_ADDR_EQUAL( &(mld1->mld6_addr), &forbid_ff0e_1) )
			{
				// fprintf(stderr, "Special group %s is forbidden\n", group_addr);
				break;
			}
			
			if( IN6_IS_ADDR_MC_GLOBAL(&(mld1->mld6_addr)) && 
				((fwd_scope == MC_LINK) || (fwd_scope == MC_LINK) || (fwd_scope == MC_GLOBAL)) )
			{
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"Scope VALID for group %s",group_addr);
				fprintf(log_file,"Scope VALID for group %s",group_addr);
			}
			else if( IN6_IS_ADDR_MC_SITELOCAL(&(mld1->mld6_addr)) && 
				((fwd_scope == MC_LINK) || (fwd_scope == MC_LINK)) )
			{
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"Scope VALID for group %s",group_addr);
				fprintf(log_file,"Scope VALID for group %s",group_addr);
			}
			else if( IN6_IS_ADDR_MC_LINKLOCAL(&(mld1->mld6_addr)) && (fwd_scope == MC_LINK) )
			{
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"Scope VALID for group %s",group_addr);
				fprintf(log_file,"Scope VALID for group %s",group_addr);
			}
			else
			{
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"Scope INVALID for group %s\n",group_addr);
				fprintf(log_file,"Scope INVALID for group %s\n",group_addr);
				break;
			}
			if( debug_lvl == VERBOSE)
				fprintf(stderr," --> JOIN (*,%s)\n", group_addr);
			fprintf(log_file," --> JOIN (*,%s)\n", group_addr);
								
			if( is_interface_set(fwd_tab,mld1->mld6_addr,extra_info.ipi6_ifindex) == TRUE)
			{
				/* if the Group is already forwarded, just restart the corresponding timer */
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"(*,%s) already forwarding to interface %d -> restart timer\n",group_addr, extra_info.ipi6_ifindex);
				fprintf(log_file,"(*,%s) already forwarding to interface %d -> restart timer\n",group_addr, extra_info.ipi6_ifindex);
				
				reset_timer(&fwd_tab,mld1->mld6_addr,extra_info.ipi6_ifindex);
				break;
			}
			
			if(immediate_report_v1 == TRUE)
				send_v1_report( mld1->mld6_addr, member_tab.oif_id );
						
			grp_data.ipv6mr_multiaddr = mld1->mld6_addr;
					
			/* add the group and the interface in the tables */
			if( is_group_joined_by_addr(member_tab,mld1->mld6_addr) == FALSE)
			{
				grp_data.ipv6mr_interface = member_tab.oif_id;
				if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_JOIN_GROUP,
					(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Unable to join (*,%s) on %d.\n",group_addr,member_tab.oif_id);
					fprintf(log_file,"Unable to join (*,%s) on %d.\n",group_addr,member_tab.oif_id);
					
					break;
				}
				/* if the group isn't forwarded yet, adding in the membership table */
				add_mult_group_by_addr(&member_tab,mld1->mld6_addr);
			}
			if( is_group_forwarded(fwd_tab,mld1->mld6_addr) == FALSE)
			{
				/* if the group isn't forwarded yet, adding in the forwarding table */
				add_forwarded_group(&fwd_tab,mld1->mld6_addr);
			}
						
			if( mn_src == TRUE )
			{
				/* to receive data sent by the MN, join group on tunnel interface */
				grp_data.ipv6mr_interface = extra_info.ipi6_ifindex;
					
				if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_JOIN_GROUP,
					(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Unable to join (*,%s) on if %d.\n",group_addr,extra_info.ipi6_ifindex);
					fprintf(log_file,"Unable to join (*,%s) on if %d.\n",group_addr,extra_info.ipi6_ifindex);
					
					break;
				}
			}
			/* add the fowarding on the interface for the group */
			add_interface(&fwd_tab,mld1->mld6_addr,extra_info.ipi6_ifindex);			
			break;
		}
			
		case(ICMP6_MEMBERSHIP_REDUCTION) :
		{
			if( member_tab.oif_id == extra_info.ipi6_ifindex)
			{
				/* we do not use MLD message sent by the HA */
				return FALSE;
			}
			inet_ntop( AF_INET6, &(mld1->mld6_addr), group_addr, INET6_ADDRSTRLEN);
			if(debug_lvl == VERBOSE)
			{
				fprintf(stderr,"--> MLD v1 Done\n");
				fprintf(stderr,"Address : %s\n", group_addr);
			}
			fprintf(log_file,"--> MLD v1 Done\n");
			fprintf(log_file,"Address : %s\n", group_addr);
			
			if( is_interface_set(fwd_tab,mld1->mld6_addr,extra_info.ipi6_ifindex) == FALSE)
			{
				/* if the Group is not forwarded, ignor message */
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"(*,%s) not forwarded to interface %d\n",group_addr, extra_info.ipi6_ifindex);
				fprintf(log_file,"(*,%s) not forwarded to interface %d\n",group_addr, extra_info.ipi6_ifindex);
				
				break;
			}
						
			grp_data.ipv6mr_multiaddr = mld1->mld6_addr;
			grp_data.ipv6mr_interface = extra_info.ipi6_ifindex;
					
			if( setsockopt( sock_lan, IPPROTO_IPV6,IPV6_LEAVE_GROUP,
				(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
			{
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"Unable to leave (*,%s) on if %d.\n",group_addr,extra_info.ipi6_ifindex);
				fprintf(log_file,"Unable to leave (*,%s) on if %d.\n",group_addr,extra_info.ipi6_ifindex);
				
				break;
			}
						
			/* del the group and the interface in the tables */
			del_interface(&fwd_tab,mld1->mld6_addr,extra_info.ipi6_ifindex);
			if( nb_interface(fwd_tab,mld1->mld6_addr) == 0 )
			{
				/* if the group isn't forwarded to at least one interface, delete in the forwarding table and memebership */
				del_forwarded_group(&fwd_tab,mld1->mld6_addr);
				del_mult_group_by_addr(&member_tab,mld1->mld6_addr);
				grp_data.ipv6mr_interface = member_tab.oif_id;
						
				if( setsockopt( sock_lan, IPPROTO_IPV6,IPV6_LEAVE_GROUP,
					(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Unable to leave (*,%s) on %d.\n",group_addr,member_tab.oif_id);
					fprintf(log_file,"Unable to leave (*,%s) on %d.\n",group_addr,member_tab.oif_id);
					
					break;
				}
			}
			break;
		}
		
		/* we accept report with code 143 and 206 */	
		case(ICMPV6_MLDv2_REPORT):
		case(ICMPV6_MLD2_REPORT) :
		{
			if( member_tab.oif_id == extra_info.ipi6_ifindex)
			{
				/* we do not treat MLD message sent by the proxy itself */
				return FALSE;
			}
			nb_grp = ntohl(icmp->icmp6_dataun.icmp6_un_data32[0]);
			if(debug_lvl == VERBOSE)
				fprintf(stderr,"--> MLD v2 Report (Num Group Records = %d)\n", nb_grp);
			fprintf(log_file,"--> MLD v2 Report (Num Group Records = %d)\n", nb_grp);
			
			
			/* for each group record */
			while( nb_grp != 0 )
			{
				inet_ntop( AF_INET6, &(mld2->mld6_addr), group_addr, INET6_ADDRSTRLEN);
				if( debug_lvl == VERBOSE)
					fprintf(stderr,"Mode : %d - Num Src : %d\n", mld2->mode, mld2->num_src);
				fprintf(log_file,"Mode : %d - Num Src : %d\n", mld2->mode, mld2->num_src);
								
				/* we don't allow ff0e::1 as it is used by Qolyester and the autoconfiguration daemon */
				inet_pton(AF_INET6, "ff0e::1", &forbid_ff0e_1);
				if( IN6_ARE_ADDR_EQUAL( &(mld2->mld6_addr), &forbid_ff0e_1) )
				{
					// fprintf(stderr, "Special group %s is forbidden\n", group_addr);
					break;
				}
				
				/* verify scope */
				if( IN6_IS_ADDR_MC_GLOBAL(&(mld2->mld6_addr)) && 
					((fwd_scope == MC_LINK) || (fwd_scope == MC_SITE) || (fwd_scope == MC_GLOBAL)) )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Scope VALID for group %s",group_addr);
					fprintf(log_file,"Scope VALID for group %s",group_addr);
				}
				else if( IN6_IS_ADDR_MC_SITELOCAL(&(mld2->mld6_addr)) && 
					((fwd_scope == MC_LINK) || (fwd_scope == MC_SITE)) )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Scope VALID for group %s",group_addr);
					fprintf(log_file,"Scope VALID for group %s",group_addr);
				}
				else if( IN6_IS_ADDR_MC_LINKLOCAL(&(mld2->mld6_addr)) && (fwd_scope == MC_LINK) )
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Scope VALID for group %s",group_addr);
					fprintf(log_file,"Scope VALID for group %s",group_addr);
				}
				else
				{
					if( debug_lvl == VERBOSE)
						fprintf(stderr,"Scope INVALID for group %s\n",group_addr);
					fprintf(log_file,"Scope INVALID for group %s\n",group_addr);
					
					mld2++;
					nb_grp--;
					continue;
				}
				
				if( mld2->num_src == 0 )
				{
					/* ASM */
					if( (mld2->mode == MLD2_MODE_IS_EXCLUDE) || (mld2->mode == MLD2_CHANGE_TO_EXCLUDE) )
					{
						/* EXCLUDE{} for this group : MN joins group */
						if( debug_lvl == VERBOSE)
							fprintf(stderr," --> JOIN (*,%s)\n", group_addr);
						fprintf(log_file," --> JOIN (*,%s)\n", group_addr);
												
						if( is_interface_set(fwd_tab,mld2->mld6_addr,extra_info.ipi6_ifindex) == TRUE)
						{
							/* if the Group is already forwarded, just restart the corresponding timer */
							if( debug_lvl == VERBOSE)
								fprintf(stderr,"(*,%s) already forwarded to interface %d -> restart timer\n",group_addr, extra_info.ipi6_ifindex);
							fprintf(log_file,"(*,%s) already forwarded to interface %d -> restart timer\n",group_addr, extra_info.ipi6_ifindex);
							
							reset_timer(&fwd_tab,mld2->mld6_addr,extra_info.ipi6_ifindex);
							mld2++;
							nb_grp--;
							continue;
						}
						
						if(immediate_report_v1 == TRUE)
							send_v1_report( mld2->mld6_addr, member_tab.oif_id );
						
						grp_data.ipv6mr_multiaddr = mld2->mld6_addr;
					
						/* add the group and the interface in the tables */
						if( is_group_joined_by_addr(member_tab,mld2->mld6_addr) == FALSE)
						{
							grp_data.ipv6mr_interface = member_tab.oif_id;
							if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_JOIN_GROUP,
								(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
							{
								if( debug_lvl == VERBOSE)
									fprintf(stderr,"Unable to join (*,%s) on %d.\n",group_addr,member_tab.oif_id);
								fprintf(log_file,"Unable to join (*,%s) on %d.\n",group_addr,member_tab.oif_id);
								
								mld2++;
								nb_grp--;
								continue;
							}
							/* if the group isn't forwarded yet, adding in the membership table */
							add_mult_group_by_addr(&member_tab,mld2->mld6_addr);
						}
						if( is_group_forwarded(fwd_tab,mld2->mld6_addr) == FALSE)
						{
							/* if the group isn't forwarded yet, adding in the forwarding table */
							add_forwarded_group(&fwd_tab,mld2->mld6_addr);
						}
						
						if( mn_src == TRUE )
						{
							/* to receive data sent by the MN, join group on tunnel interface */
							grp_data.ipv6mr_interface = extra_info.ipi6_ifindex;
					
							if( setsockopt( sock_lan, IPPROTO_IPV6, IPV6_JOIN_GROUP,
								(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
							{
								if( debug_lvl == VERBOSE)
									fprintf(stderr,"Unable to join (*,%s) on %d.\n",group_addr,extra_info.ipi6_ifindex);
								fprintf(log_file,"Unable to join (*,%s) on %d.\n",group_addr,extra_info.ipi6_ifindex);
								
								mld2++;
								nb_grp--;
								continue;
							}
						}
						/* add the fowarding on the interface for the group */
						add_interface(&fwd_tab,mld2->mld6_addr,extra_info.ipi6_ifindex);
						
					}
					else if( (mld2->mode == MLD2_MODE_IS_INCLUDE) || (mld2->mode == MLD2_CHANGE_TO_INCLUDE) )
					{
						/* INCLUDE{} for this group : MN leaves the group */
						if( debug_lvl == VERBOSE)
							fprintf(stderr," --> DONE (*,%s)\n", group_addr);
						fprintf(log_file," --> DONE (*,%s)\n", group_addr);
												
						if( is_interface_set(fwd_tab,mld2->mld6_addr,extra_info.ipi6_ifindex) == FALSE)
						{
							/* if the Group is not forwarded, ignore message */
							if( debug_lvl == VERBOSE)
								fprintf(stderr,"(*,%s) not forwarded to interface %d\n",group_addr, extra_info.ipi6_ifindex);
							fprintf(log_file,"(*,%s) not forwarded to interface %d\n",group_addr, extra_info.ipi6_ifindex);
							
							mld2++;
							nb_grp--;
							continue;
						}
						
						/* unsubscribe on ADHOC if */
						
						grp_data.ipv6mr_multiaddr = mld2->mld6_addr;
						grp_data.ipv6mr_interface = extra_info.ipi6_ifindex;
					
						if( setsockopt( sock_lan, IPPROTO_IPV6,IPV6_LEAVE_GROUP,
							(char *)&grp_data, sizeof(struct ipv6_mreq) ) == -1 )
						{
							if( debug_lvl == VERBOSE)
								fprintf(stderr,"-Unable to leave (*,%s) on %d.\n",group_addr,extra_info.ipi6_ifindex);
							fprintf(log_file,"-Unable to leave (*,%s) on %d.\n",group_addr,extra_info.ipi6_ifindex);
							
							mld2++;
							nb_grp--;
							continue;
						}
						
						/* del the group and the interface in the tables */
						del_interface(&fwd_tab,mld2->mld6_addr,extra_info.ipi6_ifindex);
						if( nb_interface(fwd_tab,mld2->mld6_addr) == 0 )
						{
							/* if the group isn't forwarded to at least one interface, delete in the forwarding table and memebership */
							del_forwarded_group(&fwd_tab,mld2->mld6_addr);
							del_mult_group_by_addr(&member_tab,mld2->mld6_addr);
							grp_data.ipv6mr_interface = member_tab.oif_id;
						
							/* LEAVE_GROUP is done in del_forwarded_group */
						
						}
					}
				}
				else
				{
					/* ToDO : switch case ASM or SSM, FILTERING... */
				}
				mld2++;
				nb_grp--;
			}
			
			break;
		}
		
		case(ICMP6_SPEC_MSG):
		{
			/* don't treat if already received */
			
			icmp->icmp6_cksum = 0;
			if( check_duplicate( (char *)icmp, (sizeof(struct icmp6_hdr) + sizeof(struct in6_addr)) ) == -1)
			{
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"--> DUP !!!\n");
				fprintf(log_file,"--> DUP !!!\n");
				
				break;
			}
					
			if( int_lan_id == extra_info.ipi6_ifindex)
			{
				/* if msg received from LAN --> error ! */
				return FALSE;
			}
			
			char mesg[8];
			
						
			if( icmp->icmp6_code == ICMP6_SPEC_REPORT)
				strcpy( mesg, "REPORT");
			if( icmp->icmp6_code == ICMP6_SPEC_QUERY)
				strcpy( mesg, "QUERY");
			
			struct in6_addr *msg = (struct in6_addr *) (offset + sizeof(struct icmp6_hdr) );
			inet_ntop( AF_INET6, msg, group_addr, INET6_ADDRSTRLEN);
						
			if(debug_lvl == VERBOSE)
				fprintf(stderr,"--> SPEC MSG (%s for %s)\n", mesg, group_addr);
			fprintf(log_file,"--> SPEC MSG (%s for %s)\n", mesg, group_addr);
		
			struct sockaddr_in6 sa6;
			memset(&sa6, 0, sizeof(struct sockaddr_in6));
			if ((inet_pton(AF_INET6, "ff02::16", &sa6.sin6_addr)) == -1)
			{
				if(debug_lvl == VERBOSE)
					fprintf(stderr,"SPEC_MSG : Problem with address\n");
				fprintf(log_file,"SPEC_MSG : Problem with address\n");
				return FALSE;
			}
			sa6.sin6_scope_id = int_adhoc_id;
			sa6.sin6_family = AF_INET6;
		
			/* resend twice to avoid packet loss */
		
			if( sendto( sock_mld, icmp, sizeof(struct icmp6_hdr) + sizeof (struct in6_addr), 0, 
				(struct sockaddr *)&sa6, sizeof(struct sockaddr_in6 )) == -1 )
				perror("Problem while forwarding special Query msg");
		
			if( sendto( sock_mld, icmp, sizeof(struct icmp6_hdr) + sizeof (struct in6_addr), 0, 
				(struct sockaddr *)&sa6, sizeof(struct sockaddr_in6 )) == -1 )
				perror("Problem while forwarding special Query msg");
		
			break;
		}
		
		default :
		{
			printf("\n");
			break;
		}
	}		

	return TRUE;
}

/**
 * Function called if multicast data are received on sock_lan 
 * @return TRUE on success, FALSE otherwise
 */
 
int fwd_rcv()
{
	int length;
	char buffer[DATA_BUFFER_SIZE];			/* buffer to receive data */
	char grp_addr[INET6_ADDRSTRLEN];	/* address for multicast group */
	char src_addr[INET6_ADDRSTRLEN];	/* address for correspondent */
	struct in6_pktinfo extra_info;
	struct sockaddr_in6 host_info;
	iface_t * tmp_iface = NULL;
	forward_t * tmp_fwd = NULL;
	/* used to join group on interface */
	struct udphdr * udp_hdr;		/* UDP header of received data */
	char * offset;
	int size;
	struct in6_addr forbid_ff0e_1;		/* get rid of autoconf messages */
	
	/* this is used to get packet information : dst address and interface */
	struct msghdr msgh;
	struct cmsghdr *cmsgh;
	struct iovec iovector;
	char control[128];
	struct ip6_hdr * ip6h;
	struct sockaddr_ll ll;
	
	int ipv6_hop_limit;
	struct in6_addr nat_v6;
	u_int8_t pseudo_header[2048];
	char udp_nxt_hdr = 17;
	u_int16_t old_check;
			
	ip6h = (struct ip6_hdr *)buffer;
	offset = (buffer + sizeof(struct ip6_hdr) );
	memset( &msgh, 0, sizeof(struct msghdr) );
	
	msgh.msg_iov = &iovector;
	msgh.msg_iovlen = 1;
	msgh.msg_control = control;
	msgh.msg_controllen = 128;
	msgh.msg_name = &host_info;
	msgh.msg_namelen = sizeof(struct sockaddr_in6);
	
	memset( &iovector, 0, sizeof(struct iovec) );
	
	iovector.iov_base = offset;
	iovector.iov_len = DATA_BUFFER_SIZE;

	/* get the message */
	if( (length = recvmsg( sock_lan, &msgh, MSG_WAITALL)) <= 0)
	{
		perror("\n(FWD)Problem with recvmsg");
		return FALSE;
	}
		
	for( cmsgh = CMSG_FIRSTHDR(&msgh); cmsgh != NULL; cmsgh = CMSG_NXTHDR(&msgh,cmsgh) )
	{
		/* find the PKT_INFO in ancillary data and copy it in struct extra_info */
	
		if( (cmsgh->cmsg_level == IPPROTO_IPV6) && (cmsgh->cmsg_type == IPV6_PKTINFO) )
			memcpy( &extra_info, (struct in6_pktinfo *)CMSG_DATA(cmsgh), sizeof(struct in6_pktinfo) );
		
		/* find the HOP count of received packet */
		if( (cmsgh->cmsg_level == IPPROTO_IPV6) && (cmsgh->cmsg_type == IPV6_HOPLIMIT) )
			memcpy( &ipv6_hop_limit, (int *)CMSG_DATA(cmsgh), sizeof(int) );
	}
	
	/* get dest address and sender address for logging */
	inet_ntop( AF_INET6, &(extra_info.ipi6_addr), grp_addr, INET6_ADDRSTRLEN);
	inet_ntop( AF_INET6, &(host_info.sin6_addr.s6_addr), src_addr, INET6_ADDRSTRLEN);
	
	/* we don't allow ff0e::1 as it is used by Qolyester and the autoconfiguration daemon */
	inet_pton(AF_INET6, "ff0e::1", &forbid_ff0e_1);
	if( IN6_ARE_ADDR_EQUAL( &(extra_info.ipi6_addr), &forbid_ff0e_1) )
	{
		// fprintf(stderr, "Special group ff0e::1 is forbidden\n");
		return FALSE;
	}
	
	if( ipv6_hop_limit == 1 )
	{
		if( debug_lvl == VERBOSE)
			fprintf( stderr, "Hop limit = 1 ... packet not forwarded (grp %s)\n", grp_addr);
		fprintf( log_file, "Hop limit = 1 ... packet not forwarded (grp %s)\n", grp_addr);
		return FALSE;
	}
	
	udp_hdr = (struct udphdr *)offset;
	
	/* !! UDP length includes (size of UDP header + payload) */
	ip6h->ip6_flow = htonl(0x60000000);
	ip6h->ip6_plen = udp_hdr->len;
	ip6h->ip6_nxt = IPPROTO_UDP;
	ip6h->ip6_hlim = 0x01;		/* leave hop limit to 1 for duplicate detection */
	ip6h->ip6_src = host_info.sin6_addr;
	ip6h->ip6_dst = extra_info.ipi6_addr;
	
	/* version with PF_PACKET */
	ll.sll_family = AF_PACKET;
	ll.sll_protocol = htons(ETH_P_IPV6);
	ll.sll_hatype = ARPHRD_TUNNEL6;
	ll.sll_pkttype = PACKET_MULTICAST;
	ll.sll_halen = 6;
	
	ll.sll_addr[0] = 0x33;
	ll.sll_addr[1] = 0x33;
	ll.sll_addr[2] = extra_info.ipi6_addr.in6_u.u6_addr8[12];
	ll.sll_addr[3] = extra_info.ipi6_addr.in6_u.u6_addr8[13];
	ll.sll_addr[4] = extra_info.ipi6_addr.in6_u.u6_addr8[14];
	ll.sll_addr[5] = extra_info.ipi6_addr.in6_u.u6_addr8[15];
	ll.sll_addr[6] = 0x00;
	ll.sll_addr[7] = 0x00;
	
	size = ( ntohs(udp_hdr->len) + sizeof(struct ip6_hdr) );
	length = sizeof(struct sockaddr_ll);
	
	if( check_duplicate( buffer, (ntohs(ip6h->ip6_plen)+sizeof(struct ip6_hdr)) ) == -1)
		return FALSE;
		
	/* put real value for hop limit before forwarding */
	ip6h->ip6_hlim = (char)(ipv6_hop_limit - 1);
		
	/* if the data is sent by a MN */
	if(  is_MN_by_id( mn_tab, extra_info.ipi6_ifindex) == TRUE )
		if( mn_src == TRUE )
		{
			/* translate source address
		
			if( (inet_pton( AF_INET6, prefix_lan, &nat_v6 )) == -1 )
			{
				if( debug_lvl == VERBOSE)
					fprintf(stderr, "Problem with NAT when source in adhoc network\n");
				fprintf(log_file, "Problem with NAT when source in adhoc network\n");
				return FALSE;
			} */
			
			nat_v6 = lan_global_addr;
			memcpy( ((char*)(&nat_v6))+8, ((char*)(&host_info.sin6_addr))+8, 8);
			ip6h->ip6_src = nat_v6;
			
			// fprintf(stderr, "UDP_LEN : %d, UDP_CHECK : %x\n", ntohs(udp_hdr->len), ntohs(udp_hdr->check));
			
			/* set to zero before real calculation */
			old_check = udp_hdr->check;
			udp_hdr->check = 0;
						
			/* build IPv6 pseudo-header for checksum calculation */
			
			bzero( pseudo_header, 2048 );
			memcpy( pseudo_header, &nat_v6, 16 );
			//memcpy( pseudo_header, &host_info.sin6_addr, 16 );
			memcpy( pseudo_header+16, &extra_info.ipi6_addr, 16 );
			memcpy( pseudo_header+32, &udp_hdr->len, 2);
			memset( pseudo_header+39, udp_nxt_hdr, 1 );
			memcpy( pseudo_header+40, udp_hdr, ntohs(udp_hdr->len) );
						
			udp_hdr->check = udp_check( ntohs(udp_hdr->len)+40, (u_int16_t*)pseudo_header);	
		
			/* Forward to LAN interface */
			ll.sll_ifindex = member_tab.oif_id;
			if( sendto( sock_fwd, buffer, size, 0, (struct sockaddr *)&ll, length ) == -1 )
			{
				perror("(FWD)Problem with write on LAN interface");
				return FALSE;
			}
			/* inc stats for MN */
			inc_mn_sent( &mn_tab, extra_info.ipi6_ifindex);
			/* restore real checksum */
			udp_hdr->check = old_check;
		}
		
	/* only FWD data if group is forwarded ie if joined in member_tab */
	if( (is_group_joined_by_addr(member_tab, extra_info.ipi6_addr) == FALSE)
		&& (is_group_joined_SSM_by_addr(member_tab, host_info.sin6_addr, extra_info.ipi6_addr ) == FALSE) )
	{
		return FALSE;
	}
	
	ip6h->ip6_src = host_info.sin6_addr;
		
	/* forward the data to the interfaces which have joined the ASM or the SSM group */
	if( is_group_forwarded_SSM(fwd_tab, host_info.sin6_addr, extra_info.ipi6_addr) == TRUE)
	{
		if( (tmp_fwd = get_forwarded_group_SSM(fwd_tab, host_info.sin6_addr, extra_info.ipi6_addr)) == NULL)
			return FALSE;
		tmp_iface = tmp_fwd->interfaces;
		while(tmp_iface != NULL)
		{
			ll.sll_ifindex = tmp_iface->iface_id;
			if( sendto( sock_fwd, buffer, size, 0, (struct sockaddr *)&ll, length ) == -1 )
			{
				perror("(FWD)Problem with write");
				return FALSE;
			}

			/* inc stats */
			inc_fwd_stat_SSM(&fwd_tab, host_info.sin6_addr, extra_info.ipi6_addr, tmp_iface->iface_id);
			inc_mn_fwded(&mn_tab, tmp_iface->iface_id);
			tmp_iface = tmp_iface->next;
		}
	}
	else if( is_group_forwarded(fwd_tab, extra_info.ipi6_addr) == TRUE)
	{
		if( (tmp_fwd = get_forwarded_group(fwd_tab, extra_info.ipi6_addr)) == NULL)
			return FALSE;
		tmp_iface = tmp_fwd->interfaces;
		while(tmp_iface != NULL)
		{
			ll.sll_ifindex = tmp_iface->iface_id;
			
			if( sendto( sock_fwd, buffer, size, 0, (struct sockaddr *)&ll, length ) == -1 )
			{
				perror("(FWD)Problem with write");
				return FALSE;
			}
			
			/* inc stats */
			inc_fwd_stat(&fwd_tab, extra_info.ipi6_addr, tmp_iface->iface_id);
			inc_mn_fwded(&mn_tab, tmp_iface->iface_id);
			tmp_iface = tmp_iface->next;
		}
	}
	return TRUE;
}



u_int16_t udp_check( int count, u_int16_t* addr)
{	
	u_int32_t sum = 0;
	
        while( count > 1 )  {
           /*  This is the inner loop */
               sum += *(addr++);
               count -= 2;
       }

           /*  Add left-over byte, if any */
       if( count > 0 )
               sum += * (unsigned char *) addr;

           /*  Fold 32-bit sum to 16 bits */
	   
	while (sum>>16)
           sum = (sum & 0xffff) + (sum >> 16);

	// fprintf(stderr,"%x, %x\n", sum, htons(~sum) );

       return (~sum);
}

int send_v1_report( struct in6_addr grp_addr, int oif )
{
	struct mld6_v1_hdr mld;
	int offset = 2;
	int length;
	int sockfd;
	
	struct sockaddr_in6 src;
	struct sockaddr_in6 dst;
	
	struct msghdr msgh;
	struct cmsghdr *cmsgh;
	struct iovec iovector;
	char control[128];
	
	/* ra is the complete HOP-by-HOP router alert option */
	
	char ra[8] = { IPPROTO_ICMPV6, 0, IPV6_TLV_PADN, 0,
			IPV6_TLV_ROUTERALERT, 2, 0, 0 };
	
	/* an ICMP socket for the home agent LAN interface */
	
	sockfd = socket( AF_INET6, SOCK_RAW, IPPROTO_ICMPV6);
	
	memset( &msgh, 0, sizeof(struct msghdr) );
	
	msgh.msg_iov = &iovector;
	msgh.msg_iovlen = 1;
	msgh.msg_control = control;
	msgh.msg_controllen = 128;
	msgh.msg_name = &dst;
	msgh.msg_namelen = sizeof(struct sockaddr_in6);
	
	memset( &iovector, 0, sizeof(struct iovec) );
	
	iovector.iov_base = &mld;
	iovector.iov_len = sizeof(struct mld6_v1_hdr);
		
	length = sizeof(struct sockaddr_in6);
	
	/* the MLD v1 report message */
	
	mld.mld_icmp6_hdr.icmp6_type = ICMP6_MEMBERSHIP_REPORT;
	mld.mld_icmp6_hdr.icmp6_code = 0;
	mld.mld_icmp6_hdr.icmp6_cksum = 0;
	mld.mld_icmp6_hdr.icmp6_dataun.icmp6_un_data32[0] = 0;
	
	mld.mld6_addr = grp_addr;
	
	/* ask kernel to compute ICMP checksum */
	
	if (setsockopt(sockfd, IPPROTO_IPV6,IPV6_CHECKSUM,&offset,sizeof(offset)) < 0)
        {  
		perror ("Couldn't setsockopt IPV6_CHECKSUM");
		return(-1);
        }
	
	/* NEW VERSION : this is done in main function with get_v6_addr() */
	
	src.sin6_family = AF_INET6;
	src.sin6_port = htons(0);
	src.sin6_addr = member_tab.oif_addr;
	src.sin6_scope_id = oif;
	
	if ( bind(sockfd, (struct sockaddr *)&src, length) == -1)
	{
		perror("Could not bind");
		return -1;
	}
	
	/* dest address is the group address */
	
	dst.sin6_family = AF_INET6;
	dst.sin6_addr = grp_addr;
	dst.sin6_port = htons(0);
	
	/* set ancillary data options + copy hop-by-hop option */
	
	cmsgh = CMSG_FIRSTHDR(&msgh);
	cmsgh->cmsg_len = CMSG_LEN(8);
	cmsgh->cmsg_level = IPPROTO_IPV6;
	cmsgh->cmsg_type = IPV6_HOPOPTS;
	
	memcpy( (void *)CMSG_DATA(cmsgh), (void *)ra, 8 );
	
	msgh.msg_controllen = cmsgh->cmsg_len;
	
	if( (sendmsg( sockfd, &msgh, 0 )) <= 0)
	{
		perror("(V1)Problem with sendmsg");
		return -1;
	}
	
	close(sockfd);
	return 0;
}

/**
 * Write the PID in a file named MLD_Proxy.pid in /var/run/
 * @param pid The PID of the daemon
 * @return TRUE on success, FALSE otherwise
*/
int write_pid(pid_t pid)
{
	FILE * fd;
	
	if( (fd=fopen("/var/run/MLD_Proxy.pid","w")) == NULL)
	{
		perror("MLD_Proxy.log");
		return FALSE;
	}
	
	fprintf(fd,"%d\n",pid);
	
	fclose(fd);
	return TRUE;
}

/**
 * Print stats on the packets sent and received for each MN and group
 * @param fwd_list The forwarding table
 * @param mn_list The MN list
 * @param list The MN list
*/
void print_stat(forward_t *fwd_list, MN_t *mn_list)
{
	forward_t *tmp_fwd = fwd_list;
	MN_t *tmp_mn = mn_list;
	iface_t *tmp_iface = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	
	while(tmp_mn != NULL)
	{
		fprintf(stderr,"\033[0;34m\t%s\n\t\tsent %d packets to %s\n\t\treceived %d packets from %s\n\n\033[0m",
		tmp_mn->tunnel_name,tmp_mn->nb_fwded,tmp_mn->tunnel_name,tmp_mn->nb_sent,tmp_mn->tunnel_name);
		tmp_mn=tmp_mn->next;
	}

	if(fwd_list == NULL)
	{
		if(debug_lvl == VERBOSE)
			fprintf(stderr,"\033[0;34m\tNo groups forwarded -> cannot print stats.\n\033[0m");
		return;
	}

	while(tmp_fwd != NULL)
	{
		tmp_iface = tmp_fwd->interfaces;
		inet_ntop(AF_INET6,tmp_fwd->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
		inet_ntop(AF_INET6,tmp_fwd->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		
		if( tmp_fwd->mode == MC_ASM)
		{
			fprintf(stderr,"\033[0;34m\tASM Group %s :\n\033[0m", grp);
		}
		else if( tmp_fwd->mode == MC_SSM)
		{
			fprintf(stderr,"\033[0;34m\tSSM Channel (%s,%s) :\n\033[0m",src,grp);
		}
		
		while(tmp_iface != NULL)
		{
			tmp_mn = get_MN_by_id(mn_list, tmp_iface->iface_id);
			fprintf(stderr,"\033[0;34m\t\tforwarded %d packets to %s\n\033[0m",tmp_iface->nb_fwded,tmp_mn->tunnel_name);
			tmp_iface=tmp_iface->next;
		}
		fprintf(stderr,"\n");
		tmp_fwd=tmp_fwd->next;
	}
	
	return;
}

/**
 * Print stats on the packets sent and received for each MN and group
 * @param fwd_list The forwarding table
 * @param mn_list The MN list
 * @param list The MN list
*/
void fprint_stat(forward_t *fwd_list, MN_t *mn_list)
{
	forward_t *tmp_fwd = fwd_list;
	MN_t *tmp_mn = mn_list;
	iface_t *tmp_iface = NULL;
	char grp[INET6_ADDRSTRLEN],src[INET6_ADDRSTRLEN];
	int fd;
	char buffer[256];

	if( (fd = open("./MLD_Proxy.stat", O_RDWR|O_CREAT|O_TRUNC)) == -1)
	{
		perror("open MLD.stat");
		exit(-1);
	}		
	
	sprintf(buffer,"\n#####   MLD_Proxy Statistics   #####\n\n");
	write(fd,buffer,strlen(buffer));
	
	while(tmp_mn != NULL)
	{
		sprintf(buffer,"\t%s\n\t\tsent %d packets to %s\n\t\treceived %d packets from %s\n\n",
		tmp_mn->tunnel_name,tmp_mn->nb_fwded,tmp_mn->tunnel_name,tmp_mn->nb_sent,tmp_mn->tunnel_name);
		tmp_mn=tmp_mn->next;
		write(fd,buffer,strlen(buffer));
	}

	if(fwd_list == NULL)
	{
		sprintf(buffer,"\tNo groups forwarded -> cannot print stats.\n");
		write(fd,buffer,strlen(buffer));
		return;
	}

	while(tmp_fwd != NULL)
	{
		tmp_iface = tmp_fwd->interfaces;
		inet_ntop(AF_INET6,tmp_fwd->grp_addr.s6_addr,grp,INET6_ADDRSTRLEN);
		inet_ntop(AF_INET6,tmp_fwd->src_addr.s6_addr,src,INET6_ADDRSTRLEN);
		
		if( tmp_fwd->mode == MC_ASM)
		{
			sprintf(buffer,"\tASM Group %s :\n", grp);
			write(fd,buffer,strlen(buffer));
		}
		else if( tmp_fwd->mode == MC_SSM)
		{
			sprintf(buffer,"\tSSM Channel (%s,%s) :\n",src,grp);
			write(fd,buffer,strlen(buffer));
		}
		
		while(tmp_iface != NULL)
		{
			tmp_mn = get_MN_by_id(mn_list, tmp_iface->iface_id);
			sprintf(buffer,"\t\tforwarded %d packets to %s\n",tmp_iface->nb_fwded,tmp_mn->tunnel_name);
			write(fd,buffer,strlen(buffer));
			tmp_iface=tmp_iface->next;
		}
		sprintf(buffer,"\n");
		write(fd,buffer,strlen(buffer));
		tmp_fwd=tmp_fwd->next;
	}
	
	fsync(fd);
	close(fd);
}

int check_duplicate(const u_char *buffer, int size)
{
  char my_sig[SIG_SIZE];
  pk_sig_t *pk_sig = pk_sigs;
  int offset = 0;

  struct timeval tv;
  gettimeofday(&tv, NULL);
  clean_duplicate(&tv);

  memset(my_sig, '\0', SIG_SIZE);

  while (offset < size)
    {
      my_sig[offset % SIG_SIZE] ^= buffer[offset];
      offset++;
    }

  while (pk_sig != NULL) 
    {
      if (memcmp(my_sig, pk_sig->sig, SIG_SIZE) == 0)
	{
	  //fprintf(stderr, "Duplicated packet\n");
	  return -1;
	}
      pk_sig = pk_sig->next;
    }
  //fprintf(stderr, "Original packet\n");

  pk_sig = (pk_sig_t *) malloc(sizeof(pk_sig_t));
  memcpy(pk_sig->sig, my_sig, SIG_SIZE);
  memcpy(&pk_sig->tv, &tv, sizeof(struct timeval));
  pk_sig->next = pk_sigs;
  pk_sigs = pk_sig;
  return 0;
}

void clean_duplicate(struct timeval *tv)
{
  pk_sig_t *pk_sig = pk_sigs, *temp = NULL;

  while (pk_sig)
    {
      if ((pk_sig->tv.tv_sec + PK_TIMEOUT) < tv->tv_sec)
	{
	  if (temp == NULL)
	    {
	      pk_sigs = pk_sig->next;
	      free(pk_sig);
	      pk_sig = pk_sigs;
	    }
	  else
	    {
	      temp->next = pk_sig->next;
	      free(pk_sig);
	      pk_sig = temp->next;
	    }
	}
      else
	{
	  temp = pk_sig;
	  pk_sig = pk_sig->next;
	}
    }
}

