/* DHCP Gateway for 'DHCPv4 Configuration of IPSec Tunnel Mode' support
 * Copyright (C) 2002 Mario Strasser <mast@gmx.net>,
 *                    Zuercher Hochschule Winterthur,
 *                    Netbeat AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: dhcp.c,v 1.1.1.1 2002/08/20 08:42:03 sri Exp $
 */

#include <stdio.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "dhcp.h"

/**
 * get_dhcp_option - returns a pointer to a dhcp option
 * p - pointer to the dhcp packet
 * returns a pointer to the option on success, NULL otherwise
 */
u_char *get_dhcp_option(struct dhcp_packet *p, u_char type)
{
    u_char *op = p->options + 4;
    /* check magic cookie */
    if (memcmp(p->options, DHCP_OPTIONS_COOKIE, 4))
        return NULL;
    /* find option */
    while (*op != DHO_END) {
        if (*op == type)
            return op;
        if (*op == DHO_PAD)
            op++;
        else
            op += op[1] + 2;
    }
    return NULL;
}

/**
 * get_dhcp_packet_len - returns the length of a dhcp packet
 * p - pointer to the dhcp packet
 * returns the length of the packet
 */
int get_dhcp_packet_len(struct dhcp_packet *p)
{
    u_char *op = p->options;
    /* check magic cookie */
    if (!memcmp(p->options, DHCP_OPTIONS_COOKIE, 4)) {
        op += 4;
        /* find last option */
        while (*op != DHO_END) {
            if (*op == DHO_PAD)
                op++;
            else
                op += op[1] + 2;
        }
    }
    return (int)op - (int)p + 1;
}

/**
 * get_dhcp_packet_type - gets the message type of a dhcp packet
 * p - pointer to the dhcp packet
 * returns the message type on success, -1 otherwise
 */
int get_dhcp_packet_type(struct dhcp_packet *p)
{
    u_char *op;

    /* get message type option */
    op = get_dhcp_option(p, DHO_DHCP_MESSAGE_TYPE);
    if (op != NULL)
        return op[2];
    else
        return -1;
}

/**
 * add_dhcp_option - appends a dhcp option to a dhcp packet
 * p - pointer to the dhcp packet
 * newop - the additional option 
 * oplen - length of the additional option
 * returns 0 on success, -1 otherwise
 */
int add_dhcp_option(struct dhcp_packet *p, u_char *newop, int oplen)
{
    u_char *op = p->options + 4;
    int len = get_dhcp_packet_len(p);

    /* check if there is enough space left */
    if (len + oplen + 1 > sizeof(struct dhcp_packet)) 
        return -1;
    /* check magic cookie */
    if (memcmp(p->options, DHCP_OPTIONS_COOKIE, 4))
        return -1;
    /* find option */
    while (*op != DHO_END) {
        if (*op == DHO_PAD)
            op++;
        else
            op += op[1] + 2;
    }
    /* add new option */
    memcpy(op, newop, oplen);
    /* and end marker */
    op[oplen] = DHO_END;

    return 0;
}

/**
 * check_dhcp_packet - checks if a dhcp packet is of a certain type
 * p - pointer to the dhcp packet
 * type - message type of which the packet must be
 * returns 0 on success, -1 otherwise
 */
int check_dhcp_packet(struct dhcp_packet *p, u_char type)
{
    /* it must be either a BOOTREQUEST or a BOOTREPLY */
    if (p->op != BOOTREQUEST && p->op != BOOTREPLY)
        return -1;
    /* check message type */
    if (get_dhcp_packet_type(p) == type)
        return 0;
    else
        return -1;
}

/**
 * show_dhcp_options - shows all options of a dhcp packet
 * op - pointer to first option
 */
void show_dhcp_options(u_char * op)
{
    u_int i, j;
    char str[300];

    while (*op != DHO_END) {
        switch (*op) {
        case DHO_PAD:
            printf("| PAD\n");
            break;
        case DHO_SUBNET_MASK:
            printf("| Net-Mask:\t");
            for (i = 2; i < op[1] + 1; i++)
                printf("%d.", (u_int) op[i]);
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_ROUTERS:
            printf("| Routers:\t");
            for (i = 2; i < op[1] + 1; i++) {
                if ((i - 1) % 4)
                    printf("%d.", (u_int) op[i]);
                else
                    printf("%d, ", (u_int) op[i]);
            }
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_DOMAIN_NAME_SERVERS:
            printf("| DNS Server:\t");
            for (i = 2; i < op[1] + 1; i++) {
                if ((i - 1) % 4)
                    printf("%d.", (u_int) op[i]);
                else
                    printf("%d, ", (u_int) op[i]);
            }
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_HOST_NAME:
            memcpy(str, &op[2], op[1]);
            str[op[1]] = 0;
            printf("| Host:\t\t%s\n", str);
            break;
        case DHO_DOMAIN_NAME:
            memcpy(str, &op[2], op[1]);
            str[op[1]] = 0;
            printf("| Domain:\t%s\n", str);
            break;
        case DHO_BROADCAST_ADDRESS:
            printf("| Broadcast:\t");
            for (i = 2; i < op[1] + 1; i++)
                printf("%d.", (u_int) op[i]);
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_NETBIOS_NAME_SERVERS:
            printf("| Wins-Server:\t");
            for (i = 2; i < op[1] + 1; i++) {
                if ((i - 1) % 4)
                    printf("%d.", (u_int) op[i]);
                else
                    printf("%d, ", (u_int) op[i]);
            }
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_NETBIOS_NODE_TYPE:
            printf("| Node-Type:\t%d\n", (u_int) op[2]);
            break;
        case DHO_DHCP_REQUESTED_ADDRESS:
            printf("| Req. IP:\t");
            for (i = 2; i < op[1] + 1; i++)
                printf("%d.", (u_int) op[i]);
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_DHCP_LEASE_TIME:
            i = op[2] + (op[3] << 8) + (op[4] << 16) + (op[5] << 24);
            printf("| Lease-Time:\t%u\n", i);
            break;
        case DHO_DHCP_MESSAGE_TYPE:
            printf("| Type:\t\t");
            switch (op[2]) {
            case DHCPDISCOVER:
                printf("DHCPDISCOVER\n");
                break;
            case DHCPOFFER:
                printf("DHCPOFFER\n");
                break;
            case DHCPREQUEST:
                printf("DHCPREQUEST\n");
                break;
            case DHCPDECLINE:
                printf("DHCPDECLINE\n");
                break;
            case DHCPACK:
                printf("DHCPACK\n");
                break;
            case DHCPNAK:
                printf("DHCPNAK\n");
                break;
            case DHCPRELEASE:
                printf("DHCPRELEASE\n");
                break;
            case DHCPINFORM:
                printf("DHCPINFORM\n");
                break;
            default:
                printf("unknown (%x02)\n", op[2]);
            }
            break;
        case DHO_DHCP_SERVER_IDENTIFIER:
            printf("| Server:\t");
            for (i = 2; i < op[1] + 1; i++)
                printf("%d.", (u_int) op[i]);
            printf("%d\n", (u_int) op[i]);
            break;
        case DHO_DHCP_PARAMETER_REQUEST_LIST:
            printf("| Req. Param:\t");
            for (i = 2; i < op[1] + 1; i++)
                printf("%02x ", (u_int) op[i]);
            printf("\n");
            break;
        case DHO_DHCP_MESSAGE:
            memcpy(str, &op[2], op[1]);
            str[op[1]] = 0;
            printf("| Message:\t%s\n", str);
            break;
        case DHO_DHCP_RENEWAL_TIME:
            i = op[2] + (op[3] << 8) + (op[4] << 16) + (op[5] << 24);
            printf("| Renewal-Time:\t%u\n", i);
            break;
        case DHO_DHCP_REBINDING_TIME:
            i = op[2] + (op[3] << 8) + (op[4] << 16) + (op[5] << 24);
            printf("| Reb.-Time:\t%u\n", i);
            break;
        case DHO_DHCP_CLIENT_IDENTIFIER:
            printf("| Client ID:\t");
            for (i = 2; i < op[1] + 2; i++)
                printf("%02x ", (u_int) op[i]);
            printf("\n");
            break;
        case DHO_DHCP_AGENT_OPTIONS:
            for (i=2; i < op[1] + 2; i += op[i + 1] + 2) {
                switch (op[i]) {
                case RAI_CIRCUIT_ID:
                    printf("| Circuit ID:\t");
                    break;
                case RAI_REMOTE_ID:
                    printf("| Remote ID:\t");
                    break;
                case RAI_AGENT_ID:
                    printf("| Agent ID:\t");
                    break;
                }
                for (j = 2; j < op[i + 1] + 2; j++)
                    printf("%02x ", (u_int) op[i + j]);
                memcpy(str, &op[i + 2], op[i + 1]);
                str[op[i + 1]] = 0;
                printf("(%s)\n", str);
            }
            break;
        default:
            printf("| %02x\t\t", (u_int) op[0]);
            for (i = 2; i < op[1] + 2; i++)
                printf("%02x ", (u_int) op[i]);
            printf("\n");
        }
        /* next option */
        if (*op == DHO_PAD)
            op++;
        else
            op += op[1] + 2;
    }
}

/**
 * show_dhcp_packet - prints the content of a dhcp packet
 * p - pointer to the dhcp packet
 * verbose - information level (0-3)
 */
void show_dhcp_packet(struct dhcp_packet *p, int verbose)
{
    int i, type;
    char sname[DHCP_SNAME_LEN + 1];
    char file[DHCP_FILE_LEN + 1];

    /* verbose = 0 -> no output */
    if (verbose == 0) return;

    /* get message type */
    type = get_dhcp_packet_type(p);
    if (type == -1) {
        fprintf(stderr, "Error: Can't show dhcp packet.\n");
        return;
    }

    /* print message type */
    printf("+------------------------------------------------------------\n");
    switch (type) {
        case DHCPDISCOVER:
            printf("| DHCPDISCOVER");
            break;
        case DHCPOFFER:
            printf("| DHCPOFFER");
            break;
        case DHCPREQUEST:
            printf("| DHCPREQUEST");
            break;
        case DHCPDECLINE:
            printf("| DHCPDECLINE");
            break;
        case DHCPACK:
            printf("| DHCPACK");
            break;
        case DHCPNAK:
            printf("| DHCPNAK");
            break;
        case DHCPRELEASE:
            printf("| DHCPRELEASE");
            break;
        case DHCPINFORM:
            printf("| DHCPINFORM");
            break;
        default:
            printf("| unknown: %x02", type);
    }
    switch (p->op) {
    case BOOTREQUEST:
        printf(" (BOOTREQUEST)\n");
        break;
    case BOOTREPLY:
        printf(" (BOOTREPLY)\n");
        break;
    default:
        printf(" (unknown: %x02)\n", p->op);
        return;
    }
    printf("+------------------------------------------------------------\n");
    printf("| Addr:\t");
    switch (p->htype) {
    case HTYPE_ETHER:
        printf("ETHER\t");
        break;
    case HTYPE_IEEE802:
        printf("IEEE 802\t");
        break;
    case HTYPE_FDDI:
        printf("FDDI\t");
        break;
    default:
        printf("unknown: %02x\t", p->htype);
    }
    printf("(Size: %d)\n", (u_int) p->hlen);
    if (verbose > 1)
        printf("| Hops: \t%d\n", (u_int) p->hops);
    printf("| TID:  \t%04x\n", (u_int) ntohl(p->xid));
    if (verbose > 1) {
        printf("| Sec.: \t%d\n", (u_int) ntohs(p->secs));
        printf("| Flags:\t%04x\n", (u_int) ntohs(p->flags));
        printf("| C IP: \t%s\n", inet_ntoa(p->ciaddr));
    }
    printf("| Y IP: \t%s\n", inet_ntoa(p->yiaddr));
    printf("| S IP: \t%s\n", inet_ntoa(p->siaddr));
    if (verbose > 1)
        printf("| R IP: \t%s\n", inet_ntoa(p->giaddr));
    printf("| MAC:  \t");
    if (p->hlen > 16)
        p->hlen = 16;
    for (i = 0; i < p->hlen - 1; i++)
        printf("%02x:", (u_int) p->chaddr[i]);
    printf("%02x\n", (u_int) p->chaddr[p->hlen - 1]);
    if (verbose > 1) {
        memset(sname, 0, DHCP_SNAME_LEN + 1);
        memcpy(sname, p->sname, DHCP_SNAME_LEN);
        printf("| Server:\t%s\n", sname);
        memset(file, 0, DHCP_FILE_LEN + 1);
        memcpy(file, p->file, DHCP_FILE_LEN);
        printf("| File: \t%s\n", file);
    }
    printf("+------------------------------------------------------------\n");
    if (verbose > 2) {
        show_dhcp_options(p->options + 4);
        printf("+------------------------------------------------------------\n");
    }
}
