/*
 * Copyright (C) 2004 LSIIT Laboratory.
 * Universit Louis Pasteur, Strasbourg, France.
 * All rights reserved.
 *
 * Original version by Christophe Jelger (jelger@dpt-info.u-strasbg.fr)
 * Developed by Frdric Beck (beck@startx.u-strasbg.fr)
 * Currently maintained and updated by Christophe Jelger
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef _MLD_PROXY_MEMBER_H_
#define _MLD_PROXY_MEMBER_H_


#include "MLD_Proxy.h"
#include <netinet/in.h>
#include <netinet/ip6.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <unistd.h>
#include <stdlib.h>

/** The structure identifying a joined group on the LAN interface */
typedef struct mult_group{
	/** If SSM Channel, source address */
	struct in6_addr source_addr;
	/** Group address */
	struct in6_addr group_addr;
	/** Multicast Mode, MC_ASM or MC_SSM */
	int mode;
	/** Multicast Scope, MC_LINK, MC_SITE or MC_GLOBAL scope */
	int scope;
	/** Pointer to the next item */
	struct mult_group *next;
	/** Pointer to the previous item */
	struct mult_group *previous;
	}mult_group_t;

/** The structure identifying the LAN interface and infos about the joined groups on it */
typedef struct membership{
	/** Name of the interface used to join multicast groups */
	char oif_name[16];
	/** System identifier for this interface */
	int oif_id;
	/** Address of this interface */
	struct in6_addr oif_addr;
	/** List of multicast groups joined */
	mult_group_t * joined_groups;
}membership_t;



/** Inits the struct membership
 * @param list The structure which will identify the LAN interface and the joined groups
 * @param name The name of the LAN interface
 * @return TRUE if succes, FALSE otherwise 
 */
int init_membership(membership_t *list, char *name);

/* add group */
/** Add a joined ASM group
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param addr A string identifying the ASM Group
 * @return TRUE if succes, FALSE otherwise 
 */
int add_mult_group_by_addr_c(membership_t *list, char *addr);
/** Add a joined ASM group
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param grp_addr The IPv6 address of the ASM Group
 * @return TRUE if succes, FALSE otherwise 
 */
int add_mult_group_by_addr(membership_t *list, struct in6_addr grp_addr);
/** Add a joined SSM Channel
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr A string identifying the Source of the SSM Channel
 * @param group_addr A string identifying the Group of the SSM Channel
 * @return TRUE if succes, FALSE otherwise 
 */
int add_mult_group_SSM_by_addr_c(membership_t *list, char *source_addr, char *group_addr);
/** Add a joined SSM Channel
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr The IPv6 address of the Source of the SSM Channel
 * @param group_addr The IPv6 address of the Group of the SSM Channel
 * @return TRUE if succes, FALSE otherwise 
 */
int add_mult_group_SSM_by_addr(membership_t *list, struct in6_addr source_addr, struct in6_addr group_addr );

/* del group */
/** Delete a joined ASM group
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param addr A string identifying the ASM Group
 * @return TRUE if succes, FALSE otherwise 
 */
int del_mult_group_by_addr_c(membership_t *list, char *addr);
/** Delete a joined ASM group
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param grp_addr The IPv6 address of the ASM Group
 * @return TRUE if succes, FALSE otherwise 
 */
int del_mult_group_by_addr(membership_t *list, struct in6_addr addr);
/** Delete a joined SSM Channel
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr A string identifying the Source of the SSM Channel
 * @param group_addr A string identifying the Group of the SSM Channel
 * @return TRUE if succes, FALSE otherwise 
 */
int del_mult_group_SSM_by_addr_c(membership_t *list, char *source_addr, char *group_addr);
/** Delete a joined SSM Channel
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr The IPv6 address of the Source of the SSM Channel
 * @param group_addr The IPv6 address of the Group of the SSM Channel
 * @return TRUE if succes, FALSE otherwise 
 */
int del_mult_group_SSM_by_addr(membership_t *list, struct in6_addr source_addr, struct in6_addr group_addr );
/** Delete all joined Multicast Groups
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return TRUE if succes, FALSE otherwise 
 */
int del_all_mult_group(membership_t *list);

/* print membership */
/** Print all Multicast joined Groups on the standard output
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return TRUE if succes, FALSE otherwise 
 */
void print_mult_group(membership_t list);
/** Print all the informations about the LAN interface on the standard output
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return TRUE if succes, FALSE otherwise 
 */
void print_membership(membership_t list);
/** Print all Multicast joined Groups in the log file
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return TRUE if succes, FALSE otherwise 
 */
void fprint_mult_group(membership_t list);
/** Print all the informations about the LAN interface in the log file
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return TRUE if succes, FALSE otherwise 
 */
void fprint_membership(membership_t list);


/** Get the number of Multicast groups joined on the LAN interface
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return The number of joined groups if succes, -1 otherwise 
 */
int nb_joined_group(membership_t list);

/* check if group is joined */
/** Check if an ASM group is joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param addr A string identifying the ASM Group
 * @return TRUE if succes, FALSE otherwise 
 */
int is_group_joined_by_addr_c(membership_t list, char *addr);
/** Check if an ASM group is joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param addr  The IPv6 address of the ASM Group
 * @return TRUE if succes, FALSE otherwise 
 */
int is_group_joined_by_addr(membership_t list, struct in6_addr addr);
/** Check if a SSM Channel is joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr A string identifying the Source of the SSM Channel
 * @param group_addr A string identifying the Group of the SSM Channel
 * @return TRUE if success, FALSE otherwise 
 */
int is_group_joined_SSM_by_addr_c(membership_t list, char *src_addr, char *grp_addr);
/** Check if a SSM Channel is joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr The IPv6 address of the Source of the SSM Channel
 * @param group_addr The IPv6 address of the Group of the SSM Channel
 * @return TRUE if success, FALSE otherwise 
 */
int is_group_joined_SSM_by_addr(membership_t list, struct in6_addr src_addr, struct in6_addr grp_addr );
/** Check if there are joined groups
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return TRUE if success, FALSE otherwise 
 */
int are_groups_joined(membership_t list);

/* get a group */
/** Get an ASM group if joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param addr A string identifying the ASM Group
 * @return A pointer to the ASM Group if succes, NULL otherwise 
 */
mult_group_t * get_group_by_addr_c(membership_t list, char *addr);
/** Get an ASM group if joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param addr  The IPv6 address of the ASM Group
 * @return A pointer to the ASM Group if succes, NULL otherwise 
 */
mult_group_t * get_group_by_addr(membership_t list, struct in6_addr addr);
/** Get a SSM Channel if joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr A string identifying the Source of the SSM Channel
 * @param group_addr A string identifying the Group of the SSM Channel
 * @return A pointer to theSSM Channel if succes, NULL otherwise 
 */
mult_group_t * get_group_SSM_by_addr_c(membership_t list, char *src_addr, char *grp_addr);
/** Get a SSM Channel if joined
 * @param list The structure which identifies the LAN interface and the joined groups
 * @param source_addr The IPv6 address of the Source of the SSM Channel
 * @param group_addr The IPv6 address of the Group of the SSM Channel
 * @return A pointer to the SSM Channel if succes, NULL otherwise 
 */
mult_group_t * get_group_SSM_by_addr(membership_t list, struct in6_addr src_addr, struct in6_addr grp_addr );
/** Get the joined group list
 * @param list The structure which identifies the LAN interface and the joined groups
 * @return A pointer to the joined group list if succes, NULL otherwise 
 */
mult_group_t * get_groups(membership_t list);

/** Free all the informations about the LAN interface, including the joined groups
 * @param list The structure which identifies the LAN interface and the joined groups
 */
void free_membership( membership_t *list);

#endif
