#!/bin/sh /etc/rc.common
####### Error code table
#101: ERROR: ports can not be empty
#102: ERROR: Start port must smaller than end port
#103: ERROR: Target interface profile is disabled
#104: ERROR: WAN IP alias must have value
#105: ERROR: Public port can not contain system reserved ports

START=95

UCI_CONFIG=firewall
IPTABLES=iptables
DEBUG_PRINT="logger"
#DEBUG_PRINT="echo"
SCRIPT_LOCK="/tmp/web_apply_lock/port_redirection"
HASHTABLE=port_redirection
UCI_TMP_PATH="/tmp/.uci/$UCI_CONFIG"
BOOT_FILE="/tmp/profile_backup/port_redirection/boot_file"
DEBUG_LOG="/tmp/profile_backup/port_redirection/debug_log"
pre_rule=0
CGI_ERROR_MSG="/tmp/cgi_error_msg"
#reserved ports: these ports are opened for special use, they should not be used in public port
reserved_ports="68 546"

source /etc/func_htable.sh

DEBUG() {
	msg=$1
	if [ "$DEBUG_PRINT" = "logger" ];then
		$DEBUG_PRINT "${msg}"
	else
		$DEBUG_PRINT ${msg} >/dev/console
	fi
}

handle_1to1() {
	#check reserved ports
	for i in $reserved_ports ;do
		if [ "$public_port_start" = "$i" ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start
	private_port=$private_port_start
	private_port2=:$private_port_start
	#clear non-used option
	uci set $UCI_CONFIG.$1.public_port_end=
}
handle_nto1() {
	#check end port > start port
	if [ "$public_port_start" -gt "$public_port_end" ] ;then
		echo -n "ERROR: Start port must smaller than end port" >$CGI_ERROR_MSG
		is_legal_rule=102
	fi
	#check reserved ports
	for i in $reserved_ports ;do
		if [ $(($i-$public_port_start)) -ge 0 -a $(($public_port_end-$i)) -ge 0 ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start:$public_port_end
	private_port=$private_port_start
	private_port2=:$private_port_start
}
handle_nton() {
	#check end port > start port
	if [ "$public_port_start" -gt "$public_port_end" ] ;then
		echo -n "ERROR: Start port must smaller than end port" >$CGI_ERROR_MSG
		is_legal_rule=102
	fi
	#check reserved ports
	for i in $reserved_ports ;do
		if [ $(($i-$public_port_start)) -ge 0 -a $(($public_port_end-$i)) -ge 0 ] ;then
			if [ "$proto" = "both" -o "$proto" = "udp" ];then
				echo -n "WARNING: UDP ports:$reserved_ports set as default for DHCP WAN are excluded from port redirection" >$CGI_ERROR_MSG
				is_legal_rule=105
				break
			fi
		fi
	done
	public_port=$public_port_start:$public_port_end
	private_port=$public_port_start:$public_port_end
	private_port2=""
	#clear non-used option
	uci set $UCI_CONFIG.$1.private_port=
}

setup_parameters() {
	is_legal_rule="yes"
	if [ "$2" = "apply" ] ;then
		status=$(uci get $UCI_CONFIG.$1.status)
		proto=$(uci get $UCI_CONFIG.$1.proto)
		port_redirect_mode=$(uci get $UCI_CONFIG.$1.redirect_mode)
		private_port_start=$(uci get $UCI_CONFIG.$1.private_port)
		private_port_end=$(uci get $UCI_CONFIG.$1.private_port_end)
		private_ip=$(uci get $UCI_CONFIG.$1.private_ip)
		public_intf=$(uci get $UCI_CONFIG.$1.public_intf)
		public_port_end=$(uci get $UCI_CONFIG.$1.public_port_end)
		public_port_start=$(uci get $UCI_CONFIG.$1.public_port)
		use_ipalias=$(uci get $UCI_CONFIG.$1.use_ipalias)
		ip_alias=$(uci get $UCI_CONFIG.$1.ip_alias)
	else
		config_get status $1 status
		config_get public_intf $1 public_intf
		config_get ip_alias $1 ip_alias
		config_get private_ip $1 private_ip
		config_get proto $1 proto
		config_get port_redirect_mode $1 redirect_mode
		config_get public_port_start $1 public_port
		config_get public_port_end $1 public_port_end
		config_get private_port_start $1 private_port
		config_get private_port_end $1 private_port_end
		config_get use_ipalias $1 use_ipalias
	fi
	flag=
	[ "$status" != "enable" ] && flag="-u"
	
	if [ "$proto" = "both" ];then
		protocol="-m service --service-flag tcp,udp"
	else
		protocol="-p $proto"
	fi
	
	ip_alias=$(echo $ip_alias | cut -d '/' -f 1)
	
	######## CHECK PORT REDIRECT MODE AND SETUP PORT INDEX ##########
	if [ "$port_redirect_mode" = "1to1" ] ;then
		handle_1to1 $1
	elif [ "$port_redirect_mode" = "nto1" ] ;then
		handle_nto1 $1
	elif [ "$port_redirect_mode" = "nton" ] ;then
		handle_nton $1
	else	#set to default for versions before 1132
		DEBUG "(WebUI port redirection), profile:$1 version upgraded completed."
		#if origianl case is likely an n-to-? case
		if [ ! "$public_port_start" = "0" ] && [ ! "$public_port_end" = "0" ] && [ ! "$private_port_start" = "0" ] && [ ! "$public_port_start" = "$public_port_end" ] ;then
			#echo "profile=$1, n-to-?" >/dev/console
			if [ "$private_port_end" = "0" ] ;then
				#echo "profile=$1, n-to-1" >/dev/console
				uci set $UCI_CONFIG.$1.redirect_mode="nto1"
				#empty port is not allowed
				if [ "$public_port_start" = "" ] && [ "$public_port_end" = "" ] && [ "$private_port_start" = "" ] ;then
					echo -n "ERROR: Port can not be empty" >$CGI_ERROR_MSG
					is_legal_rule=101
				fi
				handle_nto1 $1
			else
				#echo "profile=$1, n-to-n" >/dev/console
				uci set $UCI_CONFIG.$1.redirect_mode="nton"
				#empty port is not allowed
				if [ "$public_port_start" = "" ] && [ "$public_port_end" = "" ] ;then
					echo -n "ERROR: Port can not be empty" >$CGI_ERROR_MSG
					is_legal_rule=101
				fi
				handle_nton $1
			fi
		else
			#echo "profile=$1, 1-to-1" >/dev/console
			uci set $UCI_CONFIG.$1.redirect_mode="1to1"
			#empty port is not allowed
			if [ "$public_port_start" = "" ] && [ "$private_port_start" = "" ] ;then
				echo -n "ERROR: Port can not be empty" >$CGI_ERROR_MSG
				is_legal_rule=101
			fi
			handle_1to1 $1
		fi
	fi

	############## SETUP PUBLIC IP according to WAN INTERFACE ##############
	#ipset:
	#all_interface: contains all of ipalias_wan#
	#ipalias_wan#: contains all alias of a single WAN including itself
	#ip_wan#: WAN IP
	if [ "$public_intf" = "All" ] ;then
		uci set $UCI_CONFIG.$1.use_ipalias=
		uci set $UCI_CONFIG.$1.ip_alias=
		dip="-m set --set all_interface dst"
	fi
	if [ "$public_intf" != "All" ] ;then
		#if ipset:ip_$public_intf does not exist(i.e.,:WAN profile is disabled or someone destroy the ipset)
		ipset -L ip_$public_intf >/dev/null 2>/dev/null
		if [ "$?" != "0" ] ;then
			echo -n "ERROR: Target interface profile is disabled or not existed" >$CGI_ERROR_MSG
			is_legal_rule=103
		else
			#if "Use IP Alias"="no", we include the WAN's IP only
			if [ "$use_ipalias" = "no" ] || [ "$use_ipalias" = "wan" ] || [ "$use_ipalias" = "disable" ] ;then
				uci set $UCI_CONFIG.$1.ip_alias=
				uci set $UCI_CONFIG.$1.use_ipalias=no
				dip="-m set --set ip_$public_intf dst"
			elif [ "$use_ipalias" = "ip_alias" ] ;then	#if "Use IP Alias"="Single_Alias", we include only one of the WAN's alias but not itself
				#use the WAN IP as default if option value is empty or out-of-date version
				if [ "$ip_alias" = "" ] ;then
					echo -n "ERROR: WAN IP alias must have value" >$CGI_ERROR_MSG
					is_legal_rule=104
				else
					dip="-d $ip_alias"
				fi
			elif [ "$use_ipalias" = "all" ] ; then	#if "Use IP Alias"="All", we include the WAN IP and its all alias
				uci set $UCI_CONFIG.$1.ip_alias=
				dip="-m set --set ipalias_$public_intf dst"
			else	#set to "no" as a default for versions before 1132
				uci set $UCI_CONFIG.$1.use_ipalias=no
				uci set $UCI_CONFIG.$1.ip_alias=
				dip="-m set --set ip_$public_intf dst"
			fi
		fi
 	fi
}

add_rule_boot() {
	setup_parameters $1 boot
	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		#port redirection (PREROUTING): If the packet want to access public ip:port(redirected)
		if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
			echo "-A nat_portredirect $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}" >> $BOOT_FILE
		else
			$IPTABLES -t nat -A nat_portredirect $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
		fi
		if [ "$?" = "0" ] ;then 
			add_result=y
		else
			add_result=n
		fi
		#port redirection (POSTROUTING) for loop back: If the packet want to access public ip:port(redirected)
			#[G38092]: support special NAT loop back which packet's source IP is not in our LAN's subnet but is registered by static route
		if [ -f "$BOOT_FILE" -a "$fastboot" = 1 ] ;then
			echo "-A nat_portredirect2 $flag $protocol -m mset --set lan_nat_subnet src -d $private_ip --dport $private_port -j MASQUERADE" >> $BOOT_FILE
		else
			$IPTABLES -t nat -A nat_portredirect2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j MASQUERADE
		fi
		if [ "$?" = "0" ] ;then
			add_result=y
		else
			add_result=n
		fi
		
		#prevent files from duplicated add by fastboot if fastboot fails => clear then re-add
		ht_rm_file $HASHTABLE $1 2>/dev/null
		if [ "$add_result" = "y" ] ;then
			ht_put $HASHTABLE $1 IDX $boot_uciid_counter
		else
			echo "add_rule_boot($1): error when creating iptables rule" >>$DEBUG_LOG
		fi
		#save private ip to ipset:nat_ptre, this is used in "User Management":base_configuration.init
			#Reason: prevent port redirection traffic being dropped by User Management mechanism
		if [ "$flag" != "-u" ] ;then
			ipset -A nat_ptre $private_ip 2>/dev/null
		fi
	else
		#use a dummy rule instead
		if [ -f "$BOOT_FILE" -a "$fastboot" = "1" ] ;then
			echo "-A nat_portredirect -u -j RETURN" >> $BOOT_FILE
			echo "-A nat_portredirect2 -u -j RETURN" >> $BOOT_FILE
		else
			$IPTABLES -t nat -A nat_portredirect -u -j RETURN
			$IPTABLES -t nat -A nat_portredirect2 -u -j RETURN
		fi
		ht_put $HASHTABLE $1 IDX $boot_uciid_counter
		uci set $UCI_CONFIG.$1.status="disable"
	fi
	boot_uciid_counter=$(($boot_uciid_counter+1))
}

add_rule() {
	#echo "add_rule: $1" >/dev/console
	setup_parameters $1 apply
	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		#port redirection (PREROUTING): If the packet want to access public ip:port(redirected)
		$IPTABLES -t nat -A nat_portredirect $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
		if [ "$?" = "0" ] ;then 
			add_result=y
		else
			add_result=n
		fi
		#port redirection (POSTROUTING) for loop back: If the packet want to access public ip:port(redirected)
			#[G38092]: support special NAT loop back which packet's source IP is not in our LAN's subnet but is registered by static route
		$IPTABLES -t nat -A nat_portredirect2 $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j MASQUERADE
		if [ "$?" = "0" ] ;then 
			add_result=y
		else
			add_result=n
		fi
		if [ "$add_result" = "y" ] ;then
			ht_put $HASHTABLE $1 IDX $uciid
			if [ "$flag" != "-u" ] ;then
				ipset -A nat_ptre $private_ip 2>/dev/null
			fi
		else
			echo "add_rule($1): error when creating iptables rule" >>$DEBUG_LOG
		fi
	else
		#use a dummy rule instead
		$IPTABLES -t nat -A nat_portredirect -u -j RETURN
		$IPTABLES -t nat -A nat_portredirect2 -u -j RETURN
		ht_put $HASHTABLE $1 IDX $uciid
		uci set $UCI_CONFIG.$1.status="disable"
	fi
}

swap_rule() {
	#echo "swap_rule: $1" >/dev/console
	setup_parameters $1 apply
	#kill self in old position then insert to new position according to uciid
	orig_pos=`ht_get $HASHTABLE $1 IDX`
	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		$IPTABLES -t nat -D nat_portredirect $orig_pos
		$IPTABLES -t nat -D nat_portredirect2 $orig_pos
		$IPTABLES -t nat -I nat_portredirect $uciid $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
		$IPTABLES -t nat -I nat_portredirect2 $uciid $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j MASQUERADE
		if [ "$?" != "0" ] ;then
			echo "swap_rule($1): fails to swap profile($1), orig position($orig_pos)" >>$DEBUG_LOG
			logger "(WebUI load balance rule): fails to swap profile($1), orig position($orig_pos)"
			uci set $UCI_CONFIG.$1.status="disable"
			#use a dummy rule instead of failed case, otherwise sequence will be broken
			$IPTABLES -t nat -I nat_portredirect $uciid -u -j RETURN
			$IPTABLES -t nat -I nat_portredirect2 $uciid -u -j RETURN
		fi
	else
		$IPTABLES -t nat -D nat_portredirect $orig_pos
		$IPTABLES -t nat -D nat_portredirect2 $orig_pos
		$IPTABLES -t nat -I nat_portredirect $uciid -u -j RETURN
		$IPTABLES -t nat -I nat_portredirect2 $uciid -u -j RETURN
		uci set $UCI_CONFIG.$1.status="disable"
	fi
	#find which profile's original IDX is my current new position, do swapping IDX with it
	target_profile=$(ht_find_word_in_profiles $HASHTABLE IDX $uciid)
	if [ -n "$target_profile" ] ;then
		#In this case, $target_profile should contains only one
		test=`echo "$target_profile" |awk -F" " '{print $2}'`
		if [ -z "$test" ] ;then
			#it should be not myself
			if [ "$target_profile" != "$1" ] ;then
				#swap profile_backup
				ht_replace_entry $HASHTABLE $target_profile IDX $orig_pos
				ht_replace_entry $HASHTABLE $1 IDX $uciid
			fi
		fi
	fi
}

rename_rule() {
	#echo "rename_rule: $1" >/dev/console
	ht_rename_file $HASHTABLE $1 $2
}

replace_rule() {
	#echo "replace_rule: param1=$1" >/dev/console
	setup_parameters $1 apply
	old_private_ip=$(uci oget $UCI_CONFIG.$1.private_ip)
	ipset -D nat_ptre $old_private_ip 2>/dev/null
	if [ "$is_legal_rule" = "yes" -o "$is_legal_rule" = "105" ] ;then
		$IPTABLES -t nat -R nat_portredirect $uciid $flag $protocol $dip --dport $public_port -j DNAT --to-destination ${private_ip}${private_port2}
		$IPTABLES -t nat -R nat_portredirect2 $uciid $flag $protocol -m set --set lan_nat_subnet src -d $private_ip --dport $private_port -j MASQUERADE
		if [ "$flag" != "-u" ] ;then
			ipset -A nat_ptre $private_ip 2>/dev/null
		fi
	else
		$IPTABLES -t nat -R nat_portredirect $uciid -u -j RETURN
		$IPTABLES -t nat -R nat_portredirect2 $uciid -u -j RETURN
		uci set $UCI_CONFIG.$1.status="disable"
	fi
}

delete_rule() {
	#echo "delete_rule: $1" >/dev/console
	pos=`ht_get $HASHTABLE $1 IDX`
	ht_rm_file $HASHTABLE $1 2>/dev/null
	#echo "Deleted IDX=$pos" >/dev/console
	while true ;do
		target_profile=`ht_find_word_in_profiles $HASHTABLE IDX $(($pos+1))`
		if [ -n "$target_profile" ] ;then
			#echo "position ($pos+1) found=> renew IDX of profile:$target_profile to $pos" >/dev/console
			ht_replace_entry $HASHTABLE $target_profile IDX $pos
			pos=$(($pos+1))
		else
			#echo "position ($pos+1) not found, reach end => break" >/dev/console
			break
		fi
	done
	$IPTABLES -t nat -D nat_portredirect $uciid
	$IPTABLES -t nat -D nat_portredirect2 $uciid
	
	private_ip=$(uci oget $UCI_CONFIG.$1.private_ip)
	ipset -D nat_ptre $private_ip 2>/dev/null
}

boot() {
	touch $DEBUG_LOG
	start
}

start() {
	ht_init $HASHTABLE
	if [ -f "$BOOT_FILE" ] ;then
		rm -f $BOOT_FILE
		touch $BOOT_FILE
	else
		touch $BOOT_FILE
	fi
	echo "*nat" >> $BOOT_FILE
	fastboot=1
	boot_uciid_counter=$(($pre_rule+1))
	config_load $UCI_CONFIG
	config_foreach add_rule_boot
	echo "COMMIT" >> $BOOT_FILE
	iptables-restore -n < $BOOT_FILE 2>/dev/null
	#If iptables-restore fails, go back to regular boot
	[ "$?" != "0" ] && {
		echo "port redirection: fast boot failed, use regular boot" >>$DEBUG_LOG
		logger "port redirection: fast boot failed, use regular boot"
		fastboot=0
		boot_uciid_counter=$(($pre_rule+1))
		config_foreach add_rule_boot
	}
}

stop() {
	$IPTABLES -t nat -F nat_portredirect
	$IPTABLES -t nat -F nat_portredirect2
	ipset -F nat_ptre
}

apply_rule() {
	config_get uciaction $1 uciaction
	config_get uciid $1 uciid
	#rule index offset for the 1st rule of G36181 in boot()
	uciid=$(($uciid+$pre_rule))
	#echo "uciid: $uciid" >/dev/console
	
	case $uciaction in
		add)
			#echo "action: add" >/dev/console
			add_rule $1
			;;
		modify)
			#inspect what kind of modification is: position swapping or attribute changing
			action_type=`cat $UCI_TMP_PATH`
			#echo "action: modify, config:$1, action_type=$action_type" >/dev/console
			echo $action_type |grep '*' >/dev/null
			if [ "$?" = "0" ] ;then
				#a position swapping action
				#echo "Rule $1 SWAP to $uciid" >/dev/console
				swap_rule $1
			elif [ $(echo $action_type |grep '@') ] ;then
				#a profile renaming action
				old=`echo $action_type|sed 's/.*\.//'|awk '{FS="="; print $1}'`
				rename_rule $old $1 
			else
				#a normal attribute changing action
				#echo "Rule($1) REPLACE index:$uciid" >/dev/console
				replace_rule $1
			fi
			;;
		delete)
			#echo "action: delete" >/dev/console
			delete_rule $1
			;;
	esac
	#Notice: Here flush ALL route cache
		#Todo: flush entry according to the given private IP
	/usr/sbin/flush_route_cache.sh "port_redirection"
	#echo "clear IP:$private_ip conntracks" >/dev/console
	/usr/sbin/conntrack -D -s $private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -d $private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -r $private_ip >/dev/null 2>/dev/null
	/usr/sbin/conntrack -D -q $private_ip >/dev/null 2>/dev/null
}

apply() {
	lock $SCRIPT_LOCK
	#echo -n "firwall apply start "  >>/tmp/apply_dur.log 2>&1
	#cat /proc/uptime >> /tmp/apply_dur.log 2>&1
	tmpcfg=${UCI_CONFIG}_`cat /proc/uptime | cut -d. -f 1`
	uci fchanges export $UCI_CONFIG > /etc/config/$tmpcfg
	config_load $tmpcfg
	rm -f /etc/config/$tmpcfg
	is_change=`uci fchanges all $UCI_CONFIG`
	
	config_foreach apply_rule
	
	[ -n "$is_change" ] && uci commit $UCI_CONFIG
	lock -u $SCRIPT_LOCK
	
	#Handle return error code
	if [ "$is_legal_rule" != "yes" ] ;then
		exit $is_legal_rule
	fi
	#echo -n "firwall apply end "  >>/tmp/apply_dur.log 2>&1
	#cat /proc/uptime >> /tmp/apply_dur.log 2>&1
}