/*
 *	Forwarding decision
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/err.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/netpoll.h>
#include <linux/skbuff.h>
#include <linux/if_vlan.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"

static int deliver_clone(const struct net_bridge_port *prev,
			 struct sk_buff *skb,
			 void (*__packet_hook)(const struct net_bridge_port *p,
					       struct sk_buff *skb));

/* Don't forward packets to originating port or forwarding diasabled */
static inline int should_deliver(const struct net_bridge_port *p,
				 const struct sk_buff *skb)
{
	return (((p->flags & BR_HAIRPIN_MODE) || skb->dev != p->dev) &&
		p->state == BR_STATE_FORWARDING);
}

static inline unsigned packet_length(const struct sk_buff *skb)
{
	return skb->len - (skb->protocol == htons(ETH_P_8021Q) ? VLAN_HLEN : 0);
}

int br_dev_queue_push_xmit(struct sk_buff *skb)
{
	/* drop mtu oversized packets except gso */
	if (packet_length(skb) > skb->dev->mtu && !skb_is_gso(skb))
		kfree_skb(skb);
	else {
		/* ip_fragment doesn't copy the MAC header */
		if (nf_bridge_maybe_copy_header(skb))
			kfree_skb(skb);
		else {
			skb_push(skb, ETH_HLEN);
			dev_queue_xmit(skb);
		}
	}

	return 0;
}

int br_forward_finish(struct sk_buff *skb)
{
	return NF_HOOK(NFPROTO_BRIDGE, NF_BR_POST_ROUTING, skb, NULL, skb->dev,
		       br_dev_queue_push_xmit);

}

static void __br_deliver(const struct net_bridge_port *to, struct sk_buff *skb)
{
	skb->dev = to->dev;

	if (unlikely(netpoll_tx_running(to->dev))) {
		if (packet_length(skb) > skb->dev->mtu && !skb_is_gso(skb))
			kfree_skb(skb);
		else {
			skb_push(skb, ETH_HLEN);
			br_netpoll_send_skb(to, skb);
		}
		return;
	}

	NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_OUT, skb, NULL, skb->dev,
		br_forward_finish);
}

//draytek function

#if CONFIG_DRAYTEK_SWITCH_VLAN_MODULE
struct vlan_config_args vlan_config={0};
int is_switch_vlan_enable(void)
{
	return vlan_config.flag;
}

int get_switch_vlan_tag(unsigned int gp_num)
{
	return vlan_config.vlan_tag[gp_num];
}

int get_switch_vlan_portmap(unsigned int gp_num)
{
	return vlan_config.vlan_portmap[gp_num];
}

int set_switch_vlan_portmap(unsigned int gp_num, unsigned int portmap)
{
	vlan_config.vlan_portmap[gp_num] = portmap;
}

int get_switch_vlan_tag_lv(unsigned int sw_port)
{
	return vlan_config.vlan_tag_lv[sw_port];
}

int set_switch_vlan_tag_lv(unsigned int sw_port, unsigned int level)
{
	vlan_config.vlan_tag_lv[sw_port] = level;
}
EXPORT_SYMBOL(is_switch_vlan_enable);
EXPORT_SYMBOL(get_switch_vlan_tag);
EXPORT_SYMBOL(get_switch_vlan_portmap);
EXPORT_SYMBOL(set_switch_vlan_portmap);
EXPORT_SYMBOL(get_switch_vlan_tag_lv);
EXPORT_SYMBOL(set_switch_vlan_tag_lv);

int drop_diff_tag(char *dst, struct sk_buff *skb)
{
    unsigned char *src=skb->dev->name;
    unsigned char *chk=NULL;
    char *src_num=NULL, *dst_num=NULL, temp[8]={0};
    int rn=0;

    if( (*src == 'e')&& (*(src+4) == '.') && (*dst == 'e') && (*(dst+4) == '.') ){
		
		src_num=(strchr(src, '.')+1);
		dst_num=(strchr(dst, '.')+1);
		
		if(!strcmp(src_num, dst_num)){
		    //printk("PASS,%s -> %s, rn=%d, src=%s, dst=%s\n",skb->dev->name, dst, rn, src_num,dst_num);
			return 0;
		}else{
		    //printk("DROP, %s -> %s, rn=%d, src=%s, dst=%s\n",skb->dev->name, dst, rn, src_num,dst_num);
			return -1;
		}

    }
}


int chk_vlan_wl_flag(char *dst, struct sk_buff *skb)
{
    unsigned char *src=skb->dev->name;
    unsigned char *chk=NULL;
    char *ra_num=NULL, *eth_num=NULL, temp[8]={0};
    int rn=0, gp_num;

	
    if( ((*src == 'r') || (*(src) == 'w')) && ((*dst == 'e') && (*(dst+4) == '.')) ){

		eth_num = (strchr(dst, '.')+1);
		if(*src == 'r'){
			if(*(src+2)!='i'){
	      		rn=*(src+2) - '0'; // for 2.4G rax
		  		gp_num = vlan_config.vlan_wl_flag[rn];
			}else{
	      		rn=*(src+3) - '0'; // for 5G raix
		  		gp_num = vlan_config.vlan_dual_wl_flag[rn];
			}
		}else{
        	rn=*(src+3) - '0'; // for wds wdsx
	  		gp_num = vlan_config.vlan_wds_flag[rn];
		}
		
		/* 
		/  check which vlan group rax want to go. If vlan_wl_flag is '0', this 
		/  rax is not enabled vlan function. 
		/  If eth.xxx that xxx is different with vlan id, we will drop it .
		*/
		if(0!=gp_num){
			sprintf(temp, "%d", vlan_config.vlan_tag[gp_num-1]);
			if(!strcmp(temp, eth_num)){
			    printk("PASS,%s -> %s, rn=%d, en=%s\r\n",skb->dev->name, dst, rn, eth_num);
				return 0;
			}else{
			    printk("DROP, %s -> %s, rn=%d, en=%s\r\n\r\n",skb->dev->name, dst, rn, eth_num);
				return -1;
			}
		}else
			return 0;
    }
	return 0;
}
#endif

struct vlan_transparent vlan_trans={0}; 
//bridge forward add flag Bruce Hsu
//we mark some pkt which needs to add vlan tags,
//and we only care about ra(X) --> eth(Y) pkt,
//the opposite will be transfer by wireless driver. (i guess...)
void br_add_flag(char *dst, struct sk_buff *skb)
{
    unsigned char *src=skb->dev->name;
    unsigned char *chk=NULL;
    char *ra_num=NULL;
    int rn=0;
    char **endptr;

                        /*    Function flag
                         *      - bit 0 --> ra0 action
                         *      - bit 1 --> ra1 action
                         *      - bit 2 --> ra2 action
                         *      - bit 3 --> ra3 action
                         *      - bit 4 --> rai0 action
                         *      - bit 5 --> rai1 action
                         *      - bit 6 --> rai2 action
                         *      - bit 7 --> rai3 action
                         */    
    if(vlan_trans.flag==0){   //if we do not enable any function, quick pass!!    
        return;
    }

    if( (*src == 'r') && (*(src+1) == 'a') && (*dst == 'e') && (*(dst+1) == 't') && (*(dst+2) == 'h') ){
        //we should check this path...
        if(*(src+2) == 'i'){    //5G pkt
            ra_num=src+3;     //skip "rai"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(3>=rn)){   //rn should be 0~3
                if(vlan_trans.ravid[rn+4] == 0){  //this rai do not need tag...
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0xa5;  //magic bytes, means need action
                *(chk+1)=rn+4;    //mark rai number = rai channel + 4
                //printk("TagRai: %02X %02X\n",*chk,*(chk+1));
            }
        }else{  //2.4G pkt
            ra_num=src+2;     //skip "ra"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(3>=rn)){   //rn should be 0~3
                if(vlan_trans.ravid[rn] == 0){  //this ra do not need tag...
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0xa5;  //magic bytes, means need action
                *(chk+1)=rn;    //mark ra number
                //printk("TagRa: %02X %02X\n",*chk,*(chk+1));
            }
        }
    }else if( (*src == 'e') && (*(src+1) == 't') && (*(src+2) == 'h') && (*dst == 'r') && (*(dst+1) == 'a') ){
        // Check path:  eth -> ra/rai
        if(*(dst+2) == 'i'){    //5G pkt
            ra_num=dst+3;     //skip "rai"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(3>=rn)){   //rn should be 0~3
                if(vlan_trans.ravid[rn+4] == 0){  //this rai do not need tag...
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0x5a;  //magic bytes, means need action
                *(chk+1)=rn+4;    //mark rai number = rai channel + 4
                //printk("CheckRai: %02X %02X\n",*chk,*(chk+1));
            }
        }else{  //2.4G pkt
            ra_num=dst+2;     //skip "ra"
            rn=*ra_num - '0';    //get ra number, i can't use atoi() here...
            if((rn>=0)&&(3>=rn)){   //rn should be 0~3
                if(vlan_trans.ravid[rn] == 0){  //this ra do not need tag...
                	chk=skb->head;
					*chk=0x00; // fanny: don't care about the value. Wish check_vlan_tag() would discard this packet.
                    return;
                }
                chk=skb->head;  //add 2 bytes at head
                *chk=0x5a;  //magic bytes, means need action
                *(chk+1)=rn;    //mark ra number
                //printk("CheckRu: %02X %02X\n",*chk,*(chk+1));
            }
        }
    }

  return;
}

#define RTMP_SET_PACKET_BR_NUM(_p, _num)	(((struct sk_buff *)_p)->cb[47] = _num)
#define RTMP_GET_PACKET_BR_NUM(_p)			(((struct sk_buff *)_p)->cb[47])
static void br_add_interface_num (const struct net_bridge_port *to, struct sk_buff *skb)
{
    RTMP_SET_PACKET_BR_NUM(skb, 0);
    if (to->br && to->br->dev) {
        unsigned char *name = to->br->dev->name;
        if (name[0] == 'b' && name[1] == 'r') {
            /* br0 - 1, br1 - 2 */
            RTMP_SET_PACKET_BR_NUM(skb, (name[2] - 0x2F));
            //printk("br_add_interface_num: name = %s, idx = %d\n", to->br->dev, RTMP_GET_PACKET_BR_NUM(skb));
        }   
    }
        
}

static void __br_forward(const struct net_bridge_port *to, struct sk_buff *skb)
{
	struct net_device *indev;

	if (skb_warn_if_lro(skb)) {
		kfree_skb(skb);
		return;
	}
#if CONFIG_DRAYTEK_SWITCH_VLAN_MODULE
	if(is_switch_vlan_enable()==1){
	    if(-1==drop_diff_tag(to->dev->name, skb)){
			kfree_skb(skb);
	  		return;
		}

	    if(-1==chk_vlan_wl_flag(to->dev->name, skb)){
			kfree_skb(skb);
	  		return;
		}
	}
#endif

    br_add_flag(to->dev->name, skb);    //bruce

	indev = skb->dev;
	skb->dev = to->dev;
	skb_forward_csum(skb);

	NF_HOOK(NFPROTO_BRIDGE, NF_BR_FORWARD, skb, indev, skb->dev,
		br_forward_finish);
}

/* called with rcu_read_lock */
void br_deliver(const struct net_bridge_port *to, struct sk_buff *skb)
{
	if (should_deliver(to, skb)) {
		__br_deliver(to, skb);
		return;
	}

	kfree_skb(skb);
}

/* called with rcu_read_lock */
void br_forward(const struct net_bridge_port *to, struct sk_buff *skb, struct sk_buff *skb0)
{
	if (should_deliver(to, skb)) {
		if (skb0)
			deliver_clone(to, skb, __br_forward);
		else
			__br_forward(to, skb);
		return;
	}

	if (!skb0)
		kfree_skb(skb);
}

static int deliver_clone(const struct net_bridge_port *prev,
			 struct sk_buff *skb,
			 void (*__packet_hook)(const struct net_bridge_port *p,
					       struct sk_buff *skb))
{
	struct net_device *dev = BR_INPUT_SKB_CB(skb)->brdev;

	skb = skb_clone(skb, GFP_ATOMIC);
	if (!skb) {
		dev->stats.tx_dropped++;
		return -ENOMEM;
	}

	__packet_hook(prev, skb);
	return 0;
}

static struct net_bridge_port *maybe_deliver(
	struct net_bridge_port *prev, struct net_bridge_port *p,
	struct sk_buff *skb,
	void (*__packet_hook)(const struct net_bridge_port *p,
			      struct sk_buff *skb))
{
	int err;

	if (!should_deliver(p, skb))
		return prev;

	if (!prev)
		goto out;

	err = deliver_clone(prev, skb, __packet_hook);
	if (err)
		return ERR_PTR(err);

out:
	return p;
}

/* called under bridge lock */
static void br_flood(struct net_bridge *br, struct sk_buff *skb,
		     struct sk_buff *skb0,
		     void (*__packet_hook)(const struct net_bridge_port *p,
					   struct sk_buff *skb))
{
	struct net_bridge_port *p;
	struct net_bridge_port *prev;

	prev = NULL;

	list_for_each_entry_rcu(p, &br->port_list, list) {
		prev = maybe_deliver(prev, p, skb, __packet_hook);
		if (IS_ERR(prev))
			goto out;
	}

	if (!prev)
		goto out;

	if (skb0)
		deliver_clone(prev, skb, __packet_hook);
	else
		__packet_hook(prev, skb);
	return;

out:
	if (!skb0)
		kfree_skb(skb);
}


/* called with rcu_read_lock */
void br_flood_deliver(struct net_bridge *br, struct sk_buff *skb)
{
	br_flood(br, skb, NULL, __br_deliver);
}

/* called under bridge lock */
void br_flood_forward(struct net_bridge *br, struct sk_buff *skb,
		      struct sk_buff *skb2)
{
	br_flood(br, skb, skb2, __br_forward);
}

#ifdef CONFIG_BRIDGE_IGMP_SNOOPING
/* called with rcu_read_lock */
static void br_multicast_flood(struct net_bridge_mdb_entry *mdst,
			       struct sk_buff *skb, struct sk_buff *skb0,
			       void (*__packet_hook)(
					const struct net_bridge_port *p,
					struct sk_buff *skb))
{
	struct net_device *dev = BR_INPUT_SKB_CB(skb)->brdev;
	struct net_bridge *br = netdev_priv(dev);
	struct net_bridge_port *prev = NULL;
	struct net_bridge_port_group *p;
	struct hlist_node *rp;

	rp = rcu_dereference(br->router_list.first);
	p = mdst ? rcu_dereference(mdst->ports) : NULL;
	while (p || rp) {
		struct net_bridge_port *port, *lport, *rport;

		lport = p ? p->port : NULL;
		rport = rp ? hlist_entry(rp, struct net_bridge_port, rlist) :
			     NULL;

		port = (unsigned long)lport > (unsigned long)rport ?
		       lport : rport;

		prev = maybe_deliver(prev, port, skb, __packet_hook);
		if (IS_ERR(prev))
			goto out;

		if ((unsigned long)lport >= (unsigned long)port)
			p = rcu_dereference(p->next);
		if ((unsigned long)rport >= (unsigned long)port)
			rp = rcu_dereference(rp->next);
	}

	if (!prev)
		goto out;

	if (skb0)
		deliver_clone(prev, skb, __packet_hook);
	else
		__packet_hook(prev, skb);
	return;

out:
	if (!skb0)
		kfree_skb(skb);
}

/* called with rcu_read_lock */
void br_multicast_deliver(struct net_bridge_mdb_entry *mdst,
			  struct sk_buff *skb)
{
	br_multicast_flood(mdst, skb, NULL, __br_deliver);
}

/* called with rcu_read_lock */
void br_multicast_forward(struct net_bridge_mdb_entry *mdst,
			  struct sk_buff *skb, struct sk_buff *skb2)
{
	br_multicast_flood(mdst, skb, skb2, __br_forward);
}
#endif
