/* Copyright (C) 2000-2002 Joakim Axelsson <gozem@linux.nu>
 *                         Patrick Schaaf <bof@bof.de>
 *                         Martin Josefsson <gandalf@wlug.westbo.se>
 * Copyright (C) 2003-2004 Jozsef Kadlecsik <kadlec@blackhole.kfki.hu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  
 */

/* Shared library add-on to iptables to add IP set matching. */
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <ctype.h>
#include <errno.h>

#include <iptables.h>
#include <linux/netfilter_ipv4/ipt_set.h>
#include "libipt_set.h"

/* Function which prints out usage message. */
static void mset2_help(void)
{
	printf("set v%s options:\n"
	       " [!] --set2     name flags\n"
	       "		'name' is the set name from to match,\n" 
	       "		'flags' are the comma separated list of\n"
	       "		'src', 'dst', 'log', and 'debug'.\n"
	       "\n", IPTABLES_VERSION);
}

static const struct option mset2_opts[] = {
	{"set2", 1, 0, '1'},
	{0}
};

/* Initialize the match. */
static void mset2_init(struct xt_entry_match *match)
{
	struct ipt_mset_info_match *info = 
		(struct ipt_mset_info_match *) match->data;
	int i;
	memset(info, 0, sizeof(struct ipt_mset_info_match));
	for(i = 0; i < MAXSET; i++) {
		info->match_set[i].index = IP_SET_INVALID_ID;
	}
}

/* Function which parses command options; returns true if it ate an option */
static int mset2_parse(int c, char **argv, int invert, unsigned int *flags,
                     const void *entry, struct xt_entry_match **match)
{
	struct ipt_mset_info_match *myinfo = 
		(struct ipt_mset_info_match *) (*match)->data;
	struct ipt_set_info *info;
	
	if (*flags >= MAXSET) {
		exit_error(PARAMETER_PROBLEM,
			"max set number is %d\n", MAXSET);
	}
	
	info = &myinfo->match_set[*flags];
	switch (c) {
	case '1':		/* --set2 <set> <flag>[,<flag> */
		if (info->flags[0])
			exit_error(PARAMETER_PROBLEM,
				   "--set2 can be specified only once");

		check_inverse(optarg, &invert, &optind, 0);
		if (invert)
			info->flags[0] |= IPSET_MATCH_INV;

		if (!argv[optind]
		    || argv[optind][0] == '-'
		    || argv[optind][0] == '!')
			exit_error(PARAMETER_PROBLEM,
				   "--set2 requires two args.");

		if (strlen(argv[optind-1]) > IP_SET_MAXNAMELEN - 1)
			exit_error(PARAMETER_PROBLEM,
				   "setname `%s' too long, max %d characters.",
				   argv[optind-1], IP_SET_MAXNAMELEN - 1);

		get_set_byname(argv[optind - 1], info);
		parse_bindings(argv[optind], info);
		DEBUGP("parse: set index %u\n", info->index);
		optind++;
		
		*flags = *flags + 1;
		break;

	default:
		return 0;
	}

	return 1;
}

/* Final check; must have specified --set. */
static void mset2_check(unsigned int flags)
{
	if (!flags)
		exit_error(PARAMETER_PROBLEM,
			   "You must specify `--set2' with proper arguments");
	DEBUGP("final check OK\n");
}

static void
print_match(const char *prefix, const struct ipt_set_info *info)
{
	int i;
	char setname[IP_SET_MAXNAMELEN];

	get_set_byid(setname, info->index);
	printf("%s%s %s", 
	       (info->flags[0] & IPSET_MATCH_INV) ? "! " : "",
	       prefix,
	       setname); 
	for (i = 0; i < IP_SET_MAX_BINDINGS; i++) {
		if (!info->flags[i])
			break;		
		printf("%s%s",
		       i == 0 ? " " : ",",
		       info->flags[i] & IPSET_SRC ? "src" : "dst");
	}
	if (info->flags[0] & IPSET_MATCH_LOG)
		printf(",log");
	if (info->flags[0] & IPSET_MATCH_DEBUG)
		printf(",debug");	
	if (info->flags[0] & IPSET_MATCH_REJ)
		printf(",rej");	
	if (info->flags[0] & IPSET_MATCH_PASS)
		printf(",pass");	
	printf(" ");
}

/* Prints out the matchinfo. */
static void mset2_print(const void *ip, const struct xt_entry_match *match,
                      int numeric)
{
	struct ipt_mset_info_match *info = 
		(struct ipt_mset_info_match *) match->data;
	int i;
	for(i = 0; i < MAXSET; i++) {
		if (info->match_set[i].index < IP_SET_INVALID_ID)
			print_match("set2", &info->match_set[i]);
		else
			break;
	}
}

/* Saves the matchinfo in parsable form to stdout. */
static void mset2_save(const void *ip, const struct xt_entry_match *match)
{
	struct ipt_mset_info_match *info = 
		(struct ipt_mset_info_match *) match->data;
	int i;
	for(i = 0; i < MAXSET; i++) {
		if (info->match_set[i].index < IP_SET_INVALID_ID)
			print_match("--set2", &info->match_set[i]);
		else
			break;
	}
}

static struct iptables_match mset2_match = {
	.name		= "mset2",
	.version	= IPTABLES_VERSION,
	.size		= IPT_ALIGN(sizeof(struct ipt_mset_info_match)),
	.userspacesize	= IPT_ALIGN(sizeof(struct ipt_mset_info_match)),
	.help		= mset2_help,
	.init		= mset2_init,
	.parse		= mset2_parse,
	.final_check	= mset2_check,
	.print		= mset2_print,
	.save		= mset2_save,
	.extra_opts	= mset2_opts,
};

void _init(void)
{
	register_match(&mset2_match);
}
