/*
 * USB Serial Converter Generic functions
 *
 * Copyright (C) 1999 - 2002 Greg Kroah-Hartman (greg@kroah.com)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License version
 *	2 as published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/usb.h>
#include <linux/usb/serial.h>
#include <linux/usb/cdc.h>
#include <asm/uaccess.h>


static int debug;

#ifdef CONFIG_USB_SERIAL_GENERIC

static int generic_probe(struct usb_interface *interface,
			 const struct usb_device_id *id);

static __u16 vendor  = 0x05f9;
static __u16 product = 0xffff;

module_param(vendor, ushort, 0);
MODULE_PARM_DESC(vendor, "User specified USB idVendor");

module_param(product, ushort, 0);
MODULE_PARM_DESC(product, "User specified USB idProduct");

/* Define the specific features of Huawei E398 */
struct exp_e398_device {
	__u16 vid;	// vendor identifier
	__u16 pid; 	// product identifier 
	__u16 prot; // interface protocl	
};

static struct exp_e398_device exp_e398 = { 0x12D1, 0x1506, 0x10 };	// Huawei E398

//static struct usb_device_id generic_device_ids[2]; /* Initially all zeroes. */
//id list changed by bruce hsu   //2009/10/05
/* Vendor and product IDs */
#define OPTION_VENDOR_ID			0x0AF0
#define OPTION_PRODUCT_COLT			0x5000
#define OPTION_PRODUCT_RICOLA			0x6000
#define OPTION_PRODUCT_RICOLA_LIGHT		0x6100
#define OPTION_PRODUCT_RICOLA_QUAD		0x6200
#define OPTION_PRODUCT_RICOLA_QUAD_LIGHT	0x6300
#define OPTION_PRODUCT_RICOLA_NDIS		0x6050
#define OPTION_PRODUCT_RICOLA_NDIS_LIGHT	0x6150
#define OPTION_PRODUCT_RICOLA_NDIS_QUAD		0x6250
#define OPTION_PRODUCT_RICOLA_NDIS_QUAD_LIGHT	0x6350
#define OPTION_PRODUCT_COBRA			0x6500
#define OPTION_PRODUCT_COBRA_BUS		0x6501
#define OPTION_PRODUCT_VIPER			0x6600
#define OPTION_PRODUCT_VIPER_BUS		0x6601
#define OPTION_PRODUCT_GT_MAX_READY		0x6701
#define OPTION_PRODUCT_GT_MAX			0x6711
#define OPTION_PRODUCT_FUJI_MODEM_LIGHT		0x6721
#define OPTION_PRODUCT_FUJI_MODEM_GT		0x6741
#define OPTION_PRODUCT_FUJI_MODEM_EX		0x6761
#define OPTION_PRODUCT_FUJI_NETWORK_LIGHT	0x6731
#define OPTION_PRODUCT_FUJI_NETWORK_GT		0x6751
#define OPTION_PRODUCT_FUJI_NETWORK_EX		0x6771
#define OPTION_PRODUCT_KOI_MODEM		0x6800
#define OPTION_PRODUCT_KOI_NETWORK		0x6811
#define OPTION_PRODUCT_SCORPION_MODEM		0x6901
#define OPTION_PRODUCT_SCORPION_NETWORK		0x6911
#define OPTION_PRODUCT_ETNA_MODEM		0x7001
#define OPTION_PRODUCT_ETNA_NETWORK		0x7011
#define OPTION_PRODUCT_ETNA_MODEM_LITE		0x7021
#define OPTION_PRODUCT_ETNA_MODEM_GT		0x7041
#define OPTION_PRODUCT_ETNA_MODEM_EX		0x7061
#define OPTION_PRODUCT_ETNA_NETWORK_LITE	0x7031
#define OPTION_PRODUCT_ETNA_NETWORK_GT		0x7051
#define OPTION_PRODUCT_ETNA_NETWORK_EX		0x7071
#define OPTION_PRODUCT_ETNA_KOI_MODEM		0x7100
#define OPTION_PRODUCT_ETNA_KOI_NETWORK		0x7111

#define HUAWEI_VENDOR_ID			0x12D1
#define HUAWEI_PRODUCT_E600			0x1001
#define HUAWEI_PRODUCT_E220			0x1003
#define HUAWEI_PRODUCT_E220BIS			0x1004

#define NOVATELWIRELESS_VENDOR_ID		0x1410
#define DELL_VENDOR_ID				0x413C

#define ANYDATA_VENDOR_ID			0x16d5
#define ANYDATA_PRODUCT_ADU_E100A		0x6501
#define ANYDATA_PRODUCT_ADU_500A		0x6502

#define BANDRICH_VENDOR_ID			0x1A8D
#define BANDRICH_PRODUCT_C100_1			0x1002
#define BANDRICH_PRODUCT_C100_2			0x1003

static struct usb_device_id generic_device_ids[] = {
    { USB_DEVICE(0x0323, 0xffff) },     // This ID doesn't existed, it's used for user setting.   //bruce hsu
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_COLT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_LIGHT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_QUAD) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_QUAD_LIGHT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_NDIS) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_NDIS_LIGHT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_NDIS_QUAD) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_RICOLA_NDIS_QUAD_LIGHT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_COBRA) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_COBRA_BUS) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_VIPER) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_VIPER_BUS) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_GT_MAX_READY) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_GT_MAX) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_FUJI_MODEM_LIGHT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_FUJI_MODEM_GT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_FUJI_MODEM_EX) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_FUJI_NETWORK_LIGHT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_FUJI_NETWORK_GT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_FUJI_NETWORK_EX) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_KOI_MODEM) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_KOI_NETWORK) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_SCORPION_MODEM) },
	//{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_SCORPION_NETWORK) },	// hso driver
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_MODEM) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_NETWORK) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_MODEM_LITE) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_MODEM_GT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_MODEM_EX) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_NETWORK_LITE) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_NETWORK_GT) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_NETWORK_EX) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_KOI_MODEM) },
	{ USB_DEVICE(OPTION_VENDOR_ID, OPTION_PRODUCT_ETNA_KOI_NETWORK) },
	{ USB_DEVICE(HUAWEI_VENDOR_ID, HUAWEI_PRODUCT_E600) },
	{ USB_DEVICE(HUAWEI_VENDOR_ID, HUAWEI_PRODUCT_E220) },
	{ USB_DEVICE(HUAWEI_VENDOR_ID, HUAWEI_PRODUCT_E220BIS) },
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1100) }, /* Novatel Merlin XS620/S640 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1110) }, /* Novatel Merlin S620 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1120) }, /* Novatel Merlin EX720 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1130) }, /* Novatel Merlin S720 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1400) }, /* Novatel U730 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1410) }, /* Novatel U740 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1420) }, /* Novatel EU870 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1430) }, /* Novatel Merlin XU870 HSDPA/3G */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x1430) }, /* Novatel XU870 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x2100) }, /* Novatel EV620 CDMA/EV-DO */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x2110) }, /* Novatel Merlin ES620 / Merlin ES720 / Ovation U720 */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x2130) }, /* Novatel Merlin ES620 SM Bus */
	{ USB_DEVICE(NOVATELWIRELESS_VENDOR_ID, 0x2410) }, /* Novatel EU740 */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8114) },	/* Dell Wireless 5700 Mobile Broadband CDMA/EVDO Mini-Card == Novatel Expedite EV620 CDMA/EV-DO */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8115) },	/* Dell Wireless 5500 Mobile Broadband HSDPA Mini-Card == Novatel Expedite EU740 HSDPA/3G */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8116) },	/* Dell Wireless 5505 Mobile Broadband HSDPA Mini-Card == Novatel Expedite EU740 HSDPA/3G */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8117) },	/* Dell Wireless 5700 Mobile Broadband CDMA/EVDO ExpressCard == Novatel Merlin XV620 CDMA/EV-DO */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8118) },	/* Dell Wireless 5510 Mobile Broadband HSDPA ExpressCard == Novatel Merlin XU870 HSDPA/3G */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8128) },	/* Dell Wireless 5700 Mobile Broadband CDMA/EVDO Mini-Card == Novatel Expedite E720 CDMA/EV-DO */
	{ USB_DEVICE(DELL_VENDOR_ID, 0x8137) },	/* Dell Wireless HSDPA 5520 */
	{ USB_DEVICE(ANYDATA_VENDOR_ID, ANYDATA_PRODUCT_ADU_E100A) },
	{ USB_DEVICE(ANYDATA_VENDOR_ID, ANYDATA_PRODUCT_ADU_500A) },
	{ USB_DEVICE(BANDRICH_VENDOR_ID, BANDRICH_PRODUCT_C100_1) },
	{ USB_DEVICE(BANDRICH_VENDOR_ID, BANDRICH_PRODUCT_C100_2) },
	{ USB_DEVICE(0x0403, 0x6001) }, //Spincom 56k modem
	{ USB_DEVICE(0x0403, 0x6002) }, // gprs modem
	{ USB_DEVICE(0x0408, 0xEA02) }, //MU-Q101 , modem mode
	{ USB_DEVICE(0x0408, 0xEA03) }, //Royal Q110, modem mode
	{ USB_DEVICE(0x0408, 0xF000) }, //Royal Q110, stardard mode, not quick launch
	{ USB_DEVICE(0x04A5, 0x4068) }, //DET-US3G5
	{ USB_DEVICE(0x0572, 0x1300) }, //HSF 56K modem (conexant)
	{ USB_DEVICE(0x0572, 0x1329) }, //HUC56S 56K modem (conexant)
    { USB_DEVICE(0x05c6, 0x9000) }, /* Telit U9*/
	{ USB_DEVICE(0x07D1, 0x3E01) }, //DLink DWM 152 modem mode
	{ USB_DEVICE(0x07D1, 0x3E02) }, //DLink DWM 156 modem mode
	{ USB_DEVICE(0x07D1, 0x7E11) }, //D-Link DWM-156 HSUPA 3.75G USB Modem
	{ USB_DEVICE(0x0AF0, 0x6501) }, //Option N.V. Globe Trotter 3G+ Module
	{ USB_DEVICE(0x0AF0, 0x6600) }, //Option N.V. Globe Trotter 3G+
	{ USB_DEVICE(0x0AF0, 0x6901) }, //Option ICon 7.2 GIO201, modem mode
	{ USB_DEVICE(0x0AF0, 0x7401) }, //Optioin Icon 401, HSO driver
	{ USB_DEVICE(0x0AF0, 0x7501) }, //Vodafone Option Icon K3760, HSO driver
	{ USB_DEVICE(0x0AF0, 0xD055) }, //option icon 505
	{ USB_DEVICE(0x0F3D, 0x68AA) }, /* Sierra 320U Direct IP LTE modems */
	{ USB_DEVICE(0x0fce, 0xd0cf) }, /* Sony Ericson MD300 */
    { USB_DEVICE(0x1186, 0x3E04) }, //DLink DWM-652
	{ USB_DEVICE(0x1199, 0x0023) }, //Sierra Compass 597 EVDO Modem
	{ USB_DEVICE(0x1199, 0x0025) }, //Sierra 598 Modem
	{ USB_DEVICE(0x1199, 0x6812) }, //Sierra 875U
	{ USB_DEVICE(0x1199, 0x683C) }, //Zadacom 3G USB modem mode
	{ USB_DEVICE(0x1199, 0x6856) }, //Sierra 881U - AU
	{ USB_DEVICE(0x1199, 0x6880) }, //Sierra 885
    { USB_DEVICE(0x1199, 0x68a3) }, /* Sierra312U, Sierra C889*/
	{ USB_DEVICE(0x1199, 0x6890) }, /* Sierra C888*/
	{ USB_DEVICE(0x1199, 0x6855) }, /* SierraAC880U*/
	{ USB_DEVICE(0x12D1, 0x1001) }, //Huawei Mobile E169G USB Modem
	{ USB_DEVICE(0x12D1, 0x1003) }, //Huawei Mobile E220/E230 USB Modem
	{ USB_DEVICE(0x12D1, 0x140B) }, //Huawei Mobile EC1260 USB Modem (Reliance)
	{ USB_DEVICE(0x12D1, 0x140C) }, //Huawei Mobile 169U/160E USB Modem
	{ USB_DEVICE(0x12D1, 0x1411) }, //Huawei EC121 USB Modem
	{ USB_DEVICE(0x12D1, 0x1412) }, //Huawei Mobile 168c USB Modem
	{ USB_DEVICE(0x12D1, 0x1446) }, //Huawei Mobile E169U Storage Mode
	{ USB_DEVICE(0x12D1, 0x1464) }, //vodafone k4505 modem
	{ USB_DEVICE(0x12D1, 0x1465) }, //vodafone Huawei K3765 modem mode
	{ USB_DEVICE(0x12D1, 0x14ac) }, //Huawei Mobile 1820 modem
	{ USB_DEVICE(0x12D1, 0x1506) }, //Huawei Mobile E367 modem
	{ USB_DEVICE(0x12D1, 0x14C6) }, //Vodafone K4605
	{ USB_DEVICE(0x12D1, 0x1520) }, //vodafone Huawei K3765 storage mode
	{ USB_DEVICE(0x12D1, 0x1521) }, //vodafone k4505 storage
	{ USB_DEVICE(0x12d1, 0x141b) }, /* HUAWEI E1752 */
    { USB_DEVICE(0x12d1, 0x1c05) }, /* HUAWEI E173 */
    { USB_DEVICE(0x12d1, 0x1c08) }, /* HUAWEI E173 */
    { USB_DEVICE(0x1410, 0x4100) }, //Novatel Ovation MC727
	{ USB_DEVICE(0x1410, 0x4400) }, //Novatel HSUPA Modem (Ovation MC950D), Modem Mode
	{ USB_DEVICE(0x1410, 0x6000) }, //Novatel Ovation MC760 modem
	{ USB_DEVICE(0x1410, 0x7001) }, //Ovation MC990D, Modem Mode
	{ USB_DEVICE(0x1410, 0x6002) }, /* Novatel MC760 */
	{ USB_DEVICE(0x1614, 0x0800) }, //AMOI H01
	{ USB_DEVICE(0x16D5, 0x6502) }, //AnyDATA
	{ USB_DEVICE(0x16D8, 0x5533) }, //C-motech_CCU-550
	{ USB_DEVICE(0x16D8, 0x6280) }, //Maxon BP3-USB (CMOTECH) modem, AU Bigpond Next G Network
	{ USB_DEVICE(0x16D8, 0x6803) }, //C-Motech D-50 , storage mode
	{ USB_DEVICE(0x16D8, 0xF000) }, //Gmbh 4G systems(XS Stick W12)
	{ USB_DEVICE(0x19D2, 0x0001) }, //ZTE MF622
	{ USB_DEVICE(0x19D2, 0x0003) }, //ZTE MU318,MU350
	{ USB_DEVICE(0x19D2, 0x0015) }, //ZTE MF628
	{ USB_DEVICE(0x19D2, 0x0016) }, //ZTE MF639, ZTE MF662
	{ USB_DEVICE(0x19d2, 0x0017) }, /* ZTE MF100 */
	{ USB_DEVICE(0x19D2, 0x0003) }, //ZTE MU318
	{ USB_DEVICE(0x19D2, 0x0026) }, //ZTE AC560 Storage mode
	{ USB_DEVICE(0x19D2, 0x0073) }, //ZTE AC560 modem mode
	{ USB_DEVICE(0x19D2, 0x0031) }, //ZTE MF626/636 Modem mode
	{ USB_DEVICE(0x19D2, 0x0033) }, //UK ZTE MF636 Modem mode
	{ USB_DEVICE(0x19D2, 0x0037) }, //G19889 ONDA MAS405 HS / G20199  ONDA Onda_MT503HSA
	{ USB_DEVICE(0x19D2, 0x0052) }, //Vodafone K3565-Z Modem mode
	{ USB_DEVICE(0x19D2, 0x0055) }, //ZTE K3520 Modem mode
	{ USB_DEVICE(0x19D2, 0x0057) }, //AIKO 83D
	{ USB_DEVICE(0x19D2, 0x0063) }, //Vodafone K3565-Z
	{ USB_DEVICE(0x19D2, 0x0073) }, //ZTE AC560 modem mode
	{ USB_DEVICE(0x19D2, 0x0094) }, //ZTE AC560/581 new modem mode
	{ USB_DEVICE(0x19D2, 0x0103) }, //ZTE MF112 modem mode
	{ USB_DEVICE(0x19d2, 0x0117) }, /* ZTE MF668 */
	{ USB_DEVICE(0x19D2, 0x1008) }, //vodafone k3570-z  modem mode
	{ USB_DEVICE(0x19d2, 0x1010) }, /* ZTE-K3571-Z */
	{ USB_DEVICE(0x19D2, 0x2002) }, //vodafone k3765-z  modem
	{ USB_DEVICE(0x19D2, 0xF000) }, //ZTE Modem
	{ USB_DEVICE(0x19D2, 0xFFF1) }, //ZTE  AC8710 Modem mode
	{ USB_DEVICE(0x19D2, 0xFFFF) }, //ZTE AC8710 modem mode
	{ USB_DEVICE(0x1A8D, 0x1002) }, //BandLuxe C100 , modem mode
	{ USB_DEVICE(0x1A8D, 0x1009) }, //BandLuxe C170/C270 , modem mode
	{ USB_DEVICE(0x1a8d, 0x100d) }, /* BandLuxe C321 */
	{ USB_DEVICE(0x1AB7, 0x2000) }, //Sierra AirCard 901 TD-HSDPA modem mode
	{ USB_DEVICE(0x1BBB, 0x0000) }, //Alcatel X200X modem mode
	{ USB_DEVICE(0x1BBB, 0x0017) }, //Alcatel X200L  modem mode
	{ USB_DEVICE(0x1C9E, 0x6061) }, //Alcatel One Touch X030 Modem Mode
	{ USB_DEVICE(0x1C9E, 0x9000) }, //Mobidata MBD-220HU Modem Mode
	{ USB_DEVICE(0x1DA5, 0x4512) }, //Qisda H21 Modem Mode
	{ USB_DEVICE(0x1FE7, 0x0100) }, //Vertex VW110
    { USB_VENDOR(0x0408) }, //MU-Q101 , Royal Q110
    { USB_VENDOR(0x0421) }, //Nokia
    { USB_VENDOR(0x0572) }, //56K modem (conexant)
    { USB_VENDOR(0x05C6) }, //Option ICon
    { USB_VENDOR(0x07D1) }, //DLink
    { USB_VENDOR(0x0AF0) }, //Option ICon
    { USB_VENDOR(0x0F3D) }, //Airprime
    { USB_VENDOR(0x1199) }, //Sierra
    { USB_VENDOR(0x12D1) }, //Huawei
    { USB_VENDOR(0x1410) }, //Novatel Ovation
    { USB_VENDOR(0x16D8) }, //C-motech
    { USB_VENDOR(0x19D2) }, //ZTE
    { USB_VENDOR(0x1A8D) }, //BandLuxe
    { USB_VENDOR(0x1AB7) }, //Sierra AirCard
    { USB_VENDOR(0x1BBB) }, //Alcatel
    { USB_VENDOR(0x1C9E) }, //Alcatel/ Mobidata
    { USB_VENDOR(0x1DA5) }, //Qisda
	{ } /* Terminating entry */
};


static struct usb_device_id exception_device_ids[] = {   //bruce hsu  
    { USB_DEVICE(0x04e8, 0x6761) },
    { USB_DEVICE(0x04e9, 0x6761) },
    { USB_DEVICE(0x04e8, 0x6731) },
    { USB_DEVICE(0x04e8, 0x6780) },
    { USB_DEVICE(0x1076, 0x7F40) },     // wijet storage
    { USB_DEVICE(0x1076, 0x7F00) },     // wijet wimax
    { USB_DEVICE(0x12D1, 0x3809) },     // BM325 storage
    { USB_DEVICE(0x12D1, 0x3808) },     // BM325 wimax
    { USB_DEVICE(0x198F, 0xBCCD) },     // JINGLE storage
    { USB_DEVICE(0x198F, 0x0220) },     // JINGLE wimax
    { USB_DEVICE(0x198F, 0x0210) },     // Tatung US210 or WU211
    { USB_DEVICE(0x0B05, 0xBCCD) },     // Global 1 storage
    { USB_DEVICE(0x0B05, 0x1780) },     // Global 1 wimax
    { USB_DEVICE(0x04e8, 0x689a) },     // Samsung LTE storage mode
    { USB_DEVICE(0x04e8, 0x6889) },     // Samsung LTE modem mode
	{ USB_DEVICE(0x1004, 0x61AA) }, 	// LG_VL600 LTE modem
	{ USB_DEVICE(0x0af0, 0x6911) }, 	// Option iCON 7.2
	{ USB_DEVICE(0x0af0, 0x6971) }, 	// Option iCON 225
	{ USB_DEVICE(0x0af0, 0x7251) }, 	// Option iCON HSUPA
	{ USB_DEVICE(0x0af0, 0xd057) }, 	// Option iCON 505 (GlobeTrotter GI1505)
    { USB_DEVICE(0x12D1, 0x1506) },     // Hauwei E398 modem
    { USB_DEVICE(0x12D1, 0x1446) },     // Hauwei E398 storage
    { USB_DEVICE(0x12D1, 0x1505) },     // Hauwei E398 storage
    { USB_DEVICE(0x19D2, 0x0167) },     // ZTE MF820D modem
    { USB_DEVICE(0x19D2, 0x0166) },     // ZTE MF820D storage
    { USB_DEVICE(0x19D2, 0x0284) },     // ZTE MF880 modem
    { USB_DEVICE(0x1410, 0xB001) },     // NOVATEL 551L LTE modem
    { USB_DEVICE(0x106C, 0x3718) },     // Pantech UML290VW LTE modem
    { USB_DEVICE(0x0408, 0xEA42) },     // M-100 LTE modem
    {}
};


/* we want to look at all devices, as the vendor/product id can change
 * depending on the command line argument */
static struct usb_device_id generic_serial_ids[] = {
	{.driver_info = 42},
	{}
};

static struct usb_driver generic_driver = {
	.name =		"usbserial_generic",
	.probe =	generic_probe,
	.disconnect =	usb_serial_disconnect,
	.id_table =	generic_serial_ids,
	.no_dynamic_id =	1,
};

/* All of the device info needed for the Generic Serial Converter */
struct usb_serial_driver usb_serial_generic_device = {
	.driver = {
		.owner =	THIS_MODULE,
		.name =		"generic",
	},
	.id_table =		generic_device_ids,
	.usb_driver = 		&generic_driver,
	.num_interrupt_in =	NUM_DONT_CARE,
	.num_bulk_in =		NUM_DONT_CARE,
	.num_bulk_out =		NUM_DONT_CARE,
	.num_ports =		1,
	.shutdown =		usb_serial_generic_shutdown,
	.throttle =		usb_serial_generic_throttle,
	.unthrottle =		usb_serial_generic_unthrottle,
};

static int generic_probe(struct usb_interface *interface,
			       const struct usb_device_id *id)
{	
	const struct usb_device_id *id_pattern;
   	const struct usb_device_id *exp_pattern;
   	const struct usb_device *u_dev;
   	struct usb_interface *u_inf;
  	struct usb_interface_descriptor *iface_desc;
   	struct usb_host_endpoint *endpoint;
   	struct usb_endpoint_descriptor *u_end;
   	int cnt_if=0;
   	int cnt_end=0;
   	int cnt_intrp=0;
   	ushort my_vid = 0;	// vendor identifier 
	ushort my_pid = 0; 	// product identifier 
	ushort my_prot = 0;	// interface protocl	

   	u_dev=interface_to_usbdev(interface);
   	if(u_dev->actconfig->desc.bNumInterfaces == 1){
       	//printk("Not USB 3G ??\r\n");
      	return -ENODEV;
  	}else{
       	//printk("USB interface number: %d\r\n",u_dev->actconfig->desc.bNumInterfaces);
     	for(cnt_if=0;cnt_if<u_dev->actconfig->desc.bNumInterfaces;cnt_if++){
          	u_inf=u_dev->actconfig->interface[cnt_if];
         	iface_desc=&(u_inf->cur_altsetting->desc);
         	//printk("USB endpoint number: %d\r\n",iface_desc->bNumEndpoints);
         	for(cnt_end=0;cnt_end<iface_desc->bNumEndpoints;cnt_end++){
            	endpoint=&(u_inf->cur_altsetting->endpoint[cnt_end]);
             	u_end=&(endpoint->desc);
             	#if 0
            	if ((u_end->bEndpointAddress & 0x80) &&
		        	((u_end->bmAttributes & 3) == 0x02)) {
			       	/* we found a bulk in endpoint */
			     	printk("found bulk in\r\n");
			  	}

		     	if (((u_end->bEndpointAddress & 0x80) == 0x00) &&
		         	((u_end->bmAttributes & 3) == 0x02)) {
			     	/* we found a bulk out endpoint */
			     	printk("found bulk out\r\n");
			 	}
		     	#endif
		    	if ((u_end->bEndpointAddress & 0x80) &&
		         	((u_end->bmAttributes & 3) == 0x03)) {
			      	/* we found a interrupt in endpoint */
			      	//printk("found interrupt in\r\n");

    				if (iface_desc->bInterfaceClass==USB_CLASS_COMM&&
    				    iface_desc->bInterfaceSubClass==USB_CDC_SUBCLASS_MBIM &&
    				    iface_desc->bInterfaceProtocol==USB_CDC_PROTO_NONE) 
					{
              	        printk(KERN_INFO "[USB Generic] use MBIM Driver instead\n");
             	        return -ENODEV; 
					}
					
			      	// CDC driver:
    				// Interface Class = USB_CLASS_COMM (2)
                    // Interface SubClass = USB_CDC_SUBCLASS_ETHERNET (6)
                    // Interface Protocol = USB_CDC_PROTO_NONE (0)     				
    				if (iface_desc->bInterfaceClass==USB_CLASS_COMM&&
    				    iface_desc->bInterfaceSubClass==USB_CDC_SUBCLASS_ETHERNET&&
    				    iface_desc->bInterfaceProtocol==USB_CDC_PROTO_NONE) {
              	        printk(KERN_INFO "[USB Generic] USB Interface (Cls=%x, Sub=%x, Prot=%x) - match CDC driver (Cls=2, Sub=6, Prot=0)\n", 
             	            iface_desc->bInterfaceClass, iface_desc->bInterfaceSubClass, iface_desc->bInterfaceProtocol);
             	        return -ENODEV; 
             	    } else {
    			      	if (cnt_intrp==0) {
    			      		my_vid = u_dev->descriptor.idVendor;
        					my_pid = u_dev->descriptor.idProduct;
        					my_prot = iface_desc->bInterfaceProtocol;
        				}
                    	cnt_intrp++;                 
                    	goto FIND_INTERRUPT;
                    }  	
            	}
        	}
     	}
FIND_INTERRUPT:
     	if(cnt_intrp==0){
          	//printk("USB 3G interface : %d, but no interrupt pipe...\r\n",u_dev->actconfig->desc.bNumInterfaces);
         	return -ENODEV;
     	}     	
    }
	id_pattern = usb_match_id(interface, generic_device_ids);
   	if (id_pattern != NULL){
      	//Add exception device list here    //Bruce Hsu 20110622
      	exp_pattern=usb_match_id(interface, exception_device_ids);
      	if(exp_pattern != NULL){
          	printk(KERN_INFO "[USB Generic] this device (%04X, %04X, %x) may be a excepted device\r\n", my_vid, my_pid, my_prot);
         	int match_id = 0;
         	int match_prot = 0;
         	if (my_vid==exp_e398.vid&&my_pid==exp_e398.pid) {
         		match_id = 1;
         	   	if (my_prot==exp_e398.prot) {
         			match_prot = 1;
         		}
         	   	if (my_prot==0x16) {
         			match_prot = 1;
         		}
         	}
         	
         	// For the device (e.g. E52/E367/E72) that its ID is (12d1/1506) but it isn't a exception device  
         	if (match_id==1&&match_prot==0) {
         		printk(KERN_INFO "[USB Generic] this device (%04X, %04X, %x) isn't a excepted device\r\n", my_vid, my_pid, my_prot);
         		goto NOT_EXCEPTED_DEVICE;	
         	}
         	printk(KERN_INFO "[USB Generic] this device (%04X, %04X, %x) is a excepted device\r\n", my_vid, my_pid, my_prot);
          	return -ENODEV;
      	}
      	
NOT_EXCEPTED_DEVICE: 
		return usb_serial_probe(interface, id);
  	}
   	return -ENODEV;
}
#endif

int usb_serial_generic_register (int _debug)
{
	int retval = 0;

	debug = _debug;
#ifdef CONFIG_USB_SERIAL_GENERIC	
    // if user set vendor and product, set it into generic_device_ids first entry.  //Bruce Hsu 2011/01/25
	generic_device_ids[0].idVendor = vendor;
	generic_device_ids[0].idProduct = product;
	generic_device_ids[0].match_flags = USB_DEVICE_ID_MATCH_VENDOR | USB_DEVICE_ID_MATCH_PRODUCT;
			
	/* register our generic driver with ourselves */
	retval = usb_serial_register (&usb_serial_generic_device);
	if (retval)
		goto exit;
	retval = usb_register(&generic_driver);
	if (retval)
		usb_serial_deregister(&usb_serial_generic_device);
exit:
#endif
	return retval;
}

void usb_serial_generic_deregister (void)
{
#ifdef CONFIG_USB_SERIAL_GENERIC
	/* remove our generic driver */
	usb_deregister(&generic_driver);
	usb_serial_deregister (&usb_serial_generic_device);
#endif
}

int usb_serial_generic_open (struct usb_serial_port *port, struct file *filp)
{
	struct usb_serial *serial = port->serial;
	int result = 0;
	unsigned long flags;

	dbg("%s - port %d", __FUNCTION__, port->number);

	/* force low_latency on so that our tty_push actually forces the data through,
	   otherwise it is scheduled, and with high data rates (like with OHCI) data
	   can get lost. */
	if (port->tty)
		port->tty->low_latency = 1;

	/* clear the throttle flags */
	spin_lock_irqsave(&port->lock, flags);
	port->throttled = 0;
	port->throttle_req = 0;
	spin_unlock_irqrestore(&port->lock, flags);

	/* if we have a bulk endpoint, start reading from it */
	if (serial->num_bulk_in) {
		/* Start reading from the device */
		usb_fill_bulk_urb (port->read_urb, serial->dev,
				   usb_rcvbulkpipe(serial->dev, port->bulk_in_endpointAddress),
				   port->read_urb->transfer_buffer,
				   port->read_urb->transfer_buffer_length,
				   ((serial->type->read_bulk_callback) ?
				     serial->type->read_bulk_callback :
				     usb_serial_generic_read_bulk_callback),
				   port);
		result = usb_submit_urb(port->read_urb, GFP_KERNEL);
		if (result)
			dev_err(&port->dev, "%s - failed resubmitting read urb, error %d\n", __FUNCTION__, result);
	}

	return result;
}
EXPORT_SYMBOL_GPL(usb_serial_generic_open);

static void generic_cleanup (struct usb_serial_port *port)
{
	struct usb_serial *serial = port->serial;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (serial->dev) {
		/* shutdown any bulk reads that might be going on */
		if (serial->num_bulk_out)
			usb_kill_urb(port->write_urb);
		if (serial->num_bulk_in)
			usb_kill_urb(port->read_urb);
	}
}

void usb_serial_generic_close (struct usb_serial_port *port, struct file * filp)
{
	dbg("%s - port %d", __FUNCTION__, port->number);
	generic_cleanup (port);
}

int usb_serial_generic_write(struct usb_serial_port *port, const unsigned char *buf, int count)
{
	struct usb_serial *serial = port->serial;
	int result;
	unsigned char *data;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (count == 0) {
		dbg("%s - write request of 0 bytes", __FUNCTION__);
		return (0);
	}

	/* only do something if we have a bulk out endpoint */
	if (serial->num_bulk_out) {
		spin_lock_bh(&port->lock);
		if (port->write_urb_busy) {
			spin_unlock_bh(&port->lock);
			dbg("%s - already writing", __FUNCTION__);
			return 0;
		}
		port->write_urb_busy = 1;
		spin_unlock_bh(&port->lock);

		count = (count > port->bulk_out_size) ? port->bulk_out_size : count;

		memcpy (port->write_urb->transfer_buffer, buf, count);
		data = port->write_urb->transfer_buffer;
		usb_serial_debug_data(debug, &port->dev, __FUNCTION__, count, data);

		/* set up our urb */
		usb_fill_bulk_urb (port->write_urb, serial->dev,
				   usb_sndbulkpipe (serial->dev,
						    port->bulk_out_endpointAddress),
				   port->write_urb->transfer_buffer, count,
				   ((serial->type->write_bulk_callback) ?
				     serial->type->write_bulk_callback :
				     usb_serial_generic_write_bulk_callback), port);

		/* send the data out the bulk port */
		port->write_urb_busy = 1;
		result = usb_submit_urb(port->write_urb, GFP_ATOMIC);
		if (result) {
			dev_err(&port->dev, "%s - failed submitting write urb, error %d\n", __FUNCTION__, result);
			/* don't have to grab the lock here, as we will retry if != 0 */
			port->write_urb_busy = 0;
		} else
			result = count;

		return result;
	}

	/* no bulk out, so return 0 bytes written */
	return 0;
}

int usb_serial_generic_write_room (struct usb_serial_port *port)
{
	struct usb_serial *serial = port->serial;
	int room = 0;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (serial->num_bulk_out) {
		if (!(port->write_urb_busy))
			room = port->bulk_out_size;
	}

	dbg("%s - returns %d", __FUNCTION__, room);
	return (room);
}

int usb_serial_generic_chars_in_buffer (struct usb_serial_port *port)
{
	struct usb_serial *serial = port->serial;
	int chars = 0;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (serial->num_bulk_out) {
		if (port->write_urb_busy)
			chars = port->write_urb->transfer_buffer_length;
	}

	dbg("%s - returns %d", __FUNCTION__, chars);
	return (chars);
}

/* Push data to tty layer and resubmit the bulk read URB */
static void flush_and_resubmit_read_urb (struct usb_serial_port *port)
{
	struct usb_serial *serial = port->serial;
	struct urb *urb = port->read_urb;
	struct tty_struct *tty = port->tty;
	int result;

	/* Push data to tty */
	if (tty && urb->actual_length) {
		tty_buffer_request_room(tty, urb->actual_length);
		tty_insert_flip_string(tty, urb->transfer_buffer, urb->actual_length);
	  	tty_flip_buffer_push(tty); /* is this allowed from an URB callback ? */
	}

	/* Continue reading from device */
	usb_fill_bulk_urb (port->read_urb, serial->dev,
			   usb_rcvbulkpipe (serial->dev,
				   	    port->bulk_in_endpointAddress),
			   port->read_urb->transfer_buffer,
			   port->read_urb->transfer_buffer_length,
			   ((serial->type->read_bulk_callback) ?
			     serial->type->read_bulk_callback :
			     usb_serial_generic_read_bulk_callback), port);
	result = usb_submit_urb(port->read_urb, GFP_ATOMIC);
	if (result)
		dev_err(&port->dev, "%s - failed resubmitting read urb, error %d\n", __FUNCTION__, result);
}

void usb_serial_generic_read_bulk_callback (struct urb *urb)
{
	struct usb_serial_port *port = (struct usb_serial_port *)urb->context;
	unsigned char *data = urb->transfer_buffer;
	int is_throttled;
	unsigned long flags;

	dbg("%s - port %d", __FUNCTION__, port->number);

	if (urb->status) {
		dbg("%s - nonzero read bulk status received: %d", __FUNCTION__, urb->status);
		return;
	}

	usb_serial_debug_data(debug, &port->dev, __FUNCTION__, urb->actual_length, data);

	/* Throttle the device if requested by tty */
	if (urb->actual_length) {
		spin_lock_irqsave(&port->lock, flags);
		is_throttled = port->throttled = port->throttle_req;
		spin_unlock_irqrestore(&port->lock, flags);
		if (is_throttled) {
			/* Let the received data linger in the read URB;
			 * usb_serial_generic_unthrottle() will pick it
			 * up later. */
			dbg("%s - throttling device", __FUNCTION__);
			return;
		}
	}

	/* Handle data and continue reading from device */
	flush_and_resubmit_read_urb(port);
}
EXPORT_SYMBOL_GPL(usb_serial_generic_read_bulk_callback);

void usb_serial_generic_write_bulk_callback (struct urb *urb)
{
	struct usb_serial_port *port = (struct usb_serial_port *)urb->context;

	dbg("%s - port %d", __FUNCTION__, port->number);

	port->write_urb_busy = 0;
	if (urb->status) {
		dbg("%s - nonzero write bulk status received: %d", __FUNCTION__, urb->status);
		return;
	}

	usb_serial_port_softint(port);
}
EXPORT_SYMBOL_GPL(usb_serial_generic_write_bulk_callback);

void usb_serial_generic_throttle (struct usb_serial_port *port)
{
	unsigned long flags;

	dbg("%s - port %d", __FUNCTION__, port->number);

	/* Set the throttle request flag. It will be picked up
	 * by usb_serial_generic_read_bulk_callback(). */
	spin_lock_irqsave(&port->lock, flags);
	port->throttle_req = 1;
	spin_unlock_irqrestore(&port->lock, flags);
}

void usb_serial_generic_unthrottle (struct usb_serial_port *port)
{
	int was_throttled;
	unsigned long flags;

	dbg("%s - port %d", __FUNCTION__, port->number);

	/* Clear the throttle flags */
	spin_lock_irqsave(&port->lock, flags);
	was_throttled = port->throttled;
	port->throttled = port->throttle_req = 0;
	spin_unlock_irqrestore(&port->lock, flags);

	if (was_throttled) {
		/* Handle pending data and resume reading from device */
		flush_and_resubmit_read_urb(port);
	}
}

void usb_serial_generic_shutdown (struct usb_serial *serial)
{
	int i;

	dbg("%s", __FUNCTION__);

	/* stop reads and writes on all ports */
	for (i=0; i < serial->num_ports; ++i) {
		generic_cleanup(serial->port[i]);
	}
}

