/* vi: set sw=4 ts=4: */
/* clientpacket.c
 *
 * Packet generation and dispatching functions for the DHCP client.
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * Licensed under GPLv2 or later, see file LICENSE in this tarball for details.
 */

#include <features.h>
#if (defined(__GLIBC__) && __GLIBC__ >= 2 && __GLIBC_MINOR__ >= 1) || defined _NEWLIB_VERSION
#include <netpacket/packet.h>
#include <net/ethernet.h>
#else
#include <asm/types.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#endif

#include "common.h"
#include "dhcpd.h"
#include "dhcpc.h"
#include "options.h"

#define CUSTOM_DHCP_OPTION 1

/* Create a random xid */
uint32_t random_xid(void)
{
	static smallint initialized;

	if (!initialized) {
		srand(monotonic_us());
		initialized = 1;
	}
	return rand();
}

#if CUSTOM_DHCP_OPTION == 1

// referenced from usb_modeswitch.c
int hex2num(char c)
{
	if (c >= '0' && c <= '9')
	    return c - '0';
	if (c >= 'a' && c <= 'f')
	    return c - 'a' + 10;
	if (c >= 'A' && c <= 'F')
	    return c - 'A' + 10;
	return -1;
}

// referenced from usb_modeswitch.c
int hex2byte(const char *hex)
{
	int a, b;
	a = hex2num(*hex++);
	if (a < 0)
	    return -1;
	b = hex2num(*hex++);
	if (b < 0)
	    return -1;
	return (a << 4) | b;
}

// referenced from usb_modeswitch.c
int hexstr2bin(const char *hex, char *buf, int len)
{
	int i;
	int a;
	const char *ipos = hex;
	char *opos = buf;

	for (i = 0; i < len; i++) {
	    a = hex2byte(ipos);
	    if (a < 0)
		    return -1;

	    *opos++ = a;
	    ipos += 2;
	}

	return opos - buf;
}

#define MAX_CUSTOM_DHCP_OPTION_LEN 64
// referenced from dhcpc.c
static uint8_t* alloc_dhcp_option(int code, const char *str, int extra, int len)
{
	uint8_t *storage;

    if (!len)
        len = strlen(str);
	if (len > 255)
        len = 255;
	storage = xzalloc(len + extra + OPT_DATA);
	storage[OPT_CODE] = code;
	storage[OPT_LEN] = len + extra;
	memcpy(storage + extra + OPT_DATA, str, len);
	return storage;
}

#endif
/* initialize a packet with the proper defaults */
static void init_packet(struct dhcpMessage *packet, char type)
{
#if CUSTOM_DHCP_OPTION == 1
    int m = 0;
    char cmd[200];
    char en[5], data_type[10], numb[10], val[200], hex_val[200];
    unsigned char data[MAX_CUSTOM_DHCP_OPTION_LEN] = {0};
    char *list;
    int cid_flag = 0;
    int hname_flag = 0;
    int fqdn_flag = 0;
    int vclass_flag = 0;
    int len = 0;

    list = malloc(200);
    if (list == NULL) {
        printf("%s():%d: Malloc failed!\r\n", __FUNCTION__, __LINE__);
	}
#endif

	udhcp_init_header(packet, type);
	memcpy(packet->chaddr, client_config.arp, 6);

#if CUSTOM_DHCP_OPTION == 1
    for (m = 0; m < 6; m++) {
        sprintf(cmd, "nvram_get 2860 dhcpc_option_%d", m);
        list = run_command(cmd);
        if (strchr(list, ';') != NULL) {
            getNthValueSafe(0, list, ';', en, sizeof(en));
            getNthValueSafe(1, list, ';', data_type, sizeof(data_type));
            getNthValueSafe(2, list, ';', numb, sizeof(numb));
            getNthValueSafe(3, list, ';', val, sizeof(val));

            if (en[0] == '1') {
                if (atoi(numb) == DHCP_CLIENT_ID)
                    cid_flag = 1;
                else if (atoi(numb) == DHCP_VENDOR)
                    vclass_flag = 1;
                else if (atoi(numb) == DHCP_HOST_NAME)
                    hname_flag = 1;
                else if (atoi(numb) == DHCP_FQDN)
                    fqdn_flag = 1;

                if (val[0] != '\0' && data_type[0] == 'R') {
                    len = hexstr2bin(val, data, strlen(val)/2);
                    add_option_string(packet->options, alloc_dhcp_option(atoi(numb), data, 0, len));
                } else {
                    add_option_string(packet->options, alloc_dhcp_option(atoi(numb), val, 0, 0));
                }
            }
        }
        if (list)
            free(list);
        memset(data ,0, sizeof(data));
    }
#endif

#if CUSTOM_DHCP_OPTION == 1
	if (client_config.clientid && !cid_flag)
		add_option_string(packet->options, client_config.clientid);
	if (client_config.hostname && !hname_flag)
		add_option_string(packet->options, client_config.hostname);
	if (client_config.fqdn && !fqdn_flag)
		add_option_string(packet->options, client_config.fqdn);
	if ((type != DHCPDECLINE) && (type != DHCPRELEASE) && !vclass_flag)
		add_option_string(packet->options, client_config.vendorclass);
#else
    if (client_config.clientid)
		add_option_string(packet->options, client_config.clientid);
	if (client_config.hostname)
		add_option_string(packet->options, client_config.hostname);
	if (client_config.fqdn)
		add_option_string(packet->options, client_config.fqdn);
	if ((type != DHCPDECLINE) && (type != DHCPRELEASE))
		add_option_string(packet->options, client_config.vendorclass);
#endif
}


/* Add a parameter request list for stubborn DHCP servers. Pull the data
 * from the struct in options.c. Don't do bounds checking here because it
 * goes towards the head of the packet. */
static void add_param_req_option(struct dhcpMessage *packet)
{
	uint8_t c;
	int end = end_option(packet->options);
	int i, len = 0;

	for (i = 0; (c = dhcp_options[i].code) != 0; i++) {
		if (((dhcp_options[i].flags & OPTION_REQ)
		     && !client_config.no_default_options)
		 || (client_config.opt_mask[c >> 3] & (1 << (c & 7)))
		) {
			packet->options[end + OPT_DATA + len] = c;
			len++;
		}
	}
	if (len) {
		packet->options[end + OPT_CODE] = DHCP_PARAM_REQ;
		packet->options[end + OPT_LEN] = len;
		packet->options[end + OPT_DATA + len] = DHCP_END;
	}
}


#if ENABLE_FEATURE_UDHCPC_ARPING
/* Unicast a DHCP decline message */
int send_decline(uint32_t xid, uint32_t server, uint32_t requested)
{
	struct dhcpMessage packet;

	init_packet(&packet, DHCPDECLINE);
	packet.xid = xid;
	add_simple_option(packet.options, DHCP_REQUESTED_IP, requested);
	add_simple_option(packet.options, DHCP_SERVER_ID, server);

	bb_info_msg("Sending decline...");

	return udhcp_send_raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST,
		SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
}
#endif

/* Broadcast a DHCP discover packet to the network, with an optionally requested IP */
int send_discover(uint32_t xid, uint32_t requested)
{
	struct dhcpMessage packet;

	init_packet(&packet, DHCPDISCOVER);
	packet.xid = xid;
	if (requested)
		add_simple_option(packet.options, DHCP_REQUESTED_IP, requested);

	/* Explicitly saying that we want RFC-compliant packets helps
	 * some buggy DHCP servers to NOT send bigger packets */
	add_simple_option(packet.options, DHCP_MAX_SIZE, htons(576));

	add_param_req_option(&packet);

	//bb_info_msg("Sending discover...");
	return udhcp_send_raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST,
			SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
}


/* Broadcasts a DHCP request message */
int send_selecting(uint32_t xid, uint32_t server, uint32_t requested)
{
	struct dhcpMessage packet;
	struct in_addr addr;

	init_packet(&packet, DHCPREQUEST);
	packet.xid = xid;

	add_simple_option(packet.options, DHCP_REQUESTED_IP, requested);
	add_simple_option(packet.options, DHCP_SERVER_ID, server);
	add_param_req_option(&packet);

	addr.s_addr = requested;
	bb_info_msg("Sending select for %s...", inet_ntoa(addr));
	return udhcp_send_raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST,
				SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
}


/* Unicasts or broadcasts a DHCP renew message */
int send_renew(uint32_t xid, uint32_t server, uint32_t ciaddr)
{
	struct dhcpMessage packet;

	init_packet(&packet, DHCPREQUEST);
	packet.xid = xid;
	packet.ciaddr = ciaddr;

	add_param_req_option(&packet);
	bb_info_msg("Sending renew...");
	if (server)
		return udhcp_send_kernel_packet(&packet, ciaddr, CLIENT_PORT, server, SERVER_PORT);

	return udhcp_send_raw_packet(&packet, INADDR_ANY, CLIENT_PORT, INADDR_BROADCAST,
				SERVER_PORT, MAC_BCAST_ADDR, client_config.ifindex);
}


/* Unicasts a DHCP release message */
int send_release(uint32_t server, uint32_t ciaddr)
{
	struct dhcpMessage packet;

	init_packet(&packet, DHCPRELEASE);
	packet.xid = random_xid();
	packet.ciaddr = ciaddr;

	add_simple_option(packet.options, DHCP_SERVER_ID, server);

	bb_info_msg("Sending release...");
	return udhcp_send_kernel_packet(&packet, ciaddr, CLIENT_PORT, server, SERVER_PORT);
}


/* Returns -1 on errors that are fatal for the socket, -2 for those that aren't */
int udhcp_recv_raw_packet(struct dhcpMessage *payload, int fd)
{
	int bytes;
	struct udp_dhcp_packet packet;
	uint16_t check;

	memset(&packet, 0, sizeof(packet));
	bytes = safe_read(fd, &packet, sizeof(packet));
	if (bytes < 0) {
		DEBUG("Cannot read on raw listening socket - ignoring");
		/* NB: possible down interface, etc. Caller should pause. */
		return bytes; /* returns -1 */
	}

	if (bytes < (int) (sizeof(packet.ip) + sizeof(packet.udp))) {
		DEBUG("Packet is too short, ignoring");
		return -2;
	}

	if (bytes < ntohs(packet.ip.tot_len)) {
		/* packet is bigger than sizeof(packet), we did partial read */
		DEBUG("Oversized packet, ignoring");
		return -2;
	}

	/* ignore any extra garbage bytes */
	bytes = ntohs(packet.ip.tot_len);

	/* make sure its the right packet for us, and that it passes sanity checks */
	if (packet.ip.protocol != IPPROTO_UDP || packet.ip.version != IPVERSION
	 || packet.ip.ihl != (sizeof(packet.ip) >> 2)
	 || packet.udp.dest != htons(CLIENT_PORT)
	/* || bytes > (int) sizeof(packet) - can't happen */
	 || ntohs(packet.udp.len) != (uint16_t)(bytes - sizeof(packet.ip))
	) {
		DEBUG("Unrelated/bogus packet");
		return -2;
	}

	/* verify IP checksum */
	check = packet.ip.check;
	packet.ip.check = 0;
	if (check != udhcp_checksum(&packet.ip, sizeof(packet.ip))) {
		DEBUG("Bad IP header checksum, ignoring");
		return -2;
	}

	/* verify UDP checksum. IP header has to be modified for this */
	memset(&packet.ip, 0, offsetof(struct iphdr, protocol));
	/* ip.xx fields which are not memset: protocol, check, saddr, daddr */
	packet.ip.tot_len = packet.udp.len; /* yes, this is needed */
	check = packet.udp.check;
	packet.udp.check = 0;
	if (check && check != udhcp_checksum(&packet, bytes)) {
		bb_error_msg("packet with bad UDP checksum received, ignoring");
		return -2;
	}

	memcpy(payload, &packet.data, bytes - (sizeof(packet.ip) + sizeof(packet.udp)));

	if (payload->cookie != htonl(DHCP_MAGIC)) {
		bb_error_msg("received bogus message (bad magic), ignoring");
		return -2;
	}
	DEBUG("Got valid DHCP packet");
	return bytes - (sizeof(packet.ip) + sizeof(packet.udp));
}
