/*
 * This is a reverse-engineered driver for mobile WiMAX (802.16e) devices
 * based on Samsung CMC-730 chip.
 * Copyright (C) 2008-2009 Alexander Gordeev <lasaine@lvk.cs.msu.su>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <errno.h>
#include <fcntl.h>
#include <getopt.h>    
#include <poll.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <libusb.h>
#include <stdarg.h>

#include "protocol.h"
#include "wimax.h"
#include "tap_dev.h"
#include "../../config/autoconf.h"

/* variables for the command-line parameters */ 
static int daemonize = 0;
static int diode_on = 1;
static int detach_dvd = 0;
static char *ssid = "@yota.ru";

/* for matching by list... */
typedef struct usb_device_id_t {
	unsigned short vendorID;
	unsigned short productID;
} usb_device_id_t;

/* list of all known devices */
static usb_device_id_t wimax_dev_ids[] = {
	{ 0x04e8, 0x6761 }, // Samsung dongle
	{ 0x04e9, 0x6761 },
	{ 0x04e8, 0x6731 },
	{ 0x04e8, 0x6780 }
};

/* for other methods of matching... */
static union {
	struct {
		unsigned short vid;
		unsigned short pid;
	};
	struct {
		unsigned int bus;
		unsigned int dev;
	};
} match_params;

/* USB-related parameters */
#define IF_MODEM		0
#define IF_DVD			1

#define MAX_PACKET_LEN		0x1000

// Samsung : bulk IN pipe : 0x82, bulk OUT pipe : 0x04
unsigned char EP_IN;
unsigned char EP_OUT;

/* information collector */
struct wimax_dev_status wd_status;

char *wimax_states[] = {"INIT", "SYNC", "NEGO", "NORMAL", "SLEEP", "IDLE", "HHO", "FBSS", "RESET", "RESERVED", "UNDEFINED", "BE", "NRTPS", "RTPS", "ERTPS", "UGS", "INITIAL_RNG", "BASIC", "PRIMARY", "SECONDARY", "MULTICAST", "NORMAL_MULTICAST", "SLEEP_MULTICAST", "IDLE_MULTICAST", "FRAG_BROADCAST", "BROADCAST", "MANAGEMENT", "TRANSPORT"};

/* libusb stuff */
static struct libusb_context *ctx = NULL;
struct libusb_device_handle *devh = NULL;
static struct libusb_transfer *req_transfer = NULL;
static int kernel_driver_active = 0;

static unsigned char read_buffer[MAX_PACKET_LEN];

static int tap_fd = -1;
static char tap_dev[20] = "wimax%d";
int tap_if_up = 0;

static nfds_t nfds;
static struct pollfd* fds = NULL;

static int first_nego_flag = 0;
int device_disconnected = 0;

int wimax_err = 0;

#define CHECK_NEGATIVE(x) {if((r = (x)) < 0) return r;}
#define CHECK_DISCONNECTED(x) {if((r = (x)) == LIBUSB_ERROR_NO_DEVICE) exit_release_resources(0);}

static void exit_release_resources(int code);
extern void wimax_dump_buffer(unsigned char *data, int size, char *info);
extern void WIMAX_PRINT(int do_syslog, const char *format, ...);
extern int check_wimax_dongle_mac(unsigned char* mac, int device);
static int process_events_by_mask(int timeout, unsigned int event_mask);

struct libusb_device *dev_save;
static struct libusb_device_handle* find_wimax_device(void)
{
	struct libusb_device **devs;
	struct libusb_device *found = NULL;
	struct libusb_device *dev;
	struct libusb_device_handle *handle = NULL;
	int i = 0;
	int r;

	if (libusb_get_device_list(ctx, &devs) < 0)
	{
		WIMAX_PRINT(LOG_ERR, "Wimax : libusb_get_device_list() failed !!!");
		return NULL;
	}

	while (!found && (dev = devs[i++]) != NULL) {
		struct libusb_device_descriptor desc;
		struct libusb_config_descriptor * cfg_desc;
		unsigned int j = 0;
		unsigned short dev_vid, dev_pid;

		r = libusb_get_device_descriptor(dev, &desc);
		if (r < 0) {
			continue;
		}
		dev_vid = libusb_le16_to_cpu(desc.idVendor);
		dev_pid = libusb_le16_to_cpu(desc.idProduct);

		wd_status.dev_vid = dev_vid;
		wd_status.dev_pid = dev_pid;

		WIMAX_PRINT(LOG_INFO, "Wimax : Bus %03d Device %03d: ID %04x:%04x", libusb_get_bus_number(dev), libusb_get_device_address(dev), dev_vid, dev_pid);

		for (j = 0; j < sizeof(wimax_dev_ids) / sizeof(usb_device_id_t); j++) {
			if (dev_vid == wimax_dev_ids[j].vendorID && dev_pid == wimax_dev_ids[j].productID) {
				found = dev;
				break;
			}
		}
	}
	
	if (found) {
		dev_save = found;
		r = libusb_open(found, &handle);
		if (r < 0)
			handle = NULL;
	}

	libusb_free_device_list(devs, 1);
	return handle;
}

static int set_data(unsigned char* data, int size)
{
	int r;
	int transferred;
	wimax_dump_buffer(data, size, "Bulk write:");

	r = libusb_bulk_transfer(devh, EP_OUT, data, size, &transferred, 0);
	if (r < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : bulk write error %d", r);
		if (r == LIBUSB_ERROR_NO_DEVICE) {
			exit_release_resources(0);
		}
		return r;
	}
	if (transferred < size) {
		WIMAX_PRINT(LOG_ERR, "Wimax : short write (%d)", r);
		return -1;
	}
	return r;
}


int dray_libusb_tx(unsigned char *buf, int size)
{
	int transmitted;
	return libusb_bulk_transfer(devh, EP_OUT, buf, size, &transmitted, 0);
}

static void cb_req(struct libusb_transfer *transfer)
{
	if (transfer->status != LIBUSB_TRANSFER_COMPLETED) {
		if (transfer->status == LIBUSB_TRANSFER_NO_DEVICE) {
			device_disconnected = 1;
			return;
		}
		WIMAX_PRINT(LOG_ERR, "Wimax : async bulk read error %d", transfer->status);
	} else {
		process_response(&wd_status, transfer->buffer, transfer->actual_length);
	}

re_submit:	
	if (libusb_submit_transfer(req_transfer) < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : async read transfer sumbit failed");
	}
}

/* set close-on-exec flag on the file descriptor */
int set_coe(int fd)
{
	int flags;

	flags = fcntl(fd, F_GETFD);
	if (flags == -1) {
		WIMAX_PRINT(LOG_ERR, "Wimax : failed to set close-on-exec flag on fd %d", fd);
		return -1;
	}
	flags |= FD_CLOEXEC;
	if (fcntl(fd, F_SETFD, flags) == -1) {
		WIMAX_PRINT(LOG_ERR, "Wimax : failed to set close-on-exec flag on fd %d", fd);
		return -1;
	}

	return 0;
}

/* brings interface up and runs a user-supplied script */
static int if_create()
{
	tap_fd = tap_open(tap_dev);
	if (tap_fd < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : failed to allocate tap interface");
	#if WIMAX_DEBUG
		printf("Wimax : failed to allocate tap interface\n");
		printf("You should have TUN/TAP driver compiled in the kernel or as a kernel module.\n");
		printf("If 'modprobe tun' doesn't help then recompile your kernel.\n");
	#endif
		exit_release_resources(1);
	}
	tap_set_hwaddr(tap_fd, tap_dev, wd_status.mac);
	tap_set_mtu(tap_fd, tap_dev, 1386);
	set_coe(tap_fd);
	return 0;
}

/* brings interface down and runs a user-supplied script */
static int if_release()
{
	tap_close(tap_fd, tap_dev);
	return 0;
}


void sighandler_set_info(int signum);

/* set link_status */
void set_link_status(int link_status)
{
	wd_status.info_updated |= WDS_LINK_STATUS;

	if (wd_status.link_status == link_status) return;

	if (wd_status.link_status < 2 && link_status == 2) {
		system("echo 1 > /var/network/wimaxStatus2");
		system("wimax.sh up");
	}
	if (wd_status.link_status == 2 && link_status < 2) {
		system("echo 0 > /var/network/wimaxStatus2");
		system("wimax.sh down");
	}
	if (link_status == 1) {
		first_nego_flag = 1;
	}
	wd_status.link_status = link_status;
	sighandler_set_info(SIGUSR1);
}

/* get state */
int get_state()
{
	return wd_status.state;
}

/* set state */
void set_state(int state)
{
	wd_status.state = state;
	wd_status.info_updated |= WDS_STATE;
	if (state >= 1 && state <= 3 && wd_status.link_status != (state - 1)) {
		set_link_status(state - 1);
	}
}

static int alloc_transfers(void)
{
	req_transfer = libusb_alloc_transfer(0);
	if (!req_transfer)
		return -ENOMEM;
	libusb_fill_bulk_transfer(req_transfer, devh, EP_IN, read_buffer, sizeof(read_buffer), cb_req, NULL, 0);
	return 0;
}

int write_netif(const void *buf, int count)
{
	return tap_write(tap_fd, buf, count);
}

static int read_tap()
{
	unsigned char * buf;
	int hlen = get_header_len();
	int r;
	int len;
	
	buf = (unsigned char *)(unsigned char *)calloc(1, MAX_PACKET_LEN);
	if(!buf) {
		WIMAX_PRINT(LOG_ERR, "Wimax : can't get a buufer for reading TAP device");
		return 0;
	}
	r = tap_read(tap_fd, buf + hlen, MAX_PACKET_LEN - hlen);

	if (r < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : Error while reading from TAP interface");
		return r;
	}

	if (r == 0) {
		return 0;

	}
	len = fill_data_packet_header(buf, r);
	wimax_dump_buffer(buf, len, "Outgoing packet:");
	r = set_data(buf, len);
    free(buf);
	return r;
}

static int process_events_once(int timeout)
{
	struct timeval tv = {0, 0};
	int r;
	int libusb_delay;
	int delay;
	unsigned int i;
	char process_libusb = 0;

	r = libusb_get_next_timeout(ctx, &tv);
	if (r == 1 && tv.tv_sec == 0 && tv.tv_usec == 0) {
		r = libusb_handle_events_timeout(ctx, &tv);
	}

	delay = libusb_delay = tv.tv_sec * 1000 + tv.tv_usec;
	if (delay <= 0 || delay > timeout) {
		delay = timeout;
	}

	CHECK_NEGATIVE(poll(fds, nfds, delay));

	process_libusb = (r == 0 && delay == libusb_delay);

	for (i = 0; i < nfds; ++i) {
		if (fds[i].fd == tap_fd) {
			if (fds[i].revents) {
				CHECK_NEGATIVE(read_tap());
			}
			continue;
		}
		process_libusb |= fds[i].revents;
	}

	if (process_libusb) {
		struct timeval tv = {.tv_sec = 0, .tv_usec = 0};
		CHECK_NEGATIVE(libusb_handle_events_timeout(ctx, &tv));
	}

	return 0;
}

/* handle events until timeout is reached or all of the events in event_mask happen */
static int process_events_by_mask(int timeout, unsigned int event_mask)
{
	struct timeval start, curr;
	int r;
	int delay = timeout;

	CHECK_NEGATIVE(gettimeofday(&start, NULL));

	wd_status.info_updated &= ~event_mask;

	while ((event_mask == 0 || (wd_status.info_updated & event_mask) != event_mask) && delay >= 0) {
		long a;

		CHECK_NEGATIVE(process_events_once(delay));

		if (device_disconnected) {
			exit_release_resources(0);
		}

		CHECK_NEGATIVE(gettimeofday(&curr, NULL));

		a = (curr.tv_sec - start.tv_sec) * 1000 + (curr.tv_usec - start.tv_usec) / 1000;
		delay = timeout - a;
	}

	wd_status.info_updated &= ~event_mask;

	return (delay > 0) ? delay : 0;
}

int alloc_fds()
{
	int i;
	const struct libusb_pollfd **usb_fds = libusb_get_pollfds(ctx);

	if (!usb_fds)
		return -1;

	nfds = 0;
	while (usb_fds[nfds]) {
		nfds++;
	}
	if (tap_fd != -1)
		nfds++;

	if(fds != NULL)
		free(fds);

	fds = (struct pollfd*)calloc(nfds, sizeof(struct pollfd));
	for (i = 0; usb_fds[i]; ++i) {
		fds[i].fd = usb_fds[i]->fd;
		fds[i].events = usb_fds[i]->events;
		set_coe(usb_fds[i]->fd);
	}
	if (tap_fd != -1) {
		fds[i].fd = tap_fd;
		fds[i].events = POLLIN;
		fds[i].revents = 0;
	}
	free(usb_fds);

	return 0;
}

void cb_add_pollfd(int fd, short events, void *user_data)
{
	alloc_fds();
}

void cb_remove_pollfd(int fd, void *user_data)
{
	alloc_fds();
}

static int init(void)
{
	unsigned char req_data[200];
	int len;
	int r;

	alloc_transfers();
	CHECK_DISCONNECTED(libusb_submit_transfer(req_transfer));

	len = fill_protocol_info_req(req_data,
			USB_HOST_SUPPORT_SELECTIVE_SUSPEND | USB_HOST_SUPPORT_DL_SIX_BYTES_HEADER |
			USB_HOST_SUPPORT_UL_SIX_BYTES_HEADER | USB_HOST_SUPPORT_DL_MULTI_PACKETS);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_PROTO_FLAGS);

	len = fill_mac_lowlevel_req(req_data);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_OTHER);

	len = fill_init_cmd(req_data);
	set_data(req_data, len);

	len = fill_string_info_req(req_data);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_CHIP | WDS_FIRMWARE);

#if WIMAX_DEBUG == 1
	WIMAX_PRINT(LOG_NOTICE, "Wimax : Chip info: %s", wd_status.chip);
	WIMAX_PRINT(LOG_NOTICE, "Wimax : Firmware info: %s", wd_status.firmware);
	printf("Wimax : Chip info: %s, Firmware info: %s\n", wd_status.chip, wd_status.firmware);
#endif

	len = fill_diode_control_cmd(req_data, get_diode_on());
	set_data(req_data, len);

	len = fill_mac_req(req_data);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_MAC);

#if WIMAX_DEBUG == 1
	WIMAX_PRINT(LOG_NOTICE, "Wimax : MAC: %02x:%02x:%02x:%02x:%02x:%02x", wd_status.mac[0], wd_status.mac[1], wd_status.mac[2], wd_status.mac[3], wd_status.mac[4], wd_status.mac[5]);
#endif

	if (check_wimax_dongle_mac(wd_status.mac, 2) < 0) {
		WIMAX_PRINT(LOG_NOTICE, "Wimax : This WIMAX dongle isn't allowed");
		return -1;
	}

	len = fill_string_info_req(req_data);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_CHIP | WDS_FIRMWARE);

	len = fill_auth_policy_req(req_data);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_OTHER);

	len = fill_auth_method_req(req_data);
	set_data(req_data, len);

	process_events_by_mask(500, WDS_OTHER);

	if(!check_dongle_need_auth()){
		return 0;
	}

#if CONFIG_USER_LITHUANIA_FW == 1
	len = fill_auth_set_cmd(req_data, get_lt_ssid());
#else
	len = fill_auth_set_cmd(req_data, ssid);
#endif
	set_data(req_data, len);
	return 0;
}

static	char buf[40];
static int scan_loop(void)
{
	unsigned char req_data[200], additional_data[200];
	int len;
	int wan_wimax = 0;
	FILE *fp;
	sprintf(buf, "echo %d > /var/run/wimax.pid", (int)getpid());
	system(buf);
	system("echo 0 > /var/network/wimaxStatus2");
	while (1 == 1) {
#if WIMAX_DEBUG == 1
		printf("Wimax : link status %d \n", wd_status.link_status);
#endif
		if (wd_status.link_status == 0) {
			len = fill_find_network_req(req_data, 1);
			set_data(req_data, len);

			additional_tx_check(additional_data);

			process_events_by_mask(5000, WDS_LINK_STATUS);

			if (wd_status.link_status == 0) {
				WIMAX_PRINT(LOG_INFO, "Wimax : find network level 1");
			} else {
				WIMAX_PRINT(LOG_INFO, "Wimax : Network found.");
			}
		} else {
			len = fill_connection_params_req(req_data);
			set_data(req_data, len);

			process_events_by_mask(500, WDS_RSSI | WDS_CINR | WDS_TXPWR | WDS_FREQ | WDS_BSID);
		#if WIMAX_DEBUG
			WIMAX_PRINT(LOG_INFO, "Wimax : RSSI: %d   CINR: %f   TX Power: %d   Frequency: %d", wd_status.rssi, wd_status.cinr, wd_status.txpwr, wd_status.freq);
			WIMAX_PRINT(LOG_INFO, "Wimax : BSID: %02x:%02x:%02x:%02x:%02x:%02x", wd_status.bsid[0], wd_status.bsid[1], wd_status.bsid[2], wd_status.bsid[3], wd_status.bsid[4], wd_status.bsid[5]);
			printf("Wimax : RSSI: %d   CINR: %f   TX Power: %d   Frequency: %d\n", wd_status.rssi, wd_status.cinr, wd_status.txpwr, wd_status.freq);
			printf("Wimax : BSID: %02x:%02x:%02x:%02x:%02x:%02x\n", wd_status.bsid[0], wd_status.bsid[1], wd_status.bsid[2], wd_status.bsid[3], wd_status.bsid[4], wd_status.bsid[5]);
		#endif
			len = fill_state_req(req_data);
			set_data(req_data, len);

			process_events_by_mask(500, WDS_STATE);
		#if WIMAX_DEBUG
			WIMAX_PRINT(LOG_INFO, "Wimax : State: %s   Number: %d   Response: %d", wimax_states[wd_status.state], wd_status.state, wd_status.link_status);
			printf("Wimax : State: %s   Number: %d   Response: %d\n", wimax_states[wd_status.state], wd_status.state, wd_status.link_status);
		#endif
			if (first_nego_flag) {
				// Prevent from wimax backup connecting all the time
				wan_wimax = 0;
				fp = fopen("/var/network/wimaxUp", "r");
				if (fp) {
					if (fgetc(fp) == (int)'1')
						wan_wimax = 1;
					fclose(fp);
				}
				if (wan_wimax) {
				first_nego_flag = 0;
				len = fill_find_network_req(req_data, 2);
				set_data(req_data, len);
			}
			}

			process_events_by_mask(5000, WDS_LINK_STATUS);
		}
		if (wd_status.link_status == 0)
			wimax_err = 3;
		else
			wimax_err = 0;
	}
	return 0;
}

/* print version */
void version()
{
	printf("madwimax-0.1.1 \n");
}

static void exit_release_resources(int code)
{
	int i;
	system("rm -f /var/run/wimax.pid");
	system("rm -f /var/network/wimax");
	system("echo 0 > /var/network/wimaxStatus2");
	if(tap_fd >= 0) {
		system("wimax.sh down");
		while (wait(NULL) > 0) {}
		if_release();
		while (wait(NULL) > 0) {}
	}
	if(ctx != NULL) {
		if(req_transfer != NULL) {
			libusb_cancel_transfer(req_transfer);
			libusb_free_transfer(req_transfer);
		}
		libusb_set_pollfd_notifiers(ctx, NULL, NULL, NULL);
		if(fds != NULL) {
			free(fds);
		}
		if(devh != NULL) {
			libusb_release_interface(devh, 0);

			if(wd_status.dev_vid == 0x04E8 && wd_status.dev_pid == 0x6889) libusb_release_interface(devh, 1);

			if (kernel_driver_active)
				libusb_attach_kernel_driver(devh, 0);
			libusb_unlock_events(ctx);
			libusb_close(devh);
		}
		libusb_exit(ctx);
	}
#if WIMAX_DEBUG
	WIMAX_PRINT(LOG_INFO, "Wimax : madwimax exit");
	printf("Wimax : madwimax exit\n");
#endif
	exit(code);
}

static void sighandler_exit(int signum)
{
	exit_release_resources(0);
}

static void sighandler_wait_child(int signum)
{
	int status;
	wait3(&status, WNOHANG, NULL);
#if WIMAX_DEBUG == 1
	WIMAX_PRINT(LOG_INFO, "Wimax : Child exited with status %d", status);
#endif
}

void sighandler_set_info(int signum)
{
	int rssi, cinr_percent;
	FILE *f = fopen("/var/network/wimax", "w");
	if (f) {
		if (wd_status.link_status >= 1) {
			fprintf(f, "%d %02x:%02x:%02x:%02x:%02x:%02x %02x:%02x:%02x:%02x:%02x:%02x ", wd_status.link_status, 
				wd_status.bsid[0], wd_status.bsid[1], wd_status.bsid[2], wd_status.bsid[3], wd_status.bsid[4], wd_status.bsid[5], 
				wd_status.mac[0], wd_status.mac[1], wd_status.mac[2], wd_status.mac[3], wd_status.mac[4], wd_status.mac[5]);
			rssi = (int)wd_status.rssi;
			if (rssi >= 0 || rssi < -150)	// fix overflow
				fprintf(f, "--- ");
			else
				fprintf(f, "%d ", rssi);
			if ( wd_status.cinr < -100.0 ||  wd_status.cinr > 100.0)	// fix overflow
				fprintf(f, "--- 0");
			else {
				cinr_percent = (wd_status.cinr < 0.0)?0:((int)(((long)(wd_status.cinr)) << 2)); // percentage : (cinr)*(100/25)
				if (cinr_percent <= 0)
					cinr_percent = 1;
				if (cinr_percent >= 100)
					cinr_percent = 99;
				fprintf(f, "%2.2f %d", wd_status.cinr, cinr_percent);
			}
		} else {
			fprintf(f, "%d --- --- ", wd_status.link_status);
			fprintf(f, "-150 --- 0");
		}

		if (strlen(wd_status.firmware) > 0){
			fprintf(f, " %s", wd_status.firmware);
		}else{
			fprintf(f, " ---");
		}
		fprintf(f, " %d", wimax_err);
		fclose(f);
	}
}

int main(int argc, char **argv)
{
	struct sigaction sigact;
	int r = 1, act_count;

	sigact.sa_handler = sighandler_exit;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;
	sigaction(SIGINT, &sigact, NULL);
	sigaction(SIGTERM, &sigact, NULL);
	sigaction(SIGQUIT, &sigact, NULL);
	sigact.sa_handler = sighandler_wait_child;
	sigaction(SIGCHLD, &sigact, NULL);

	sigact.sa_handler = sighandler_set_info;
	sigaction(SIGUSR1, &sigact, NULL);

	if (daemonize) {
		daemon(0, 0);
	}

	WIMAX_PRINT(LOG_INFO, "Wimax : Driver initialization!!!!!!!!!!!!");
	memset(&wd_status, 0, sizeof(struct wimax_dev_status));

	r = libusb_init(&ctx);
	if (r < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : failed to initialise libusb");
		exit_release_resources(1);
	}

	devh = find_wimax_device();
	if (devh == NULL) {
		WIMAX_PRINT(LOG_ERR, "Wimax : Could not find/open device");
		exit_release_resources(1);
	}

	WIMAX_PRINT(LOG_NOTICE, "Wimax : Device found [0x%04X:0x%04X]", wd_status.dev_vid, wd_status.dev_pid);

	if (detach_dvd && libusb_kernel_driver_active(devh, IF_DVD) == 1) {
		WIMAX_PRINT(LOG_NOTICE, "Wimax : libusb kernel driver active! Try to detach kernel driver!");
		r = libusb_detach_kernel_driver(devh, IF_DVD);
		if (r < 0) {
			WIMAX_PRINT(LOG_ERR, "Wimax : kernel driver detach error %d", r);
		} else {
			WIMAX_PRINT(LOG_INFO, "Wimax : detached pseudo-DVD kernel driver");
		}
	}

	if (libusb_kernel_driver_active(devh, IF_MODEM) == 1) {
		kernel_driver_active = 1;
		r = libusb_detach_kernel_driver(devh, IF_MODEM);
		if (r < 0) {
			WIMAX_PRINT(LOG_ERR, "Wimax : kernel driver detach error %d", r);
		} else {
			WIMAX_PRINT(LOG_ERR, "Wimax : detached modem kernel driver");
		}
	}

	if(detach_dvd)
		r = libusb_claim_interface(devh, IF_DVD);
	else
		r = libusb_claim_interface(devh, IF_MODEM);
	if (r < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : Claim usb interface error %d", r); // LIBUSB_ERROR_NOT_FOUND 
		exit_release_resources(1);
	}

	WIMAX_PRINT(LOG_INFO, "Wimax : Claimed interface %d", detach_dvd?IF_DVD:IF_MODEM);
#if WIMAX_DEBUG == 1
	printf("Wimax : Claimed interface\n");
#endif
	alloc_fds();
	libusb_set_pollfd_notifiers(ctx, cb_add_pollfd, cb_remove_pollfd, NULL);

	EP_IN = (2 | LIBUSB_ENDPOINT_IN); /* default endpoint number for Samsung */
	EP_OUT = (4 | LIBUSB_ENDPOINT_OUT);

	r = init();
	if (r < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : init error %d", r);
		exit_release_resources(1);
	}
	if_create();
	cb_add_pollfd(tap_fd, POLLIN, NULL);

	WIMAX_PRINT(1, "Wimax : before scan_loop()");
	r = scan_loop();
	if (r < 0) {
		WIMAX_PRINT(LOG_ERR, "Wimax : scan_loop error %d", r);
		exit_release_resources(1);
	}

	exit_release_resources(0);
	return 0;
}

