#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <limits.h>
#include <netinet/in.h>
#include <iptables.h>

#define QOS_PRE_CHAIN "qos_prerouting_rule_chain"
#define QOS_POST_CHAIN "qos_postrouting_rule_chain"

#define	QoS_DISABLED		0
#define	QoS_BIDIRECTION		1
#define	QoS_UPLOAD			2
#define	QoS_DOWNLOAD		3
#define	QoS_VOIP_ONLY		5

#define DRAY_VQ_ACT_ADD 0
#define DRAY_VQ_ACT_UPT 1
#define DRAY_VQ_ACT_DEL 2

// copy from kernel
struct my_list_head {
	struct my_list_head *next, *prev;
};

struct dray_vq_event_st 
{
	struct my_list_head list;
	int act;
	unsigned long ip1;
	int port1;
	unsigned long ip2;
	int port2;
};

#define DRAY_VQ_QOS_CALL_MAX 32

struct dray_vq_sip {
	unsigned long ip1;
	int port1;
	long time;
	unsigned long long traffic;
};

static struct dray_vq_sip calls[DRAY_VQ_QOS_CALL_MAX];
static int call_num = 0;

#define DRAY_VQ_CHECK_TIME 30
#define DRAY_VQ_IDLE_TIME 120

static int fd = -1;
static int end = 0;
static int update = 0;
static char cmd_buf[128];
static int qos_enable = QoS_DISABLED;
static int call_max = 0;
static long check_time = 0;

#define CHUNK 256    // i think Chunk size=256 is quite enough   //bruce hsu
static char *slurp(FILE *p)
{
    size_t off = 0, tot = CHUNK, n;
    char *buf = malloc(CHUNK+1); /* Zero terminate */
    if (!buf)
        return NULL;
    while((n = fread(buf+off, 1, CHUNK, p)) > 0) {
        off += n, tot += n;
        buf = realloc(buf, tot);
    }
    buf[off] = '\0';  /* Zero terminate - we have room */
    return buf;
}

char *run_command(const char *cmd)
{
    FILE *p = popen(cmd, "r");
    char *buf = NULL;
    if(p) {
        buf = slurp(p);
        pclose(p);
    } else {
        perror(cmd);
    }
	if (buf && isspace(buf[strlen(buf)-1]))	// remove '0x0a' at end of buf
		buf[strlen(buf)-1] = '\0';
    return buf;
}

static long get_sys_uptime()
{
	long ret = 0;
	char *str = run_command("cat /proc/uptime | awk '{FS=\".\"}{print $1}'");
	if (str) {
		ret = atol(str);
		free(str);
	}
	return ret;
}

static long time_has_passed(long last_time, long period)
{
	long now = get_sys_uptime();
	if (now >= last_time)
		return (now - last_time > period)?now:0;
	else	// overflow
		return ((LONG_MAX - last_time) + now > period)?now:0;
}

static void add_rec(unsigned long ip, int port)
{
	char ip_str[16];
	struct in_addr in;

	in.s_addr = ip;
	strncpy(ip_str, (char *)inet_ntoa(in), sizeof(ip_str));
	ip_str[sizeof(ip_str)-1] = 0;

	//printf("\r\n[nancy] add_rec %s, %d\r\n", ip_str, port);

	if (qos_enable==QoS_BIDIRECTION || qos_enable==QoS_DOWNLOAD || qos_enable==QoS_VOIP_ONLY) {
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -I %s 4 -p udp --sport %d -s %s -j MARK --set-mark 41", QOS_PRE_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -I %s 5 -p udp --sport %d -s %s -j RETURN", QOS_PRE_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	}
	if (qos_enable==QoS_BIDIRECTION || qos_enable==QoS_UPLOAD || qos_enable==QoS_VOIP_ONLY) {
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -I %s 4 -p udp --dport %d -d %s -j MARK --set-mark 41", QOS_POST_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -I %s 5 -p udp --dport %d -d %s -j RETURN", QOS_POST_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	}
}

static void delete_rec(unsigned long ip, int port)
{
	char ip_str[16];
	struct in_addr in;

	in.s_addr = ip;
	strncpy(ip_str, (char *)inet_ntoa(in), sizeof(ip_str));
	ip_str[sizeof(ip_str)-1] = 0;

	//printf("\r\n[nancy] delete_rec %s, %d\r\n", ip_str, port);
	
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -D %s -p udp --sport %d -s %s -j MARK --set-mark 41", QOS_PRE_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -D %s -p udp --sport %d -s %s -j RETURN", QOS_PRE_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -D %s -p udp --dport %d -d %s -j MARK --set-mark 41", QOS_POST_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
	snprintf(cmd_buf, sizeof(cmd_buf), "iptables -t mangle -D %s -p udp --dport %d -d %s -j RETURN", QOS_POST_CHAIN, port, ip_str);
	cmd_buf[sizeof(cmd_buf)-1] = 0;
	system(cmd_buf);
}

static void dray_voip_qos_flush_call(void)
{
	int i;
	//printf("\r\n[dray_voip_qos_flush_call]\r\n");
	for (i = 0; i < call_max; i++) {
		if (calls[i].ip1 != 0) {
			delete_rec(calls[i].ip1, calls[i].port1);
			memset(&(calls[i]), 0, sizeof(struct dray_vq_sip));
		}
	}
}

static void dray_voip_qos_try_add_call(unsigned long ip1, int port1)
{
	int i;
	//printf("\r\n[dray_voip_qos_try_add_call]\r\n");
	for (i = 0; i < call_max; i++) {
		if (calls[i].ip1 == 0) {
			calls[i].ip1 = ip1;
			calls[i].port1 = port1;
			calls[i].time = get_sys_uptime();
			calls[i].traffic = 0;
			add_rec(ip1, port1);
			call_num++;
			return;
		}
	}
}

static void dray_voip_qos_try_update_call(unsigned long old_ip, int old_port, unsigned long new_ip, int new_port)
{
	int i;
	//printf("\r\n[dray_voip_qos_try_update_call]\r\n");
	for (i = 0; i < call_max; i++) {
		if (calls[i].ip1 == old_ip && calls[i].port1 == old_port) {
			calls[i].ip1 = new_ip;
			calls[i].port1 = new_port;
			calls[i].time = get_sys_uptime();
			delete_rec(old_ip, old_port);
			add_rec(new_ip, new_port);
			return;
		}
	}
}

static void dray_voip_qos_try_del_call(unsigned long ip1, int port1)
{
	int i;
	//printf("\r\n[dray_voip_qos_try_del_call]\r\n");
	for (i = 0; i < call_max; i++) {
		if (calls[i].ip1 == ip1 && calls[i].port1 == port1) {
			delete_rec(calls[i].ip1, calls[i].port1);
			memset(&(calls[i]), 0, sizeof(struct dray_vq_sip));
			call_num--;
			return;
		}
	}
}

struct xt_mark_target_info_v1 {
	unsigned long mark;
	u_int8_t mode;
};

static iptc_handle_t vq_handle = 0;

static void check_idle_call(void)
{
	int i;
	long now;
	unsigned long long traffic[DRAY_VQ_QOS_CALL_MAX];
	const char *this;
	const char *target_name;

	now = time_has_passed(check_time, DRAY_VQ_CHECK_TIME);
	if (!now)
		return;
	check_time = now;

	//printf("\r\n[check_idle_call]\r\n");

	for (i = 0; i < call_max; i++)
		traffic[i] = 0;

	vq_handle = iptc_init("mangle");
	if (!vq_handle)
		return;

	/* this : PREROUTING, INPUT, FORWARD, OUTPUT...etc.. */
	for (this = iptc_first_chain(&vq_handle); this; this = iptc_next_chain(&vq_handle)) {
		const struct ipt_entry *entry;

		if (strcmp(this, QOS_PRE_CHAIN) && strcmp(this, QOS_POST_CHAIN))
			continue;

		entry = iptc_first_rule(this, &vq_handle);
		while (entry) {
			target_name = iptc_get_target(entry, &vq_handle);
			if(!strcmp(target_name, "MARK")){
				const struct ipt_entry_target *t;
				const struct xt_mark_target_info_v1 *markinfo;

				t = ipt_get_target((struct ipt_entry *)entry);
				markinfo = (const struct xt_mark_target_info_v1 *)t->data;

				if (markinfo->mark == 41) { // VoIP Mark
					if (entry->ip.dst.s_addr && entry->counters.bcnt) {
						for (i = 0; i < call_max; i++) {
							if (calls[i].ip1 == entry->ip.dst.s_addr)
								traffic[i] += entry->counters.bcnt;
						}
					}
				}
			}
			entry = iptc_next_rule(entry, &vq_handle);
		}		
	}

	iptc_free(&vq_handle);

	for (i = 0; i < call_max; i++) {
		if (calls[i].ip1 != 0) {
			if (calls[i].traffic != traffic[i]) {
				calls[i].traffic = traffic[i];
				calls[i].time = now;
			} else if (now - calls[i].time > DRAY_VQ_IDLE_TIME) {
				//printf("\r\n[dray_voip_qos_timeout]\r\n");
				delete_rec(calls[i].ip1, calls[i].port1);
				memset(&(calls[i]), 0, sizeof(struct dray_vq_sip));
			}
		}
	}
}

static void get_qos_config(void)
{
	int i;
	char *str = run_command("nvram_get 2860 QoSEnable");
	if (str) {
		i = atoi(str);
		if (i >= QoS_DISABLED && i <= QoS_VOIP_ONLY)
			qos_enable = i;
		free(str);
	}
	str = run_command("nvram_get 2860 QoSVCalls");
	if (str) {
		i = atoi(str);
		if (i >= 0 && i <= DRAY_VQ_QOS_CALL_MAX) {
			dray_voip_qos_flush_call();
			call_max = i;
		}
		free(str);
	}
}

static void parse_event(struct dray_vq_event_st *event)
{
	int i;
	switch (event->act) {
	case DRAY_VQ_ACT_ADD:
		dray_voip_qos_try_add_call(event->ip1, event->port1);
		break;
	case DRAY_VQ_ACT_UPT:
		dray_voip_qos_try_update_call(event->ip1, event->port1, event->ip2, event->port2);
		break;
	case DRAY_VQ_ACT_DEL:
		dray_voip_qos_try_del_call(event->ip1, event->port1);
	}
}

static void action_handler(int signum)
{
	switch (signum) {
	case SIGTERM:
	case SIGINT:
		end = 1;
		break;
	case SIGUSR1:
		update = 1;
	}
}

int main(int argc, char** argv)
{
	struct sigaction sa;
	fd_set read_set;
    struct timeval timeout;
    int ret;
	struct dray_vq_event_st event;

	memset(&calls, 0, sizeof(struct dray_vq_sip)*DRAY_VQ_QOS_CALL_MAX);

	memset(&sa, 0, sizeof(sa));
	sigaddset(&sa.sa_mask, SIGTERM);
	sigaddset(&sa.sa_mask, SIGINT);
	sigaddset(&sa.sa_mask, SIGUSR1);
	sa.sa_handler = action_handler;
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGUSR1, &sa, NULL);

	signal(SIGHUP, SIG_IGN);

	for (ret = 0; ret < 20 && fd <= 0; ret++) {
		fd = open("/dev/voipqos", O_RDWR);
		sleep(1);
	}
	if (fd <= 0) {
		printf("voip_qos_run: cannot open /dev/voipqos\r\n");
		return 0;
	}

	get_qos_config();

	while (!end) {
		FD_ZERO(&read_set);
		FD_SET(fd, &read_set);
		timeout.tv_sec = 30;
		timeout.tv_usec = 0;
		ret = select(fd+1, &read_set, 0, 0, &timeout);
		if (ret >= 0) {
			while (read(fd, &event, sizeof(struct dray_vq_event_st)))
				parse_event(&event);
		} else {
			if (errno != EINTR) {
				printf("voip_qos_run: select error\r\n");
				break;
			}
		}
		if (update) {
			get_qos_config();
			update = 0;
		}
		check_idle_call();
	}
	close(fd);

	return 0;
}

